"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PackagePolicyServiceImpl = void 0;
exports._applyIndexPrivileges = _applyIndexPrivileges;
exports._compilePackagePolicyInputs = _compilePackagePolicyInputs;
exports._normalizePackagePolicyKuery = _normalizePackagePolicyKuery;
exports._validateRestrictedFieldsNotModifiedOrThrow = _validateRestrictedFieldsNotModifiedOrThrow;
exports.getPackagePolicySavedObjectType = getPackagePolicySavedObjectType;
exports.packagePolicyService = void 0;
exports.preconfigurePackageInputs = preconfigurePackageInputs;
exports.sendUpdatePackagePolicyTelemetryEvent = sendUpdatePackagePolicyTelemetryEvent;
exports.updatePackageInputs = updatePackageInputs;
var _lodash = require("lodash");
var _fp = require("lodash/fp");
var _i18n = require("@kbn/i18n");
var _std = require("@kbn/std");
var _server = require("@kbn/core/server");
var _uuid = require("uuid");
var _jsYaml = require("js-yaml");
var _gt = _interopRequireDefault(require("semver/functions/gt"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _pMap = _interopRequireDefault(require("p-map"));
var _utils = require("../errors/utils");
var _http_authorization_header = require("../../common/http_authorization_header");
var _services = require("../../common/services");
var _constants2 = require("../../common/constants");
var _errors = require("../errors");
var _types = require("../types");
var _constants3 = require("../constants");
var _agentless_policy_helper = require("../../common/services/agentless_policy_helper");
var _cloud_connector = require("../../common/constants/cloud_connector");
var _create_so_find_iterable = require("./utils/create_so_find_iterable");
var _security = require("./security");
var _agent_policy = require("./agent_policy");
var _packages = require("./epm/packages");
var _assets = require("./epm/packages/assets");
var _agent = require("./epm/agent/agent");
var _saved_object = require("./saved_object");
var _ = require(".");
var _cleanup = require("./epm/packages/cleanup");
var _upgrade_sender = require("./upgrade_sender");
var _package_policies = require("./package_policies");
var _input_type_packages = require("./epm/packages/input_type_packages");
var _audit_logging = require("./audit_logging");
var _secrets = require("./secrets");
var _get = require("./epm/packages/get");
var _outputs_helpers = require("./agent_policies/outputs_helpers");
var _policy_namespaces = require("./spaces/policy_namespaces");
var _helpers = require("./spaces/helpers");
var _package_policy = require("./spaces/package_policy");
var _upgrade = require("./package_policies/upgrade");
var _get_input_with_ids = require("./package_policies/get_input_with_ids");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */ /* eslint-disable max-classes-per-file */
async function getPkgInfoAssetsMap({
  savedObjectsClient,
  packageInfos,
  logger
}) {
  const packageInfosandAssetsMap = new Map();
  await (0, _pMap.default)(packageInfos, async pkgInfo => {
    const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
      logger,
      packageInfo: pkgInfo,
      savedObjectsClient
    });
    packageInfosandAssetsMap.set(`${pkgInfo.name}-${pkgInfo.version}`, {
      assetsMap,
      pkgInfo
    });
  }, {
    concurrency: _constants3.MAX_CONCURRENT_PACKAGE_ASSETS
  });
  return packageInfosandAssetsMap;
}
async function getPackagePolicySavedObjectType() {
  return (await (0, _helpers.isSpaceAwarenessEnabled)()) ? _constants2.PACKAGE_POLICY_SAVED_OBJECT_TYPE : _constants2.LEGACY_PACKAGE_POLICY_SAVED_OBJECT_TYPE;
}
function _normalizePackagePolicyKuery(savedObjectType, kuery) {
  if (savedObjectType === _constants2.LEGACY_PACKAGE_POLICY_SAVED_OBJECT_TYPE) {
    return (0, _saved_object.normalizeKuery)(savedObjectType, kuery.replace(new RegExp(`${_constants2.PACKAGE_POLICY_SAVED_OBJECT_TYPE}\\.`, 'g'), `${savedObjectType}.`));
  } else {
    return (0, _saved_object.normalizeKuery)(savedObjectType, kuery.replace(new RegExp(`${_constants2.LEGACY_PACKAGE_POLICY_SAVED_OBJECT_TYPE}\\.`, 'g'), `${savedObjectType}.`));
  }
}

/**
 * Validates package policy variables and extracts cloud connector configuration.
 *
 * Currently supports AWS cloud connector only. This function checks for both:
 * - `aws.role_arn` and `aws.credentials.external_id` (CSPM and Asset Discovery)
 * - `role_arn` and `external_id` (direct cloud connector variables)
 *
 * This is a temporary implementation pending the generic solution proposed by the
 * Package Spec team. Once approved, this function should be updated to use the
 * standardized approach.
 *
 * @see https://github.com/elastic/security-team/issues/13277#issuecomment-3245273903 for updates
 * @todo Remove hardcoded checks for `aws.role_arn` and `aws.credentials.external_id`
 *       and implement the generic Package Spec solution once approved.
 */

const extractPackagePolicyVars = (cloudProvider, packagePolicy, logger) => {
  logger.get('extract package policy vars');
  if (packagePolicy.supports_cloud_connector && cloudProvider === 'aws') {
    var _packagePolicy$inputs, _packagePolicy$inputs2, _vars$role_arn, _vars$AWS_ROLE_ARN_VA;
    const vars = (_packagePolicy$inputs = packagePolicy.inputs.find(input => input.enabled)) === null || _packagePolicy$inputs === void 0 ? void 0 : (_packagePolicy$inputs2 = _packagePolicy$inputs.streams[0]) === null || _packagePolicy$inputs2 === void 0 ? void 0 : _packagePolicy$inputs2.vars;
    if (!vars) {
      logger.error('Package policy must contain vars');
      throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain vars');
    }
    const roleArn = ((_vars$role_arn = vars.role_arn) === null || _vars$role_arn === void 0 ? void 0 : _vars$role_arn.value) || ((_vars$AWS_ROLE_ARN_VA = vars[_cloud_connector.AWS_ROLE_ARN_VAR_NAME]) === null || _vars$AWS_ROLE_ARN_VA === void 0 ? void 0 : _vars$AWS_ROLE_ARN_VA.value);
    if (roleArn) {
      var _vars$external_id, _vars$external_id$val;
      const externalId = (_vars$external_id = vars.external_id) !== null && _vars$external_id !== void 0 && (_vars$external_id$val = _vars$external_id.value) !== null && _vars$external_id$val !== void 0 && _vars$external_id$val.isSecretRef ? vars.external_id : vars[_cloud_connector.AWS_CREDENTIALS_EXTERNAL_ID_VAR_NAME];
      const awsCloudConnectorVars = {
        role_arn: {
          type: 'text',
          value: roleArn
        },
        external_id: externalId
      };
      return awsCloudConnectorVars;
    }
  }
};
class PackagePolicyClientImpl {
  getLogger(...childContextPaths) {
    return _.appContextService.getLogger().get('PackagePolicyClient', ...childContextPaths);
  }
  async create(soClient, esClient, packagePolicy, options = {}, context, request) {
    var _options$packageInfo, _enrichedPackagePolic3, _options$packageInfo2, _pkgInfo$elasticsearc, _secretReferences, _options$user$usernam, _options$user, _options$user$usernam2, _options$user2, _options$bumpRevision;
    const logger = this.getLogger('create');
    logger.debug(() => {
      var _packagePolicy$packag;
      return `Creating [${(_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name}] package policy with soClient scoped to [${soClient.getCurrentNamespace()}] and options.spaceId [${options.spaceId}]`;
    });
    const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
    const packagePolicyId = (options === null || options === void 0 ? void 0 : options.id) || (0, _uuid.v4)();
    let authorizationHeader = options.authorizationHeader;
    if (!authorizationHeader && request) {
      authorizationHeader = _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request);
    }
    const savedObjectType = await getPackagePolicySavedObjectType();
    const basePkgInfo = (_options$packageInfo = options === null || options === void 0 ? void 0 : options.packageInfo) !== null && _options$packageInfo !== void 0 ? _options$packageInfo : packagePolicy.package ? await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName: packagePolicy.package.name,
      pkgVersion: packagePolicy.package.version,
      ignoreUnverified: true,
      prerelease: true
    }) : undefined;
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id: packagePolicyId,
      name: packagePolicy.name,
      savedObjectType
    });
    let secretReferences;
    this.keepPolicyIdInSync(packagePolicy);
    await (0, _package_policies.preflightCheckPackagePolicy)(soClient, packagePolicy, basePkgInfo);
    let enrichedPackagePolicy = await packagePolicyService.runExternalCallbacks('packagePolicyCreate', packagePolicy, soClient, esClient, context, request);
    const agentPolicies = [];
    for (const policyId of enrichedPackagePolicy.policy_ids) {
      var _enrichedPackagePolic;
      const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, policyId, true);
      if (!agentPolicy) {
        throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
      }
      agentPolicies.push(agentPolicy);

      // If package policy did not set an output_id, see if the agent policy's output is compatible
      if (!packagePolicy.output_id && agentPolicy && (_enrichedPackagePolic = enrichedPackagePolicy.package) !== null && _enrichedPackagePolic !== void 0 && _enrichedPackagePolic.name) {
        var _enrichedPackagePolic2;
        await (0, _outputs_helpers.validateAgentPolicyOutputForIntegration)(soClient, agentPolicy, packagePolicy, (_enrichedPackagePolic2 = enrichedPackagePolicy.package) === null || _enrichedPackagePolic2 === void 0 ? void 0 : _enrichedPackagePolic2.name);
      }
      validateIsNotHostedPolicy(agentPolicy, options === null || options === void 0 ? void 0 : options.force);
      if (useSpaceAwareness) {
        validateReusableIntegrationsAndSpaceAwareness(enrichedPackagePolicy, agentPolicies);
      }
      (0, _agentless_policy_helper.validateDeploymentModesForInputs)(packagePolicy.inputs, agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless || packagePolicy.supports_agentless ? 'agentless' : 'default', basePkgInfo);
    }
    const spaceIds = agentPolicies.flatMap(ap => {
      var _ap$space_ids;
      return (_ap$space_ids = ap.space_ids) !== null && _ap$space_ids !== void 0 ? _ap$space_ids : [];
    });
    // trailing whitespace causes issues creating API keys
    enrichedPackagePolicy.name = enrichedPackagePolicy.name.trim();
    if (!(options !== null && options !== void 0 && options.skipUniqueNameVerification)) {
      await requireUniqueName(soClient, enrichedPackagePolicy, undefined, spaceIds);
    }
    if (enrichedPackagePolicy.namespace) {
      await (0, _policy_namespaces.validatePolicyNamespaceForSpace)({
        namespace: enrichedPackagePolicy.namespace,
        spaceId: soClient.getCurrentNamespace()
      });
    }
    await (0, _policy_namespaces.validateAdditionalDatastreamsPermissionsForSpace)({
      additionalDatastreamsPermissions: enrichedPackagePolicy.additional_datastreams_permissions,
      spaceId: soClient.getCurrentNamespace()
    });

    // Make sure the associated package is installed
    if (!((_enrichedPackagePolic3 = enrichedPackagePolicy.package) !== null && _enrichedPackagePolic3 !== void 0 && _enrichedPackagePolic3.name)) {
      throw new _errors.FleetError('Package policy without package are not supported');
    }
    if (!(options !== null && options !== void 0 && options.skipEnsureInstalled)) {
      await (0, _packages.ensureInstalledPackage)({
        esClient,
        spaceId: (options === null || options === void 0 ? void 0 : options.spaceId) || _constants.DEFAULT_SPACE_ID,
        savedObjectsClient: soClient,
        pkgName: enrichedPackagePolicy.package.name,
        pkgVersion: enrichedPackagePolicy.package.version,
        force: options === null || options === void 0 ? void 0 : options.force,
        authorizationHeader
      });
    }

    // Handle component template/mappings updates for experimental features, e.g. synthetic source
    await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
      soClient,
      esClient,
      packagePolicy: enrichedPackagePolicy
    });
    const pkgInfo = (_options$packageInfo2 = options === null || options === void 0 ? void 0 : options.packageInfo) !== null && _options$packageInfo2 !== void 0 ? _options$packageInfo2 : await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName: enrichedPackagePolicy.package.name,
      pkgVersion: enrichedPackagePolicy.package.version,
      prerelease: true
    });
    let inputs = (0, _get_input_with_ids.getInputsWithIds)(enrichedPackagePolicy, packagePolicyId, undefined, pkgInfo);

    // Check if it is a limited package, and if so, check that the corresponding agent policy does not
    // already contain a package policy for this package
    if ((0, _services.isPackageLimited)(pkgInfo)) {
      for (const agentPolicy of agentPolicies) {
        if (agentPolicy && (0, _services.doesAgentPolicyAlreadyIncludePackage)(agentPolicy, pkgInfo.name)) {
          throw new _errors.FleetError(`Unable to create integration policy. Integration '${pkgInfo.name}' already exists on this agent policy.`);
        }
      }
    }
    validatePackagePolicyOrThrow(enrichedPackagePolicy, pkgInfo);
    (0, _package_policies.canDeployCustomPackageAsAgentlessOrThrow)(packagePolicy, pkgInfo);
    if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
      var _enrichedPackagePolic4, _enrichedPackagePolic5;
      const secretsRes = await (0, _secrets.extractAndWriteSecrets)({
        packagePolicy: {
          ...enrichedPackagePolicy,
          inputs
        },
        packageInfo: pkgInfo,
        esClient
      });
      enrichedPackagePolicy = secretsRes.packagePolicy;
      secretReferences = secretsRes.secretReferences;

      // Create cloud connector for package policy if it is supported and package supports agentless
      if ((_enrichedPackagePolic4 = enrichedPackagePolicy) !== null && _enrichedPackagePolic4 !== void 0 && _enrichedPackagePolic4.supports_agentless && (_enrichedPackagePolic5 = enrichedPackagePolicy) !== null && _enrichedPackagePolic5 !== void 0 && _enrichedPackagePolic5.supports_cloud_connector) {
        const cloudConnector = await this.createCloudConnectorForPackagePolicy(soClient, enrichedPackagePolicy, agentPolicies[0]);
        if (cloudConnector) {
          enrichedPackagePolicy.cloud_connector_id = cloudConnector.id;
        }
      }
      inputs = enrichedPackagePolicy.inputs;
    }
    const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
      logger,
      packageInfo: pkgInfo,
      savedObjectsClient: soClient
    });
    inputs = _compilePackagePolicyInputs(pkgInfo, enrichedPackagePolicy.vars || {}, inputs, assetsMap);
    const elasticsearchPrivileges = (_pkgInfo$elasticsearc = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc === void 0 ? void 0 : _pkgInfo$elasticsearc.privileges;
    if (pkgInfo.type === 'input') {
      await (0, _input_type_packages.installAssetsForInputPackagePolicy)({
        soClient,
        esClient,
        pkgInfo,
        packagePolicy: enrichedPackagePolicy,
        force: !!(options !== null && options !== void 0 && options.force),
        logger
      });
    }
    const requiresRoot = (0, _services.isRootPrivilegesRequired)(pkgInfo);
    if (enrichedPackagePolicy.package && requiresRoot) {
      enrichedPackagePolicy.package = {
        ...enrichedPackagePolicy.package,
        requires_root: requiresRoot,
        fips_compatible: (0, _services.checkIntegrationFipsLooseCompatibility)(pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.name, pkgInfo)
      };
    }
    const isoDate = new Date().toISOString();
    const newSo = await soClient.create(savedObjectType, {
      ...enrichedPackagePolicy,
      ...(enrichedPackagePolicy.package ? {
        package: (0, _lodash.omit)(enrichedPackagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      ...(elasticsearchPrivileges && {
        elasticsearch: {
          privileges: elasticsearchPrivileges
        }
      }),
      ...(((_secretReferences = secretReferences) === null || _secretReferences === void 0 ? void 0 : _secretReferences.length) && {
        secret_references: secretReferences
      }),
      latest_revision: true,
      revision: 1,
      created_at: isoDate,
      created_by: (_options$user$usernam = options === null || options === void 0 ? void 0 : (_options$user = options.user) === null || _options$user === void 0 ? void 0 : _options$user.username) !== null && _options$user$usernam !== void 0 ? _options$user$usernam : 'system',
      updated_at: isoDate,
      updated_by: (_options$user$usernam2 = options === null || options === void 0 ? void 0 : (_options$user2 = options.user) === null || _options$user2 === void 0 ? void 0 : _options$user2.username) !== null && _options$user$usernam2 !== void 0 ? _options$user$usernam2 : 'system'
    }, {
      ...options,
      id: packagePolicyId
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`attempt to create package policy saved object failed`));
    for (const agentPolicy of agentPolicies) {
      if (useSpaceAwareness && agentPolicy && agentPolicy.space_ids && agentPolicy.space_ids.length > 1) {
        var _soClient$getCurrentN;
        await (0, _package_policy.updatePackagePolicySpaces)({
          packagePolicyId: newSo.id,
          currentSpaceId: (_soClient$getCurrentN = soClient.getCurrentNamespace()) !== null && _soClient$getCurrentN !== void 0 ? _soClient$getCurrentN : _constants.DEFAULT_SPACE_ID,
          newSpaceIds: agentPolicy.space_ids
        });
      }
    }
    if ((_options$bumpRevision = options === null || options === void 0 ? void 0 : options.bumpRevision) !== null && _options$bumpRevision !== void 0 ? _options$bumpRevision : true) {
      for (const policyId of enrichedPackagePolicy.policy_ids) {
        await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, policyId, {
          user: options === null || options === void 0 ? void 0 : options.user
        });
      }
    }
    const createdPackagePolicy = (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(newSo);
    logger.debug(`Created new package policy with id ${newSo.id} and version ${newSo.version}`);
    return packagePolicyService.runExternalCallbacks('packagePolicyPostCreate', createdPackagePolicy, soClient, esClient);
  }
  keepPolicyIdInSync(packagePolicy) {
    var _packagePolicy$policy;
    if (packagePolicy.policy_ids) {
      if (packagePolicy.policy_ids.length === 0 && packagePolicy.policy_id !== undefined) {
        packagePolicy.policy_id = null;
      }
    } else {
      packagePolicy.policy_ids = [];
    }
    if (packagePolicy.policy_id && !((_packagePolicy$policy = packagePolicy.policy_ids) !== null && _packagePolicy$policy !== void 0 && _packagePolicy$policy[0])) {
      packagePolicy.policy_ids = [packagePolicy.policy_id];
    } else if (!packagePolicy.policy_id && packagePolicy.policy_ids[0]) {
      packagePolicy.policy_id = packagePolicy.policy_ids[0];
    }
  }
  async bulkCreate(soClient, esClient, packagePolicies, options) {
    var _options$bumpRevision2;
    const [useSpaceAwareness, savedObjectType, packageInfos] = await Promise.all([(0, _helpers.isSpaceAwarenessEnabled)(), getPackagePolicySavedObjectType(), getPackageInfoForPackagePolicies(packagePolicies, soClient)]);
    await (0, _pMap.default)(packagePolicies, async packagePolicy => {
      const basePkgInfo = packagePolicy.package ? packageInfos.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`) : undefined;
      if (!packagePolicy.id) {
        packagePolicy.id = _server.SavedObjectsUtils.generateId();
      }
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'create',
        id: packagePolicy.id,
        name: packagePolicy.name,
        savedObjectType
      });
      this.keepPolicyIdInSync(packagePolicy);
      await (0, _package_policies.preflightCheckPackagePolicy)(soClient, packagePolicy, basePkgInfo);
    });
    const agentPolicyIds = new Set(packagePolicies.flatMap(pkgPolicy => pkgPolicy.policy_ids));
    const agentPolicies = await _agent_policy.agentPolicyService.getByIds(soClient, [...agentPolicyIds]);
    const agentPoliciesIndexById = (0, _fp.indexBy)('id', agentPolicies);
    for (const agentPolicy of agentPolicies) {
      validateIsNotHostedPolicy(agentPolicy, options === null || options === void 0 ? void 0 : options.force);
    }
    if (useSpaceAwareness) {
      for (const packagePolicy of packagePolicies) {
        validateReusableIntegrationsAndSpaceAwareness(packagePolicy, packagePolicy.policy_ids.map(policyId => agentPoliciesIndexById[policyId]).filter(policy => policy !== undefined));
      }
    }
    const isoDate = new Date().toISOString();
    const policiesToCreate = [];
    const failedPolicies = [];
    const logger = this.getLogger('bulkCreate');
    logger.debug(`Starting bulk create of package policy`);
    const packagePoliciesWithIds = packagePolicies.map(p => {
      if (!p.id) {
        p.id = _server.SavedObjectsUtils.generateId();
      }
      return p;
    });
    const packageInfosandAssetsMap = await getPkgInfoAssetsMap({
      logger,
      packageInfos: [...packageInfos.values()],
      savedObjectsClient: soClient
    });
    await (0, _pMap.default)(packagePoliciesWithIds, async packagePolicy => {
      try {
        var _packagePolicy$id, _options$user$usernam3, _options$user3, _options$user$usernam4, _options$user4;
        const packagePolicyId = (_packagePolicy$id = packagePolicy.id) !== null && _packagePolicy$id !== void 0 ? _packagePolicy$id : (0, _uuid.v4)();
        const agentPolicyIdsOfPackagePolicy = packagePolicy.policy_ids;
        if (!packagePolicy.package) {
          throw new _errors.FleetError('Package policy without package are not supported');
        }
        const {
          id,
          ...pkgPolicyWithoutId
        } = packagePolicy;
        const packageInfoAndAsset = packageInfosandAssetsMap.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
        if (!packageInfoAndAsset) {
          throw new _errors.FleetError(`Package info and assets not found: ${packagePolicy.package.name}-${packagePolicy.package.version}`);
        }
        const {
          pkgInfo,
          assetsMap
        } = packageInfoAndAsset;
        let inputs = (0, _get_input_with_ids.getInputsWithIds)(packagePolicy, packagePolicyId, undefined, pkgInfo);
        validatePackagePolicyOrThrow(packagePolicy, pkgInfo);
        (0, _package_policies.canDeployCustomPackageAsAgentlessOrThrow)(packagePolicy, pkgInfo);
        inputs = pkgInfo ? await _compilePackagePolicyInputs(pkgInfo, packagePolicy.vars || {}, inputs, assetsMap) : inputs;
        const elasticsearch = pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.elasticsearch;
        const requiresRoot = (0, _services.isRootPrivilegesRequired)(pkgInfo);
        if (packagePolicy.package && requiresRoot) {
          packagePolicy.package = {
            ...packagePolicy.package,
            requires_root: requiresRoot,
            fips_compatible: (0, _services.checkIntegrationFipsLooseCompatibility)(pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.name, pkgInfo)
          };
        }
        policiesToCreate.push({
          type: savedObjectType,
          id: packagePolicyId,
          attributes: {
            ...pkgPolicyWithoutId,
            ...(packagePolicy.package ? {
              package: (0, _lodash.omit)(packagePolicy.package, 'experimental_data_stream_features')
            } : {}),
            inputs,
            elasticsearch,
            policy_id: agentPolicyIdsOfPackagePolicy[0],
            policy_ids: agentPolicyIdsOfPackagePolicy,
            latest_revision: true,
            revision: 1,
            created_at: isoDate,
            created_by: (_options$user$usernam3 = options === null || options === void 0 ? void 0 : (_options$user3 = options.user) === null || _options$user3 === void 0 ? void 0 : _options$user3.username) !== null && _options$user$usernam3 !== void 0 ? _options$user$usernam3 : 'system',
            updated_at: isoDate,
            updated_by: (_options$user$usernam4 = options === null || options === void 0 ? void 0 : (_options$user4 = options.user) === null || _options$user4 === void 0 ? void 0 : _options$user4.username) !== null && _options$user$usernam4 !== void 0 ? _options$user$usernam4 : 'system'
          }
        });
      } catch (error) {
        failedPolicies.push({
          packagePolicy,
          error
        });
        logger.error(error);
      }
    });
    const {
      saved_objects: createdObjects
    } = await soClient.bulkCreate(policiesToCreate).catch(_utils.catchAndSetErrorStackTrace.withMessage('failed to bulk create package policies'));

    // Filter out invalid SOs
    const newSos = createdObjects.filter(so => !so.error && so.attributes);
    packagePoliciesWithIds.forEach(packagePolicy => {
      const hasCreatedSO = newSos.find(so => so.id === packagePolicy.id);
      const hasFailed = failedPolicies.some(({
        packagePolicy: failedPackagePolicy
      }) => failedPackagePolicy.id === packagePolicy.id);
      if (hasCreatedSO !== null && hasCreatedSO !== void 0 && hasCreatedSO.error && !hasFailed) {
        var _hasCreatedSO$error;
        failedPolicies.push({
          packagePolicy,
          error: (_hasCreatedSO$error = hasCreatedSO === null || hasCreatedSO === void 0 ? void 0 : hasCreatedSO.error) !== null && _hasCreatedSO$error !== void 0 ? _hasCreatedSO$error : new _errors.FleetError('Failed to create package policy.')
        });
      }
    });
    if (useSpaceAwareness) {
      for (const newSo of newSos) {
        // Do not support multpile spaces for reusable integrations
        if (newSo.attributes.policy_ids.length > 1) {
          continue;
        }
        const agentPolicy = agentPoliciesIndexById[newSo.attributes.policy_ids[0]];
        if (agentPolicy && agentPolicy.space_ids && agentPolicy.space_ids.length > 1) {
          var _soClient$getCurrentN2;
          await (0, _package_policy.updatePackagePolicySpaces)({
            packagePolicyId: newSo.id,
            currentSpaceId: (_soClient$getCurrentN2 = soClient.getCurrentNamespace()) !== null && _soClient$getCurrentN2 !== void 0 ? _soClient$getCurrentN2 : _constants.DEFAULT_SPACE_ID,
            newSpaceIds: agentPolicy.space_ids
          });
        }
      }
    }

    // Assign it to the given agent policy

    if ((_options$bumpRevision2 = options === null || options === void 0 ? void 0 : options.bumpRevision) !== null && _options$bumpRevision2 !== void 0 ? _options$bumpRevision2 : true) {
      for (const agentPolicyId of agentPolicyIds) {
        await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, agentPolicyId, {
          user: options === null || options === void 0 ? void 0 : options.user,
          asyncDeploy: options === null || options === void 0 ? void 0 : options.asyncDeploy
        });
      }
    }
    logger.debug(`Created new package policies`);
    return {
      created: newSos.map(newSo => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(newSo)),
      failed: failedPolicies
    };
  }

  /** Purpose of this function is to take a package policy and compile the inputs
   This is primarily used by the Synthetics UI to display the inputs which are passed to agent
   Purpose is to debug the inputs which are passed to the agent and also compared them to the config
   which is passed to public service locations */
  async inspect(soClient, packagePolicy) {
    if (!packagePolicy.id) {
      packagePolicy.id = _server.SavedObjectsUtils.generateId();
    }
    const packageInfos = await getPackageInfoForPackagePolicies([packagePolicy], soClient);
    if (!packagePolicy.package) {
      throw new _errors.FleetError('Package policy without package are not supported');
    }
    const pkgInfo = packageInfos.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
    if (!pkgInfo) {
      throw new _errors.FleetError(`Package info and assets not found: ${packagePolicy.package.name}-${packagePolicy.package.version}`);
    }
    let inputs = (0, _get_input_with_ids.getInputsWithIds)(packagePolicy, packagePolicy.id, undefined, pkgInfo);
    const {
      id,
      ...pkgPolicyWithoutId
    } = packagePolicy;
    const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
      logger: this.getLogger('inspect'),
      packageInfo: pkgInfo,
      savedObjectsClient: soClient
    });
    inputs = pkgInfo ? await _compilePackagePolicyInputs(pkgInfo, packagePolicy.vars || {}, inputs, assetsMap) : inputs;
    const elasticsearch = pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.elasticsearch;
    return {
      id: packagePolicy.id,
      ...pkgPolicyWithoutId,
      ...(packagePolicy.package ? {
        package: (0, _lodash.omit)(packagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      elasticsearch
    };
  }
  async get(soClient, id, options = {}) {
    var _packagePolicy$packag2;
    const logger = this.getLogger('get');
    logger.debug(() => {
      var _options$spaceId;
      return `Getting package policy [${id}] for space [${(_options$spaceId = options.spaceId) !== null && _options$spaceId !== void 0 ? _options$spaceId : soClient.getCurrentNamespace()}]`;
    });

    // We're using `getByIds()` here, instead of just `soClient.get()`, because when using an unscoped
    // SO client we are not able to use `*` in the `esClient.get()` `options.namespace`.
    const [packagePolicy] = await this.getByIDs(soClient, [id], {
      spaceIds: options.spaceId ? [options.spaceId] : undefined
    }).catch(async err => {
      // Emulate prior implementation that threw a Saved Objects error so that backwards compatibility is maintained
      if (err instanceof _errors.FleetNotFoundError) {
        throw _server.SavedObjectsErrorHelpers.createGenericNotFoundError(await (0, _agent_policy.getAgentPolicySavedObjectType)(), id);
      }
      throw _server.SavedObjectsErrorHelpers.createBadRequestError(err.message);
    });
    if (!packagePolicy) {
      logger.debug(`Package Policy [${id}] was not found`);
      return null;
    }
    if ((_packagePolicy$packag2 = packagePolicy.package) !== null && _packagePolicy$packag2 !== void 0 && _packagePolicy$packag2.name) {
      var _packagePolicy$packag3, _installation$attribu;
      const installation = await soClient.get(_constants2.PACKAGES_SAVED_OBJECT_TYPE, (_packagePolicy$packag3 = packagePolicy.package) === null || _packagePolicy$packag3 === void 0 ? void 0 : _packagePolicy$packag3.name).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to get package [${packagePolicy.package.name}]`));

      // If possible, return the experimental features map for the package policy's `package` field
      if (installation && !installation.error && (_installation$attribu = installation.attributes) !== null && _installation$attribu !== void 0 && _installation$attribu.experimental_data_stream_features) {
        var _installation$attribu2;
        packagePolicy.package.experimental_data_stream_features = (_installation$attribu2 = installation.attributes) === null || _installation$attribu2 === void 0 ? void 0 : _installation$attribu2.experimental_data_stream_features;
      }
    }
    return packagePolicy;
  }
  async findAllForAgentPolicy(soClient, agentPolicyId, options = {}) {
    const logger = this.getLogger('findAllForAgentPolicy');
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    logger.debug(() => `Finding all package policies for agent policy [${agentPolicyId}] with options [${JSON.stringify(options)}], soClient scoped to [${soClient.getCurrentNamespace()}]`);
    const savedObjectType = await getPackagePolicySavedObjectType();
    const packagePolicySO = await soClient.find({
      type: savedObjectType,
      filter: `${savedObjectType}.attributes.policy_ids:${(0, _saved_object.escapeSearchQueryPhrase)(agentPolicyId)} AND ${savedObjectType}.attributes.latest_revision:true`,
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: isSpacesEnabled ? options.spaceIds : undefined
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Error encountered while attempting to get all package policies for agent policy [${agentPolicyId}]`));
    if (!packagePolicySO) {
      logger.debug(() => `No package policies found for agent policy id [${agentPolicyId}]`);
      return [];
    }
    const packagePolicies = packagePolicySO.saved_objects.map(so => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(so));
    for (const packagePolicy of packagePolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        name: packagePolicy.name,
        savedObjectType
      });
    }
    logger.debug(() => `returning ${packagePolicies.length} package policies for agent policy [${agentPolicyId}]`);
    return packagePolicies;
  }
  async getByIDs(soClient, ids, options = {}) {
    const logger = this.getLogger('getByIDs');
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    logger.debug(() => `Retrieving package policies [${ids.join(', ')}] using soClient scoped to [${soClient.getCurrentNamespace()}] and options [${JSON.stringify(options)}]`);
    const packagePolicySO = await soClient.bulkGet(ids.map(id => ({
      id,
      type: savedObjectType,
      namespaces: isSpacesEnabled ? options.spaceIds : undefined
    }))).catch(_utils.catchAndSetErrorStackTrace.withMessage('bulkGet of package policies failed'));
    const packagePolicies = packagePolicySO.saved_objects.map(so => {
      if (so.error) {
        if (options.ignoreMissing && so.error.statusCode === 404) {
          return null;
        } else if (so.error.statusCode === 404) {
          throw new _errors.PackagePolicyNotFoundError(`Package policy ${so.id} not found`, {
            packagePolicyId: so.id
          });
        } else {
          throw new _errors.FleetError(so.error.message);
        }
      }
      return (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(so);
    }).filter(packagePolicy => packagePolicy !== null);
    for (const packagePolicy of packagePolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: packagePolicy.id,
        name: packagePolicy.name,
        savedObjectType
      });
    }
    logger.debug(`returning [${packagePolicies.length}] package policies`);
    return packagePolicies;
  }
  async list(soClient, options) {
    const logger = this.getLogger('list');
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery,
      fields
    } = options;
    logger.debug(() => `Retrieving list of package policies with soClient scoped to [${soClient.getCurrentNamespace()}] and options:${JSON.stringify(options)}`);
    const filter = _normalizePackagePolicyKuery(savedObjectType, kuery ? `${savedObjectType}.attributes.latest_revision:true AND (${kuery})` : `${savedObjectType}.attributes.latest_revision:true`);
    const packagePolicies = await soClient.find({
      type: savedObjectType,
      sortField,
      sortOrder,
      page,
      perPage,
      fields,
      filter,
      namespaces: isSpacesEnabled && options.spaceId ? [options.spaceId] : undefined
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage('failed to find package policies'));
    for (const packagePolicy of (_packagePolicies$save = packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.saved_objects) !== null && _packagePolicies$save !== void 0 ? _packagePolicies$save : []) {
      var _packagePolicies$save;
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        name: packagePolicy.attributes.name,
        savedObjectType
      });
    }
    const response = {
      items: packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.saved_objects.map(so => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(so)),
      total: packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.total,
      page,
      perPage
    };
    logger.debug(`Query matched [${response.total}] package policies - returning page [${page}] with [${response.items.length}] polices`);
    return response;
  }
  async listIds(soClient, options) {
    const logger = this.getLogger('listIds');
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery
    } = options;
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    logger.debug(() => `Fetching list of package policies IDs with soClient scoped to [${soClient.getCurrentNamespace()}] using options [${JSON.stringify(options)}]`);
    const filter = _normalizePackagePolicyKuery(savedObjectType, kuery ? `${savedObjectType}.attributes.latest_revision:true AND (${kuery})` : `${savedObjectType}.attributes.latest_revision:true`);
    const packagePolicies = await soClient.find({
      type: savedObjectType,
      sortField,
      sortOrder,
      page,
      perPage,
      fields: ['name'],
      filter,
      namespaces: isSpacesEnabled ? options.spaceIds : undefined
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage('failed to find package policies IDs'));
    for (const packagePolicy of packagePolicies.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        name: packagePolicy.attributes.name,
        savedObjectType
      });
    }
    const response = {
      items: packagePolicies.saved_objects.map(packagePolicySO => packagePolicySO.id),
      total: packagePolicies.total,
      page,
      perPage
    };
    logger.debug(`Found [${response.total}] package policies - returning page [${page}] with [${response.items.length}] polices IDs`);
    return response;
  }
  async update(soClient, esClient, id, packagePolicyUpdate, options) {
    var _packagePolicy$packag4, _pkgInfo$elasticsearc2, _secretReferences2, _options$user$usernam5, _options$user5, _oldPackagePolicy$pac, _secretsToDelete;
    const logger = this.getLogger('update');
    logger.debug(() => `Updating package policy [${id}] with soClient scoped to [${soClient.getCurrentNamespace()}]`);
    const savedObjectType = await getPackagePolicySavedObjectType();
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id,
      name: packagePolicyUpdate.name,
      savedObjectType
    });
    this.keepPolicyIdInSync(packagePolicyUpdate);
    await (0, _package_policies.preflightCheckPackagePolicy)(soClient, packagePolicyUpdate);
    let enrichedPackagePolicy;
    let secretReferences;
    let secretsToDelete;
    try {
      logger.debug(`Starting update of package policy ${id}`);
      enrichedPackagePolicy = await packagePolicyService.runExternalCallbacks('packagePolicyUpdate', packagePolicyUpdate, soClient, esClient);
    } catch (error) {
      logger.error(`An error occurred executing "packagePolicyUpdate" callback: ${error}`);
      logger.error(error);
      if (error.apiPassThrough) {
        throw error;
      }
      enrichedPackagePolicy = packagePolicyUpdate;
    }
    const packagePolicy = {
      ...enrichedPackagePolicy,
      name: enrichedPackagePolicy.name.trim()
    };
    const oldPackagePolicy = await this.get(soClient, id);
    if (packagePolicyUpdate.is_managed && !(options !== null && options !== void 0 && options.force)) {
      throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot update package policy ${id}`);
    }
    if (!oldPackagePolicy) {
      throw new _errors.PackagePolicyNotFoundError('Package policy not found');
    } else {
      this.keepPolicyIdInSync(oldPackagePolicy);
    }
    const agentPolicies = [];
    for (const policyId of packagePolicyUpdate.policy_ids) {
      const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, policyId, false);
      if (!agentPolicy) {
        throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
      }
      agentPolicies.push(agentPolicy);
    }
    const spaceIds = agentPolicies.flatMap(ap => {
      var _ap$space_ids2;
      return (_ap$space_ids2 = ap.space_ids) !== null && _ap$space_ids2 !== void 0 ? _ap$space_ids2 : [];
    });
    if (packagePolicy.name && packagePolicy.name !== oldPackagePolicy.name && !(options !== null && options !== void 0 && options.skipUniqueNameVerification)) {
      await requireUniqueName(soClient, enrichedPackagePolicy, id, spaceIds);
    }
    if (packagePolicy.namespace) {
      await (0, _policy_namespaces.validatePolicyNamespaceForSpace)({
        namespace: packagePolicy.namespace,
        spaceId: soClient.getCurrentNamespace()
      });
    }
    await (0, _policy_namespaces.validateAdditionalDatastreamsPermissionsForSpace)({
      additionalDatastreamsPermissions: enrichedPackagePolicy.additional_datastreams_permissions,
      spaceId: soClient.getCurrentNamespace()
    });

    // eslint-disable-next-line prefer-const
    let {
      version,
      ...restOfPackagePolicy
    } = packagePolicy;
    if (!((_packagePolicy$packag4 = packagePolicy.package) !== null && _packagePolicy$packag4 !== void 0 && _packagePolicy$packag4.name)) {
      throw new _errors.FleetError('Package policy without package are not supported');
    }
    const pkgInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName: packagePolicy.package.name,
      pkgVersion: packagePolicy.package.version,
      prerelease: true
    });
    let inputs = (0, _get_input_with_ids.getInputsWithIds)(restOfPackagePolicy, oldPackagePolicy.id, undefined, pkgInfo);
    inputs = enforceFrozenInputs(oldPackagePolicy.inputs, inputs, options === null || options === void 0 ? void 0 : options.force);
    _validateRestrictedFieldsNotModifiedOrThrow({
      pkgInfo,
      oldPackagePolicy,
      packagePolicyUpdate
    });
    validatePackagePolicyOrThrow(packagePolicy, pkgInfo);
    (0, _package_policies.canDeployCustomPackageAsAgentlessOrThrow)(packagePolicy, pkgInfo);
    if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
      const secretsRes = await (0, _secrets.extractAndUpdateSecrets)({
        oldPackagePolicy,
        packagePolicyUpdate: {
          ...restOfPackagePolicy,
          inputs
        },
        packageInfo: pkgInfo,
        esClient
      });
      restOfPackagePolicy = secretsRes.packagePolicyUpdate;
      secretReferences = secretsRes.secretReferences;
      secretsToDelete = secretsRes.secretsToDelete;
      inputs = restOfPackagePolicy.inputs;
    }
    const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
      logger,
      packageInfo: pkgInfo,
      savedObjectsClient: soClient
    });
    inputs = _compilePackagePolicyInputs(pkgInfo, restOfPackagePolicy.vars || {}, inputs, assetsMap);
    const elasticsearchPrivileges = (_pkgInfo$elasticsearc2 = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc2 === void 0 ? void 0 : _pkgInfo$elasticsearc2.privileges;
    const requiresRoot = (0, _services.isRootPrivilegesRequired)(pkgInfo);
    if (restOfPackagePolicy.package && requiresRoot) {
      restOfPackagePolicy.package = {
        ...restOfPackagePolicy.package,
        requires_root: requiresRoot,
        fips_compatible: (0, _services.checkIntegrationFipsLooseCompatibility)(pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.name, pkgInfo)
      };
    }
    for (const agentPolicy of agentPolicies) {
      if (agentPolicy) {
        validateReusableIntegrationsAndSpaceAwareness(packagePolicy, [agentPolicy]);
      }

      // Validate that if supports_agentless is true, the package actually supports agentless
      if (packagePolicy.supports_agentless && !(0, _agentless_policy_helper.isAgentlessIntegration)(pkgInfo)) {
        throw new _errors.PackagePolicyValidationError(`Package "${pkgInfo.name}" does not support agentless deployment mode`);
      }
      (0, _agentless_policy_helper.validateDeploymentModesForInputs)(packagePolicy.inputs, agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless || packagePolicy.supports_agentless ? 'agentless' : 'default', pkgInfo);
    }

    // Handle component template/mappings updates for experimental features, e.g. synthetic source
    await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
      soClient,
      esClient,
      packagePolicy: restOfPackagePolicy
    });

    // If the package version has increased, save the previous package policy revision.
    if (_.appContextService.getExperimentalFeatures().enablePackageRollback && packagePolicy.package && oldPackagePolicy.package && (0, _gt.default)(packagePolicy.package.version, oldPackagePolicy.package.version)) {
      logger.debug(`Saving previous revision of package policy ${id} with package version ${oldPackagePolicy.version}`);
      const currentPackagePolicySO = await soClient.get(savedObjectType, id);
      const previousRevisionSO = {
        ...currentPackagePolicySO,
        id: `${id}:prev`,
        attributes: {
          ...currentPackagePolicySO.attributes,
          latest_revision: false
        }
      };
      try {
        await soClient.update(savedObjectType, `${id}:prev`, previousRevisionSO.attributes);
      } catch (error) {
        if (error.output.statusCode === 404) {
          await soClient.create(savedObjectType, previousRevisionSO.attributes, {
            id: `${id}:prev`
          });
        } else {
          throw error;
        }
      }
    }
    logger.debug(`Updating SO with revision ${oldPackagePolicy.revision + 1}`);
    await soClient.update(savedObjectType, id, {
      ...restOfPackagePolicy,
      ...(restOfPackagePolicy.package ? {
        package: (0, _lodash.omit)(restOfPackagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      ...(elasticsearchPrivileges && {
        elasticsearch: {
          privileges: elasticsearchPrivileges
        }
      }),
      ...(((_secretReferences2 = secretReferences) === null || _secretReferences2 === void 0 ? void 0 : _secretReferences2.length) && {
        secret_references: secretReferences
      }),
      revision: oldPackagePolicy.revision + 1,
      updated_at: new Date().toISOString(),
      updated_by: (_options$user$usernam5 = options === null || options === void 0 ? void 0 : (_options$user5 = options.user) === null || _options$user5 === void 0 ? void 0 : _options$user5.username) !== null && _options$user$usernam5 !== void 0 ? _options$user$usernam5 : 'system'
    }, {
      version
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`update of package policy [${id}] failed`));
    const newPolicy = await this.get(soClient, id);

    // if we have moved to an input package we need to create the index templates
    // for the package policy as input packages create index templates per package policy
    if (pkgInfo && pkgInfo.type === 'input' && oldPackagePolicy.package && ((_oldPackagePolicy$pac = oldPackagePolicy.package) === null || _oldPackagePolicy$pac === void 0 ? void 0 : _oldPackagePolicy$pac.version) !== pkgInfo.version) {
      if (oldPackagePolicy.package) {
        var _oldPackagePolicy$pac2, _oldPackagePolicy$pac3;
        const oldPackage = await (0, _packages.getPackageInfo)({
          savedObjectsClient: soClient,
          pkgName: (_oldPackagePolicy$pac2 = oldPackagePolicy.package) === null || _oldPackagePolicy$pac2 === void 0 ? void 0 : _oldPackagePolicy$pac2.name,
          pkgVersion: (_oldPackagePolicy$pac3 = oldPackagePolicy.package) === null || _oldPackagePolicy$pac3 === void 0 ? void 0 : _oldPackagePolicy$pac3.version,
          prerelease: true
        });
        if (oldPackage.type === 'integration') {
          await (0, _input_type_packages.installAssetsForInputPackagePolicy)({
            logger,
            soClient,
            esClient,
            pkgInfo,
            packagePolicy: newPolicy,
            force: true
          });
        }
      }
    }

    // Bump revision of all associated agent policies (old and new)
    const associatedPolicyIds = new Set([...oldPackagePolicy.policy_ids, ...newPolicy.policy_ids]);
    logger.debug(`Bumping revision of associated agent policies ${associatedPolicyIds}`);
    const bumpPromise = (0, _pMap.default)(associatedPolicyIds, policyId => {
      var _newPolicy$package;
      const isEndpointPolicy = ((_newPolicy$package = newPolicy.package) === null || _newPolicy$package === void 0 ? void 0 : _newPolicy$package.name) === 'endpoint';
      // Check if the agent policy is in both old and updated package policies
      const assignedInOldPolicy = oldPackagePolicy.policy_ids.includes(policyId);
      const assignedInNewPolicy = newPolicy.policy_ids.includes(policyId);

      // Remove protection if policy is unassigned (in old but not in updated) or policy is assigned (in updated but not in old)
      const removeProtection = isEndpointPolicy && (assignedInOldPolicy && !assignedInNewPolicy || !assignedInOldPolicy && assignedInNewPolicy);
      if ((options === null || options === void 0 ? void 0 : options.bumpRevision) !== false) {
        return _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, policyId, {
          user: options === null || options === void 0 ? void 0 : options.user,
          removeProtection
        });
      }
    }, {
      concurrency: _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
    });
    const assetRemovePromise = (0, _cleanup.removeOldAssets)({
      soClient,
      pkgName: newPolicy.package.name,
      currentVersion: newPolicy.package.version
    });
    const deleteSecretsPromise = (_secretsToDelete = secretsToDelete) !== null && _secretsToDelete !== void 0 && _secretsToDelete.length ? (0, _secrets.deleteSecretsIfNotReferenced)({
      esClient,
      soClient,
      ids: secretsToDelete.map(s => s.id)
    }) : Promise.resolve();
    await Promise.all([bumpPromise, assetRemovePromise, deleteSecretsPromise]);
    sendUpdatePackagePolicyTelemetryEvent(soClient, [packagePolicyUpdate], [oldPackagePolicy]);

    // Run external post-update callbacks and return
    const response = packagePolicyService.runExternalCallbacks('packagePolicyPostUpdate', newPolicy, soClient, esClient);
    logger.debug(`Package policy ${id} update completed`);
    return response;
  }
  async bulkUpdate(soClient, esClient, packagePolicyUpdates, options = {}) {
    const logger = this.getLogger('bulkUpdate');
    const savedObjectType = await getPackagePolicySavedObjectType();
    logger.debug(() => `Processing [${packagePolicyUpdates.length}] updates with soClient scoped to [${soClient.getCurrentNamespace()}]`);
    for (const packagePolicy of packagePolicyUpdates) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'update',
        id: packagePolicy.id,
        name: packagePolicy.name,
        savedObjectType
      });
    }
    const oldPackagePolicies = options !== null && options !== void 0 && options.oldPackagePolicies ? options === null || options === void 0 ? void 0 : options.oldPackagePolicies : await this.getByIDs(soClient, packagePolicyUpdates.map(p => p.id));
    if (!oldPackagePolicies || oldPackagePolicies.length === 0) {
      throw new _errors.PackagePolicyNotFoundError('Package policy not found');
    }
    const packageInfos = await getPackageInfoForPackagePolicies(packagePolicyUpdates, soClient, true);
    const oldPackageInfos = await getPackageInfoForPackagePolicies(oldPackagePolicies, soClient, true);
    const allPackageInfos = [...packageInfos.entries(), ...oldPackageInfos.entries()].reduce((acc, [pkgKey, pkgInfo]) => {
      acc.set(pkgKey, pkgInfo);
      return acc;
    }, new Map());
    const allSecretsToDelete = [];
    const packageInfosandAssetsMap = await getPkgInfoAssetsMap({
      logger,
      packageInfos: [...packageInfos.values()],
      savedObjectsClient: soClient
    });
    const policiesToUpdate = [];
    const failedPolicies = [];
    const previousPolicyRevisionsToCreate = [];
    const previousPolicyRevisionsToUpdate = [];
    const secretStorageEnabled = await (0, _secrets.isSecretStorageEnabled)(esClient, soClient);
    const assetsToInstallFn = [];
    await (0, _pMap.default)(packagePolicyUpdates, async packagePolicyUpdate => {
      try {
        var _packagePolicy$packag5, _pkgInfo$elasticsearc3, _secretReferences3, _options$user$usernam6, _options$user6;
        const id = packagePolicyUpdate.id;
        this.keepPolicyIdInSync(packagePolicyUpdate);
        let enrichedPackagePolicy;
        try {
          logger.debug(`Starting update of package policy ${id}`);
          enrichedPackagePolicy = await packagePolicyService.runExternalCallbacks('packagePolicyUpdate', packagePolicyUpdate, soClient, esClient);
        } catch (error) {
          logger.error(`An error occurred executing "packagePolicyUpdate" callback: ${error}`);
          logger.error(error);
          if (error.apiPassThrough) {
            throw error;
          }
          enrichedPackagePolicy = packagePolicyUpdate;
        }
        const packagePolicy = {
          ...enrichedPackagePolicy,
          name: enrichedPackagePolicy.name.trim()
        };
        await (0, _package_policies.preflightCheckPackagePolicy)(soClient, packagePolicy);
        const oldPackagePolicy = oldPackagePolicies.find(p => p.id === id);
        if (!oldPackagePolicy) {
          throw new _errors.PackagePolicyNotFoundError(`Package policy [${id}] not found`, {
            packagePolicyId: id
          });
        } else {
          this.keepPolicyIdInSync(oldPackagePolicy);
        }

        // If the package version has increased, save the previous package policy revision.
        if (_.appContextService.getExperimentalFeatures().enablePackageRollback && packagePolicy.package && oldPackagePolicy.package && (0, _gt.default)(packagePolicy.package.version, oldPackagePolicy.package.version) && !packagePolicyUpdate.is_managed) {
          const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
          // checking is_managed flag on agent policy, it is not always set on package policy
          const agentPolicies = await _agent_policy.agentPolicyService.getByIds(internalSoClientWithoutSpaceExtension, packagePolicy.policy_ids.map(policyId => ({
            id: policyId,
            spaceId: '*'
          })));
          validateReusableIntegrationsAndSpaceAwareness(packagePolicy, agentPolicies);
          if (!agentPolicies.some(policy => policy.is_managed)) {
            logger.debug(`Saving previous revision of package policy ${id} with package version ${oldPackagePolicy.version}`);
            const currentPackagePolicySO = await soClient.get(savedObjectType, id);
            const previousRevisionSO = {
              ...currentPackagePolicySO,
              id: `${id}:prev`,
              attributes: {
                ...currentPackagePolicySO.attributes,
                latest_revision: false
              }
            };
            try {
              await soClient.get(savedObjectType, `${id}:prev`);
              previousPolicyRevisionsToUpdate.push(previousRevisionSO);
            } catch (error) {
              if (error.output.statusCode === 404) {
                previousPolicyRevisionsToCreate.push(previousRevisionSO);
              } else {
                throw error;
              }
            }
          }
        }
        let secretReferences;
        const {
          version
        } = packagePolicyUpdate;
        // id and version are not part of the saved object attributes
        // eslint-disable-next-line prefer-const
        let {
          version: _version,
          id: _id,
          ...restOfPackagePolicy
        } = packagePolicy;
        if (packagePolicyUpdate.is_managed && !(options !== null && options !== void 0 && options.force)) {
          throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot update package policy ${id}`);
        }
        if (!((_packagePolicy$packag5 = packagePolicy.package) !== null && _packagePolicy$packag5 !== void 0 && _packagePolicy$packag5.name)) {
          throw new _errors.FleetError('Package policies without package are not supported');
        }
        const pkgInfoAndAsset = packageInfosandAssetsMap.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
        if (!pkgInfoAndAsset) {
          throw new _errors.FleetError('Package info and assets not found');
        }
        const {
          pkgInfo,
          assetsMap
        } = pkgInfoAndAsset;
        let inputs = (0, _get_input_with_ids.getInputsWithIds)(restOfPackagePolicy, oldPackagePolicy.id, undefined, pkgInfo);
        inputs = enforceFrozenInputs(oldPackagePolicy.inputs, inputs, options === null || options === void 0 ? void 0 : options.force);
        validatePackagePolicyOrThrow(packagePolicy, pkgInfo);
        (0, _package_policies.canDeployCustomPackageAsAgentlessOrThrow)(packagePolicy, pkgInfo);
        if (secretStorageEnabled) {
          const secretsRes = await (0, _secrets.extractAndUpdateSecrets)({
            oldPackagePolicy,
            packagePolicyUpdate: {
              ...restOfPackagePolicy,
              inputs
            },
            packageInfo: pkgInfo,
            esClient
          });
          restOfPackagePolicy = secretsRes.packagePolicyUpdate;
          secretReferences = secretsRes.secretReferences;
          allSecretsToDelete.push(...secretsRes.secretsToDelete);
          inputs = restOfPackagePolicy.inputs;
        }
        inputs = _compilePackagePolicyInputs(pkgInfo, restOfPackagePolicy.vars || {}, inputs, assetsMap);
        const elasticsearchPrivileges = (_pkgInfo$elasticsearc3 = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc3 === void 0 ? void 0 : _pkgInfo$elasticsearc3.privileges;
        const requiresRoot = (0, _services.isRootPrivilegesRequired)(pkgInfo);
        if (restOfPackagePolicy.package && requiresRoot) {
          restOfPackagePolicy.package = {
            ...restOfPackagePolicy.package,
            requires_root: requiresRoot,
            fips_compatible: (0, _services.checkIntegrationFipsLooseCompatibility)(pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.name, pkgInfo)
          };
        }
        if (pkgInfo && pkgInfo.type === 'input' && oldPackagePolicy.package && oldPackagePolicy.package.version !== pkgInfo.version) {
          const oldPackageInfo = allPackageInfos.get(`${oldPackagePolicy.package.name}-${oldPackagePolicy.package.version}`);
          if ((oldPackageInfo === null || oldPackageInfo === void 0 ? void 0 : oldPackageInfo.type) === 'integration') {
            assetsToInstallFn.push(async () => {
              const updatedPackagePolicy = await this.get(soClient, id);
              if (!updatedPackagePolicy) {
                return;
              }
              await (0, _input_type_packages.installAssetsForInputPackagePolicy)({
                logger,
                soClient,
                esClient,
                pkgInfo,
                packagePolicy: updatedPackagePolicy,
                force: true
              });
            });
          }
        }
        if (!(options !== null && options !== void 0 && options.fromBulkUpgrade)) {
          // Handle component template/mappings updates for experimental features, e.g. synthetic source
          await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
            soClient,
            esClient,
            packagePolicy
          });
        }
        policiesToUpdate.push({
          type: savedObjectType,
          id,
          attributes: {
            ...restOfPackagePolicy,
            ...(restOfPackagePolicy.package ? {
              package: (0, _lodash.omit)(restOfPackagePolicy.package, 'experimental_data_stream_features')
            } : {}),
            inputs,
            ...(elasticsearchPrivileges && {
              elasticsearch: {
                privileges: elasticsearchPrivileges
              }
            }),
            ...(((_secretReferences3 = secretReferences) === null || _secretReferences3 === void 0 ? void 0 : _secretReferences3.length) && {
              secret_references: secretReferences
            }),
            revision: oldPackagePolicy.revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: (_options$user$usernam6 = options === null || options === void 0 ? void 0 : (_options$user6 = options.user) === null || _options$user6 === void 0 ? void 0 : _options$user6.username) !== null && _options$user$usernam6 !== void 0 ? _options$user$usernam6 : 'system'
          },
          version
        });
        logger.debug(`saved object update document for package policy [${id}] was built`);
      } catch (error) {
        logger.debug(() => `Preparing update to package policy [${packagePolicyUpdate.id}] failed:\n${JSON.stringify(error)}`);
        failedPolicies.push({
          packagePolicy: packagePolicyUpdate,
          error
        });
      }
    });

    // Store previous revision.
    if (_.appContextService.getExperimentalFeatures().enablePackageRollback) {
      if (previousPolicyRevisionsToCreate.length > 0) {
        await soClient.bulkCreate(previousPolicyRevisionsToCreate).catch(_utils.catchAndSetErrorStackTrace.withMessage('Saved objects bulk create of previous package policy revisions failed'));
      }
      if (previousPolicyRevisionsToUpdate.length > 0) {
        await soClient.bulkUpdate(previousPolicyRevisionsToUpdate).catch(_utils.catchAndSetErrorStackTrace.withMessage('Saved objects bulk update of previous package policy revisions failed'));
      }
    }

    // Update package policies SO.
    const {
      saved_objects: updatedPolicies
    } = await soClient.bulkUpdate(policiesToUpdate).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Saved objects bulk update failed`));

    // Bump revision of all associated agent policies (old and new)
    const associatedPolicyIds = new Set([...packagePolicyUpdates.flatMap(p => p.policy_ids), ...oldPackagePolicies.flatMap(p => p.policy_ids)]);
    const [endpointPackagePolicyUpdatesIds, endpointOldPackagePoliciesIds] = [packagePolicyUpdates, oldPackagePolicies].map(packagePolicies => new Set(packagePolicies.filter(p => {
      var _p$package;
      return ((_p$package = p.package) === null || _p$package === void 0 ? void 0 : _p$package.name) === 'endpoint';
    }).map(p => p.policy_ids).flat()));
    const installAssetsPromise = (0, _pMap.default)(assetsToInstallFn, fn => fn(), {
      concurrency: _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_10
    });
    const bumpPromise = (0, _pMap.default)(associatedPolicyIds, async agentPolicyId => {
      // Check if the agent policy is in both old and updated package policies
      const assignedInOldPolicies = endpointOldPackagePoliciesIds.has(agentPolicyId);
      const assignedInUpdatedPolicies = endpointPackagePolicyUpdatesIds.has(agentPolicyId);

      // Remove protection if policy is unassigned (in old but not in updated) or policy is assigned (in updated but not in old)
      const removeProtection = assignedInOldPolicies && !assignedInUpdatedPolicies || !assignedInOldPolicies && assignedInUpdatedPolicies;
      logger.debug(`bumping revision for agent policy id [${agentPolicyId}]`);
      await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, agentPolicyId, {
        user: options === null || options === void 0 ? void 0 : options.user,
        removeProtection,
        asyncDeploy: options === null || options === void 0 ? void 0 : options.asyncDeploy
      });
    }).finally(() => {
      logger.debug(`bumping of revision for associated agent policies done`);
    });
    const pkgVersions = {};
    packagePolicyUpdates.forEach(({
      package: pkg
    }) => {
      if (pkg) {
        pkgVersions[pkg.name + '-' + pkg.version] = {
          name: pkg.name,
          version: pkg.version
        };
      }
    });
    const removeAssetPromise = (0, _pMap.default)(Object.keys(pkgVersions), async pkgVersion => {
      const {
        name,
        version
      } = pkgVersions[pkgVersion];
      await (0, _cleanup.removeOldAssets)({
        soClient,
        pkgName: name,
        currentVersion: version
      });
    });
    const deleteSecretsPromise = allSecretsToDelete.length ? (0, _secrets.deleteSecretsIfNotReferenced)({
      esClient,
      soClient,
      ids: allSecretsToDelete.map(s => s.id)
    }) : Promise.resolve();
    await Promise.all([bumpPromise, removeAssetPromise, deleteSecretsPromise, installAssetsPromise]);
    sendUpdatePackagePolicyTelemetryEvent(soClient, packagePolicyUpdates, oldPackagePolicies);
    updatedPolicies.forEach(policy => {
      if (policy.error) {
        const hasAlreadyFailed = failedPolicies.some(failedPolicy => failedPolicy.packagePolicy.id === policy.id);
        if (!hasAlreadyFailed) {
          failedPolicies.push({
            packagePolicy: packagePolicyUpdates.find(p => p.id === policy.id),
            error: policy.error
          });
        }
      }
    });
    let updatedPoliciesSuccess = updatedPolicies.filter(policy => !policy.error && policy.attributes).map(soPolicy => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(soPolicy));
    updatedPoliciesSuccess = (await (0, _pMap.default)((0, _lodash.chunk)(updatedPoliciesSuccess, _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS), async updatedPoliciesChunk => {
      const updatedPoliciesComplete = await this.getByIDs(soClient, updatedPoliciesChunk.map(p => p.id));
      return (0, _pMap.default)(updatedPoliciesComplete, packagePolicy => packagePolicyService.runExternalCallbacks('packagePolicyPostUpdate', packagePolicy, soClient, esClient));
    }, {
      concurrency: _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_10
    })).flat();
    logger.debug(() => `Done. Successful updates: [${updatedPoliciesSuccess.length}] | Failures: [${failedPolicies.length}]`);
    return {
      updatedPolicies: updatedPoliciesSuccess,
      failedPolicies
    };
  }
  async delete(soClient, esClient, ids, options, context, request) {
    const savedObjectType = await getPackagePolicySavedObjectType();
    for (const id of ids) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'delete',
        id,
        savedObjectType
      });
    }
    const result = [];
    const logger = this.getLogger('delete');
    logger.debug(`Deleting package policies ${ids}`);
    const packagePolicies = await this.getByIDs(soClient, ids, {
      ignoreMissing: true,
      spaceIds: options === null || options === void 0 ? void 0 : options.spaceIds
    });
    if (!packagePolicies || packagePolicies.length === 0) {
      logger.debug(`No package policies to delete`);
      return [];
    }
    try {
      await packagePolicyService.runDeleteExternalCallbacks(packagePolicies, soClient, esClient, context, request);
    } catch (error) {
      logger.error(`An error occurred executing "packagePolicyDelete" callback: ${error}`);
      logger.error(error);
    }
    const uniqueAgentPolicyIds = [...new Set(packagePolicies.flatMap(packagePolicy => packagePolicy.policy_ids))];
    const hostedAgentPolicies = [];
    const agentlessAgentPolicies = [];
    for (const agentPolicyId of uniqueAgentPolicyIds) {
      try {
        const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, agentPolicyId);
        if (!agentPolicy) {
          throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
        }
        validateIsNotHostedPolicy(agentPolicy, options === null || options === void 0 ? void 0 : options.force, 'Cannot remove integrations of hosted agent policy');
        // collect agentless agent policies to delete
        if (agentPolicy.supports_agentless) {
          agentlessAgentPolicies.push(agentPolicyId);
        }
      } catch (e) {
        var _e$output, _e$output$payload, _e$output2;
        logger.error(`An error occurred while checking if policies are hosted: ${e === null || e === void 0 ? void 0 : (_e$output = e.output) === null || _e$output === void 0 ? void 0 : (_e$output$payload = _e$output.payload) === null || _e$output$payload === void 0 ? void 0 : _e$output$payload.message}`);
        // in case of orphaned policies don't add the id to the hostedAgentPolicies array
        if ((e === null || e === void 0 ? void 0 : (_e$output2 = e.output) === null || _e$output2 === void 0 ? void 0 : _e$output2.statusCode) !== 404) hostedAgentPolicies.push(agentPolicyId);
      }
    }
    const idsToDelete = [];
    ids.forEach(id => {
      try {
        const packagePolicy = packagePolicies.find(p => p.id === id);
        if (!packagePolicy) {
          throw new _errors.PackagePolicyNotFoundError(`Saved object [${savedObjectType}/${id}] not found`);
        }
        if (packagePolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
          throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot delete package policy ${id}`);
        }
        if (packagePolicy.policy_ids.some(policyId => hostedAgentPolicies.includes(policyId))) {
          throw new _errors.HostedAgentPolicyRestrictionRelatedError('Cannot remove integrations of hosted agent policy');
        }
        idsToDelete.push(id);
      } catch (error) {
        result.push({
          id,
          success: false,
          ...(0, _errors.fleetErrorToResponseOptions)(error)
        });
      }
    });
    const secretsToDelete = [];
    if (idsToDelete.length > 0) {
      const {
        statuses
      } = await soClient.bulkDelete(idsToDelete.map(id => ({
        id,
        type: savedObjectType
      })), {
        force: true // need to delete through multiple space
      }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Bulk delete of package policies [${idsToDelete.join(', ')}] failed`));
      statuses.forEach(({
        id,
        success,
        error
      }) => {
        const packagePolicy = packagePolicies.find(p => p.id === id);
        if (success && packagePolicy) {
          var _packagePolicy$packag6, _packagePolicy$packag7, _packagePolicy$packag8, _packagePolicy$secret;
          result.push({
            id,
            name: packagePolicy.name,
            success: true,
            package: {
              name: ((_packagePolicy$packag6 = packagePolicy.package) === null || _packagePolicy$packag6 === void 0 ? void 0 : _packagePolicy$packag6.name) || '',
              title: ((_packagePolicy$packag7 = packagePolicy.package) === null || _packagePolicy$packag7 === void 0 ? void 0 : _packagePolicy$packag7.title) || '',
              version: ((_packagePolicy$packag8 = packagePolicy.package) === null || _packagePolicy$packag8 === void 0 ? void 0 : _packagePolicy$packag8.version) || ''
            },
            policy_id: packagePolicy.policy_id,
            policy_ids: packagePolicy.policy_ids
          });
          if (packagePolicy !== null && packagePolicy !== void 0 && (_packagePolicy$secret = packagePolicy.secret_references) !== null && _packagePolicy$secret !== void 0 && _packagePolicy$secret.length) {
            secretsToDelete.push(...packagePolicy.secret_references.map(s => s.id));
          }
        } else if (!success && error) {
          result.push({
            id,
            success: false,
            statusCode: error.statusCode,
            body: {
              message: error.message
            }
          });
        }
      });
    }
    if (!(options !== null && options !== void 0 && options.skipUnassignFromAgentPolicies)) {
      let uniquePolicyIdsR = [...new Set(result.filter(r => r.success && r.policy_ids && r.policy_ids.length > 0).flatMap(r => r.policy_ids))];
      uniquePolicyIdsR = (0, _lodash.without)(uniquePolicyIdsR, ...agentlessAgentPolicies);
      const agentPoliciesWithEndpointPackagePolicies = result.reduce((acc, cur) => {
        var _cur$package;
        if (cur.success && cur.policy_ids && ((_cur$package = cur.package) === null || _cur$package === void 0 ? void 0 : _cur$package.name) === 'endpoint') {
          for (const policyId of cur.policy_ids) {
            acc.add(policyId);
          }
        }
        return acc;
      }, new Set());
      const agentPolicies = await _agent_policy.agentPolicyService.getByIds(soClient, uniquePolicyIdsR);
      for (const policyId of uniquePolicyIdsR) {
        const agentPolicy = agentPolicies.find(p => p.id === policyId);
        if (agentPolicy) {
          // is the agent policy attached to package policy with endpoint
          await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, policyId, {
            user: options === null || options === void 0 ? void 0 : options.user,
            removeProtection: agentPoliciesWithEndpointPackagePolicies.has(policyId)
          });
        }
      }
    }
    if (secretsToDelete.length > 0) {
      await (0, _secrets.deleteSecretsIfNotReferenced)({
        esClient,
        soClient,
        ids: secretsToDelete
      });
    }
    try {
      await packagePolicyService.runPostDeleteExternalCallbacks(result, soClient, esClient, context, request);
      logger.debug(`Deleted package policies ${ids}`);
    } catch (error) {
      logger.error(`An error occurred executing "packagePolicyPostDelete" callback: ${error}`);
      logger.error(error);
    }
    return result;
  }
  async bulkUpgrade(soClient, esClient, ids, options, pkgVersion) {
    return (0, _upgrade._packagePoliciesBulkUpgrade)({
      packagePolicyService: this,
      soClient,
      esClient,
      ids,
      options,
      pkgVersion
    });
  }
  async upgrade(soClient, esClient, id, options, packagePolicy, pkgVersion) {
    return (0, _upgrade._packagePoliciesUpgrade)({
      packagePolicyService: this,
      soClient,
      esClient,
      id,
      options,
      packagePolicy,
      pkgVersion
    });
  }
  async getUpgradeDryRunDiff(soClient, id, packagePolicy, pkgVersion) {
    return (0, _upgrade._packagePoliciesGetUpgradeDryRunDiff)({
      packagePolicyService: this,
      soClient,
      id,
      packagePolicy,
      pkgVersion
    });
  }
  async enrichPolicyWithDefaultsFromPackage(soClient, newPolicy) {
    let newPackagePolicy = newPolicy;
    if (newPolicy.package) {
      const newPP = await this.buildPackagePolicyFromPackageWithVersion(soClient, newPolicy.package.name, newPolicy.package.version);
      if (newPP) {
        var _newPolicy$namespace, _newPolicy$descriptio, _newPolicy$enabled, _newPolicy$package2, _newPolicy$policy_id, _newPolicy$policy_ids, _newPolicy$inputs$;
        const inputs = newPolicy.inputs.map(input => {
          var _defaultInput$streams;
          const defaultInput = newPP.inputs.find(i => i.type === input.type && (!input.policy_template || input.policy_template === i.policy_template));
          return {
            ...defaultInput,
            enabled: input.enabled,
            type: input.type,
            // to propagate "enabled: false" to streams
            streams: defaultInput === null || defaultInput === void 0 ? void 0 : (_defaultInput$streams = defaultInput.streams) === null || _defaultInput$streams === void 0 ? void 0 : _defaultInput$streams.map(stream => ({
              ...stream,
              enabled: input.enabled
            }))
          };
        });
        newPackagePolicy = {
          ...newPP,
          name: newPolicy.name,
          namespace: (_newPolicy$namespace = newPolicy === null || newPolicy === void 0 ? void 0 : newPolicy.namespace) !== null && _newPolicy$namespace !== void 0 ? _newPolicy$namespace : '',
          description: (_newPolicy$descriptio = newPolicy.description) !== null && _newPolicy$descriptio !== void 0 ? _newPolicy$descriptio : '',
          enabled: (_newPolicy$enabled = newPolicy.enabled) !== null && _newPolicy$enabled !== void 0 ? _newPolicy$enabled : true,
          package: {
            ...newPP.package,
            experimental_data_stream_features: (_newPolicy$package2 = newPolicy.package) === null || _newPolicy$package2 === void 0 ? void 0 : _newPolicy$package2.experimental_data_stream_features
          },
          policy_id: (_newPolicy$policy_id = newPolicy.policy_id) !== null && _newPolicy$policy_id !== void 0 ? _newPolicy$policy_id : undefined,
          policy_ids: (_newPolicy$policy_ids = newPolicy.policy_ids) !== null && _newPolicy$policy_ids !== void 0 ? _newPolicy$policy_ids : undefined,
          output_id: newPolicy.output_id,
          inputs: (_newPolicy$inputs$ = newPolicy.inputs[0]) !== null && _newPolicy$inputs$ !== void 0 && _newPolicy$inputs$.streams ? newPolicy.inputs : inputs,
          vars: newPolicy.vars || newPP.vars,
          supports_agentless: newPolicy.supports_agentless,
          supports_cloud_connector: newPolicy.supports_cloud_connector,
          cloud_connector_id: newPolicy.cloud_connector_id,
          additional_datastreams_permissions: newPolicy.additional_datastreams_permissions
        };
      }
    }
    return newPackagePolicy;
  }
  async buildPackagePolicyFromPackageWithVersion(soClient, pkgName, pkgVersion) {
    const packageInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName,
      pkgVersion,
      skipArchive: true,
      prerelease: true
    });
    if (packageInfo) {
      return (0, _services.packageToPackagePolicy)(packageInfo, '');
    }
  }
  async buildPackagePolicyFromPackage(soClient, pkgName, options) {
    const pkgInstallObj = await (0, _packages.getInstallationObject)({
      savedObjectsClient: soClient,
      pkgName,
      logger: options === null || options === void 0 ? void 0 : options.logger
    });
    let pkgInstall = pkgInstallObj === null || pkgInstallObj === void 0 ? void 0 : pkgInstallObj.attributes;
    if (!pkgInstall && options !== null && options !== void 0 && options.installMissingPackage) {
      const esClient = await _.appContextService.getInternalUserESClient();
      const result = await (0, _packages.ensureInstalledPackage)({
        esClient,
        pkgName,
        savedObjectsClient: soClient
      });
      if (result.package) {
        pkgInstall = result.package;
      }
    }
    if (pkgInstall) {
      const packageInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: pkgInstall.name,
        pkgVersion: pkgInstall.version,
        prerelease: true
      });
      if (packageInfo) {
        return (0, _services.packageToPackagePolicy)(packageInfo, '');
      }
    }
  }
  async runExternalCallbacks(externalCallbackType, packagePolicy, soClient, esClient, context, request) {
    var _appContextService$ge;
    const logger = this.getLogger('runExternalCallbacks');
    const numberOfCallbacks = (_appContextService$ge = _.appContextService.getExternalCallbacks(externalCallbackType)) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.size;
    let runResult;
    logger.debug(`Running [${numberOfCallbacks}] external callbacks for [${externalCallbackType}]`);
    try {
      if (externalCallbackType === 'packagePolicyPostDelete') {
        runResult = await this.runPostDeleteExternalCallbacks(packagePolicy, soClient, esClient, context, request);
      } else if (externalCallbackType === 'packagePolicyDelete') {
        runResult = await this.runDeleteExternalCallbacks(packagePolicy, soClient, esClient);
      } else {
        if (!Array.isArray(packagePolicy)) {
          let newData = packagePolicy;
          const externalCallbacks = _.appContextService.getExternalCallbacks(externalCallbackType);
          if (externalCallbacks && externalCallbacks.size > 0) {
            let updatedNewData = newData;
            for (const callback of externalCallbacks) {
              let thisCallbackResponse;
              try {
                if (externalCallbackType === 'packagePolicyPostCreate') {
                  thisCallbackResponse = await callback(updatedNewData, soClient, esClient, context, request);
                  updatedNewData = _types.PackagePolicySchema.validate((0, _lodash.omit)(thisCallbackResponse, 'spaceIds'));
                } else if (externalCallbackType === 'packagePolicyPostUpdate') {
                  thisCallbackResponse = await callback(updatedNewData, soClient, esClient, context, request);
                  updatedNewData = _types.PackagePolicySchema.validate((0, _lodash.omit)(thisCallbackResponse, 'spaceIds'));
                } else {
                  thisCallbackResponse = await callback(updatedNewData, soClient, esClient, context, request);
                }
                if (externalCallbackType === 'packagePolicyCreate') {
                  updatedNewData = _types.NewPackagePolicySchema.validate((0, _lodash.omit)(thisCallbackResponse, 'spaceIds'));
                } else if (externalCallbackType === 'packagePolicyUpdate') {
                  const omitted = {
                    ...(0, _lodash.omit)(thisCallbackResponse, ['id', 'spaceIds', 'version', 'revision', 'updated_at', 'updated_by', 'created_at', 'created_by', 'elasticsearch']),
                    inputs: thisCallbackResponse.inputs.map(input => (0, _lodash.omit)(input, ['compiled_input']))
                  };
                  updatedNewData = _types.UpdatePackagePolicySchema.validate(omitted);
                }
              } catch (callbackError) {
                logger.debug(() => `The following [${externalCallbackType}] external callback threw an error (first 1k characters of callback):\n${callback.toString().substring(0, 1000)}`);

                // Convert schema validation errors to Fleet errors
                if (callbackError.constructor.name === 'ValidationError' || callbackError.name === 'ValidationError') {
                  throw new _errors.PackagePolicyValidationError(callbackError.message);
                }
                throw callbackError;
              }
            }
            newData = updatedNewData;
          }
          runResult = newData;
        }
      }
    } catch (error) {
      logger.error(`Error running external callbacks for [${externalCallbackType}]:`);
      logger.error(error);
      throw error;
    }
    logger.debug(`Running of [${externalCallbackType}] external callbacks done`);
    return runResult;
  }
  async runPostDeleteExternalCallbacks(deletedPackagePolicies, soClient, esClient, context, request) {
    const logger = this.getLogger('runPostDeleteExternalCallbacks');
    const externalCallbacks = _.appContextService.getExternalCallbacks('packagePolicyPostDelete');
    const errorsThrown = [];
    logger.debug(`Running [${externalCallbacks === null || externalCallbacks === void 0 ? void 0 : externalCallbacks.size}] external callbacks`);
    if (externalCallbacks && externalCallbacks.size > 0) {
      for (const callback of externalCallbacks) {
        // Failures from an external callback should not prevent other external callbacks from being
        // executed. Errors (if any) will be collected and `throw`n after processing the entire set
        try {
          await callback(deletedPackagePolicies, soClient, esClient, context, request);
        } catch (error) {
          logger.debug(() => `The following external callback threw error:\n${error.message}\n${error.stack}\nFirst 1k characters of callback:\n${callback.toString().substring(0, 1000)}`);
          errorsThrown.push(error);
        }
      }
      if (errorsThrown.length > 0) {
        throw new _errors.FleetError(`${errorsThrown.length} errors encountered while executing package post delete external callbacks`, errorsThrown);
      }
    }
    logger.debug(`Running of external callbacks done`);
  }
  async runDeleteExternalCallbacks(deletedPackagePolices, soClient, esClient) {
    const logger = this.getLogger('runDeleteExternalCallbacks');
    const externalCallbacks = _.appContextService.getExternalCallbacks('packagePolicyDelete');
    const errorsThrown = [];
    logger.debug(`Running [${externalCallbacks === null || externalCallbacks === void 0 ? void 0 : externalCallbacks.size}] external callbacks`);
    if (externalCallbacks && externalCallbacks.size > 0) {
      for (const callback of externalCallbacks) {
        // Failures from an external callback should not prevent other external callbacks from being
        // executed. Errors (if any) will be collected and `throw`n after processing the entire set
        try {
          await callback(deletedPackagePolices, soClient, esClient);
        } catch (error) {
          logger.debug(() => `The following external callback threw error:\n${error.message}\n${error.stack}\nFirst 1k characters of callback:\n${callback.toString().substring(0, 1000)}`);
          errorsThrown.push(error);
        }
      }
      if (errorsThrown.length > 0) {
        throw new _errors.FleetError(`${errorsThrown.length} errors encountered while executing package delete external callbacks`, errorsThrown);
      }
    }
    logger.debug(`Running of package policy delete external callbacks done`);
  }
  async removeOutputFromAll(esClient, outputId, options) {
    const savedObjectType = await getPackagePolicySavedObjectType();
    const packagePolicies = (await _.appContextService.getInternalUserSOClientWithoutSpaceExtension().find({
      type: savedObjectType,
      fields: ['name', 'enabled', 'policy_ids', 'inputs', 'output_id'],
      searchFields: ['output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    })).saved_objects.map(so => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(so));
    if (packagePolicies.length > 0) {
      const getPackagePolicyUpdate = packagePolicy => ({
        name: packagePolicy.name,
        enabled: packagePolicy.enabled,
        policy_ids: packagePolicy.policy_ids,
        inputs: packagePolicy.inputs,
        output_id: packagePolicy.output_id === outputId ? null : packagePolicy.output_id,
        package: packagePolicy.package
      });

      // Validate that the new cleared/default output is valid for the package policies
      // (from each of the associated agent policies) before updating any of them
      await (0, _pMap.default)(packagePolicies, async packagePolicy => {
        var _packagePolicy$spaceI;
        const soClient = _.appContextService.getInternalUserSOClientForSpaceId((_packagePolicy$spaceI = packagePolicy.spaceIds) === null || _packagePolicy$spaceI === void 0 ? void 0 : _packagePolicy$spaceI[0]);
        const existingPackagePolicy = await this.get(soClient, packagePolicy.id);
        if (!existingPackagePolicy) {
          throw new _errors.PackagePolicyNotFoundError('Package policy not found');
        }
        for (const policyId of packagePolicy.policy_ids) {
          var _packagePolicy$packag9;
          if ((_packagePolicy$packag9 = packagePolicy.package) !== null && _packagePolicy$packag9 !== void 0 && _packagePolicy$packag9.name) {
            const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, policyId, true);
            if (agentPolicy) {
              await (0, _outputs_helpers.validateAgentPolicyOutputForIntegration)(soClient, agentPolicy, packagePolicy, packagePolicy.package.name, false);
            }
          }
        }
      }, {
        concurrency: _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
      await (0, _pMap.default)(packagePolicies, packagePolicy => {
        var _packagePolicy$spaceI2;
        const soClient = _.appContextService.getInternalUserSOClientForSpaceId((_packagePolicy$spaceI2 = packagePolicy.spaceIds) === null || _packagePolicy$spaceI2 === void 0 ? void 0 : _packagePolicy$spaceI2[0]);
        return this.update(soClient, esClient, packagePolicy.id, getPackagePolicyUpdate(packagePolicy), {
          force: options === null || options === void 0 ? void 0 : options.force
        });
      }, {
        concurrency: _constants3.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
    }
  }
  async fetchAllItemIds(soClient, {
    perPage = 1000,
    kuery,
    spaceIds
  } = {}) {
    const logger = this.getLogger('fetchAllItemIds');
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    const namespaces = isSpacesEnabled ? spaceIds : undefined;
    logger.debug(() => `fetching all items ids using soClient scoped to [${soClient.getCurrentNamespace()}] and kuery [${kuery}] and spaceIds [${spaceIds === null || spaceIds === void 0 ? void 0 : spaceIds.join(', ')}]`);
    const filter = _normalizePackagePolicyKuery(savedObjectType, kuery ? `${savedObjectType}.attributes.latest_revision:true AND (${kuery})` : `${savedObjectType}.attributes.latest_revision:true`);
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: savedObjectType,
        perPage,
        sortField: 'created_at',
        sortOrder: 'asc',
        fields: [],
        filter,
        namespaces
      },
      resultsMapper: data => {
        return data.saved_objects.map(packagePolicySO => packagePolicySO.id);
      }
    });
  }
  async fetchAllItems(soClient, options = {}) {
    const logger = this.getLogger('fetchAllItems');
    logger.debug(() => `Fetching all items using soClient scoped to [${soClient.getCurrentNamespace()}] and options: ${JSON.stringify(options)}`);
    const {
      perPage = 1000,
      kuery,
      sortOrder = 'asc',
      sortField = 'created_at',
      spaceIds
    } = options;
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    const namespaces = isSpacesEnabled ? spaceIds : undefined;
    const filter = _normalizePackagePolicyKuery(savedObjectType, kuery ? `${savedObjectType}.attributes.latest_revision:true AND (${kuery})` : `${savedObjectType}.attributes.latest_revision:true`);
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: savedObjectType,
        sortField,
        sortOrder,
        perPage,
        filter,
        namespaces
      },
      resultsMapper(data) {
        return data.saved_objects.map(packagePolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: packagePolicySO.id,
            name: packagePolicySO.attributes.name,
            savedObjectType
          });
          return (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(packagePolicySO);
        });
      }
    });
  }
  async getPackagePolicySavedObjects(soClient, options = {}) {
    const {
      perPage = _constants2.SO_SEARCH_LIMIT,
      spaceIds
    } = options;
    const savedObjectType = await getPackagePolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    const namespaces = isSpacesEnabled ? spaceIds : undefined;
    const packagePolicies = await soClient.find({
      ...(options || {}),
      type: savedObjectType,
      perPage,
      namespaces
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage('failed to find package policies'));
    for (const packagePolicy of (_packagePolicies$save2 = packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.saved_objects) !== null && _packagePolicies$save2 !== void 0 ? _packagePolicies$save2 : []) {
      var _packagePolicies$save2;
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        name: packagePolicy.attributes.name,
        savedObjectType
      });
    }
    return packagePolicies;
  }
  async rollback(soClient, packagePolicies) {
    const savedObjectType = await getPackagePolicySavedObjectType();

    // Need to break down policies by namespace for bulk operations.
    // Create temporary SOs with id `id:copy` to allow cancellation in case rollback fails.
    const policiesToCreate = {};
    const policiesToUpdate = {};
    const previousVersionPolicies = {};
    packagePolicies.forEach(policy => {
      var _policy$namespaces;
      const namespace = ((_policy$namespaces = policy.namespaces) === null || _policy$namespaces === void 0 ? void 0 : _policy$namespaces[0]) || policy.attributes.namespace;
      if (namespace) {
        if (!policy.id.endsWith(':prev')) {
          const previousRevision = packagePolicies.find(p => p.id === `${policy.id}:prev`);
          if (previousRevision !== null && previousRevision !== void 0 && previousRevision.attributes) {
            var _policy$attributes$re;
            if (!policiesToCreate[namespace]) {
              policiesToCreate[namespace] = [];
            }
            policiesToCreate[namespace].push({
              ...policy,
              id: `${policy.id}:copy`
            });
            if (!policiesToUpdate[namespace]) {
              policiesToUpdate[namespace] = [];
            }
            policiesToUpdate[namespace].push({
              ...policy,
              attributes: {
                ...(previousRevision === null || previousRevision === void 0 ? void 0 : previousRevision.attributes),
                revision: ((_policy$attributes$re = policy === null || policy === void 0 ? void 0 : policy.attributes.revision) !== null && _policy$attributes$re !== void 0 ? _policy$attributes$re : 0) + 1,
                // Bump revision
                latest_revision: true
              }
            });
          }
        } else {
          if (!previousVersionPolicies[namespace]) {
            previousVersionPolicies[namespace] = [];
          }
          previousVersionPolicies[namespace].push(policy);
        }
      }
    });

    // Create temporary saved objects.
    for (const [namespace, policies] of Object.entries(policiesToCreate)) {
      await soClient.bulkCreate(policies, {
        namespace
      }).catch(_utils.catchAndSetErrorStackTrace.withMessage('failed to bulk create package policies'));
      for (const policy of policies) {
        _audit_logging.auditLoggingService.writeCustomSoAuditLog({
          action: 'create',
          id: policy.id,
          savedObjectType
        });
      }
    }

    // Update policies with previous revision data.
    for (const [namespace, policies] of Object.entries(policiesToUpdate)) {
      await soClient.bulkUpdate(policies, {
        namespace
      }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Saved objects bulk update failed`));
      for (const policy of policies) {
        _audit_logging.auditLoggingService.writeCustomSoAuditLog({
          action: 'update',
          id: policy.id,
          savedObjectType
        });
      }
    }
    return {
      updatedPolicies: policiesToUpdate,
      copiedPolicies: policiesToCreate,
      previousVersionPolicies
    };
  }
  async restoreRollback(soClient, rollbackResult) {
    const savedObjectType = await getPackagePolicySavedObjectType();
    const {
      copiedPolicies
    } = rollbackResult;
    for (const [namespace, policies] of Object.entries(copiedPolicies)) {
      // Update policies with copied data.
      const policiesToUpdate = policies.map(policy => ({
        ...policy,
        id: policy.id.replace(':copy', '')
      }));
      await soClient.bulkUpdate(policiesToUpdate, {
        namespace
      }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Saved objects bulk update failed`));
      for (const policy of policies) {
        _audit_logging.auditLoggingService.writeCustomSoAuditLog({
          action: 'update',
          id: policy.id.replace(':copy', ''),
          savedObjectType
        });
      }

      // Delete temporary saved objects.
      await this.deleteRollbackSavedObjects(soClient, namespace, policies);
    }
  }
  async cleanupRollbackSavedObjects(soClient, rollbackResult) {
    const {
      copiedPolicies,
      previousVersionPolicies
    } = rollbackResult;
    // Delete temporary saved objects.
    for (const [namespace, policies] of Object.entries(copiedPolicies)) {
      await this.deleteRollbackSavedObjects(soClient, namespace, policies);
    }
    // Delete previous revisions saved objects.
    for (const [namespace, policies] of Object.entries(previousVersionPolicies)) {
      await this.deleteRollbackSavedObjects(soClient, namespace, policies);
    }
  }
  async deleteRollbackSavedObjects(soClient, namespace, policies) {
    const savedObjectType = await getPackagePolicySavedObjectType();
    const objects = policies.map(policy => ({
      id: policy.id,
      type: savedObjectType
    }));
    const {
      statuses
    } = await soClient.bulkDelete(objects, {
      force: true,
      namespace
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Bulk delete of package policies [${policies.map(p => p.id).join(', ')}] failed`));
    for (const policy of policies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'delete',
        id: policy.id,
        savedObjectType
      });
    }
    if (statuses.some(status => !status.success)) {
      throw new _errors.PackageRollbackError(`Failed to delete some previous package policy revisions: ${statuses.filter(status => !status.success).map(status => {
        var _status$error;
        return ((_status$error = status.error) === null || _status$error === void 0 ? void 0 : _status$error.message) || 'Unknown error';
      }).join(', ')}`);
    }
  }
  async bumpAgentPolicyRevisionAfterRollback(soClient, rollbackResult) {
    const {
      updatedPolicies
    } = rollbackResult;
    for (const [namespace, policies] of Object.entries(updatedPolicies)) {
      const agentPolicyIds = policies.flatMap(policy => policy.attributes.policy_ids || []);
      await _agent_policy.agentPolicyService.bumpAgentPoliciesByIds(agentPolicyIds, {}, namespace);
    }
  }
  async createCloudConnectorForPackagePolicy(soClient, enrichedPackagePolicy, agentPolicy) {
    var _agentPolicy$agentles, _agentPolicy$agentles2, _agentPolicy$agentles3, _agentPolicy$agentles4;
    const logger = this.getLogger('createCloudConnectorForPackagePolicy');
    const cloudProvider = (_agentPolicy$agentles = agentPolicy.agentless) === null || _agentPolicy$agentles === void 0 ? void 0 : (_agentPolicy$agentles2 = _agentPolicy$agentles.cloud_connectors) === null || _agentPolicy$agentles2 === void 0 ? void 0 : _agentPolicy$agentles2.target_csp;
    const agentlessCloudConnectorsEnabled = (_agentPolicy$agentles3 = agentPolicy.agentless) === null || _agentPolicy$agentles3 === void 0 ? void 0 : (_agentPolicy$agentles4 = _agentPolicy$agentles3.cloud_connectors) === null || _agentPolicy$agentles4 === void 0 ? void 0 : _agentPolicy$agentles4.enabled;
    if (!agentlessCloudConnectorsEnabled) {
      logger.debug('No agentless cloud connectors enabled for cloud provider');
      return;
    }
    const cloudConnectorVars = extractPackagePolicyVars(cloudProvider, enrichedPackagePolicy, logger);
    if (cloudConnectorVars && enrichedPackagePolicy !== null && enrichedPackagePolicy !== void 0 && enrichedPackagePolicy.supports_cloud_connector) {
      if (enrichedPackagePolicy !== null && enrichedPackagePolicy !== void 0 && enrichedPackagePolicy.cloud_connector_id) {
        const existingCloudConnector = await soClient.get(_constants3.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, enrichedPackagePolicy.cloud_connector_id);
        logger.info(`Updating cloud connector: ${enrichedPackagePolicy.cloud_connector_id}`);
        try {
          const cloudConnector = await _.cloudConnectorService.update(soClient, enrichedPackagePolicy.cloud_connector_id, {
            vars: cloudConnectorVars,
            packagePolicyCount: existingCloudConnector.attributes.packagePolicyCount + 1
          });
          logger.info(`Successfully updated cloud connector: ${cloudConnector.id}`);
          return cloudConnector;
        } catch (e) {
          logger.error(`Error updating cloud connector: ${e}`);
          throw new _errors.CloudConnectorUpdateError(`${e}`);
        }
      } else {
        logger.info(`Creating cloud connector: ${enrichedPackagePolicy.cloud_connector_id}`);
        try {
          const cloudConnector = await _.cloudConnectorService.create(soClient, {
            name: `${cloudProvider}-cloud-connector: ${enrichedPackagePolicy.name}`,
            vars: cloudConnectorVars,
            cloudProvider
          });
          logger.info(`Successfully created cloud connector: ${cloudConnector.id}`);
          return cloudConnector;
        } catch (error) {
          logger.error(`Error creating cloud connector: ${error}`);
          throw new _errors.CloudConnectorCreateError(`${error}`);
        }
      }
    }
  }
}
class PackagePolicyServiceImpl extends PackagePolicyClientImpl {
  asScoped(request) {
    const preflightCheck = async ({
      fleetAuthz: fleetRequiredAuthz
    }) => {
      const authz = await (0, _security.getAuthzFromRequest)(request);
      if ((0, _security.doesNotHaveRequiredFleetAuthz)(authz, fleetRequiredAuthz)) {
        throw new _errors.FleetUnauthorizedError('Not authorized to this action on integration policies');
      }
      if ((await (0, _helpers.isSpaceAwarenessMigrationPending)()) === true) {
        throw new _errors.FleetError('Migration to space awareness is pending');
      }
    };
    return new PackagePolicyClientWithAuthz(preflightCheck);
  }
  get asInternalUser() {
    const preflightCheck = async () => {
      if ((await (0, _helpers.isSpaceAwarenessMigrationPending)()) === true) {
        throw new _errors.FleetError('Migration to space awareness is pending');
      }
    };
    return new PackagePolicyClientWithAuthz(preflightCheck);
  }
}
exports.PackagePolicyServiceImpl = PackagePolicyServiceImpl;
var _runPreflight = /*#__PURE__*/new WeakMap();
class PackagePolicyClientWithAuthz extends PackagePolicyClientImpl {
  constructor(preflightCheck) {
    super();
    _classPrivateFieldInitSpec(this, _runPreflight, async fleetAuthzConfig => {
      if (this.preflightCheck) {
        return await this.preflightCheck(fleetAuthzConfig);
      }
    });
    this.preflightCheck = preflightCheck;
  }
  async bulkCreate(soClient, esClient, packagePolicies, options) {
    await _classPrivateFieldGet(_runPreflight, this).call(this, {
      fleetAuthz: {
        integrations: {
          writeIntegrationPolicies: true
        }
      }
    });
    return super.bulkCreate(soClient, esClient, packagePolicies, options);
  }
  async update(soClient, esClient, id, packagePolicyUpdate, options) {
    await _classPrivateFieldGet(_runPreflight, this).call(this, {
      fleetAuthz: {
        integrations: {
          writeIntegrationPolicies: true
        }
      }
    });
    return super.update(soClient, esClient, id, packagePolicyUpdate, options);
  }
  async create(soClient, esClient, packagePolicy, options, context, request) {
    await _classPrivateFieldGet(_runPreflight, this).call(this, {
      fleetAuthz: {
        integrations: {
          writeIntegrationPolicies: true
        }
      }
    });

    // Add debug logging
    const logger = _.appContextService.getLogger();
    logger.debug(`PackagePolicyService.create called with supports_agentless: ${packagePolicy.supports_agentless}`);
    return super.create(soClient, esClient, packagePolicy, options, context, request);
  }
}
function validatePackagePolicyOrThrow(packagePolicy, pkgInfo) {
  const validationResults = (0, _services.validatePackagePolicy)(packagePolicy, pkgInfo, _jsYaml.load);
  if ((0, _services.validationHasErrors)(validationResults)) {
    const responseFormattedValidationErrors = Object.entries((0, _std.getFlattenedObject)(validationResults)).map(([key, value]) => {
      try {
        const message = !!value ? JSON.stringify(value) : value;
        return {
          key,
          message
        };
      } catch (e) {
        return {
          key,
          message: value
        };
      }
    }).filter(({
      message
    }) => !!message);
    if (responseFormattedValidationErrors.length) {
      throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.packagePolicyInvalidError', {
        defaultMessage: 'Package policy is invalid: {errors}',
        values: {
          errors: responseFormattedValidationErrors.map(({
            key,
            message
          }) => `${key}: ${message}`).join('\n')
        }
      }));
    }
  }
}
function _compilePackagePolicyInputs(pkgInfo, vars, inputs, assetsMap) {
  return inputs.map(input => {
    const compiledInput = _compilePackagePolicyInput(pkgInfo, vars, input, assetsMap);
    const compiledStreams = _compilePackageStreams(pkgInfo, vars, input, assetsMap);
    return {
      ...input,
      compiled_input: compiledInput,
      streams: compiledStreams
    };
  });
}
function _compilePackagePolicyInput(pkgInfo, vars, input, assetsMap) {
  var _pkgInfo$policy_templ, _pkgInfo$policy_templ2;
  const packagePolicyTemplate = input.policy_template ? (_pkgInfo$policy_templ = pkgInfo.policy_templates) === null || _pkgInfo$policy_templ === void 0 ? void 0 : _pkgInfo$policy_templ.find(policyTemplate => policyTemplate.name === input.policy_template) : (_pkgInfo$policy_templ2 = pkgInfo.policy_templates) === null || _pkgInfo$policy_templ2 === void 0 ? void 0 : _pkgInfo$policy_templ2[0];
  if (!input.enabled || !packagePolicyTemplate) {
    return undefined;
  }
  const packageInputs = (0, _services.getNormalizedInputs)(packagePolicyTemplate);
  if (!packageInputs.length) {
    return undefined;
  }
  const packageInput = packageInputs.find(pkgInput => pkgInput.type === input.type);
  if (!packageInput) {
    throw new _errors.InputNotFoundError(`Input template not found, unable to find input type ${input.type}`);
  }
  if (!packageInput.template_path) {
    return undefined;
  }
  const [pkgInputTemplate] = (0, _assets.getAssetsDataFromAssetsMap)(pkgInfo, assetsMap, path => path.endsWith(`/agent/input/${packageInput.template_path}`));
  if (!pkgInputTemplate || !pkgInputTemplate.buffer) {
    throw new _errors.InputNotFoundError(`Unable to load input template at /agent/input/${packageInput.template_path}`);
  }
  return (0, _agent.compileTemplate)(
  // Populate template variables from package- and input-level vars
  Object.assign({}, vars, input.vars), pkgInputTemplate.buffer.toString());
}
function _compilePackageStreams(pkgInfo, vars, input, assetsMap) {
  return input.streams.map(stream => _compilePackageStream(pkgInfo, vars, input, stream, assetsMap));
}

// temporary export to enable testing pending refactor https://github.com/elastic/kibana/issues/112386
// TODO: Move this logic into `package_policies_to_agent_permissions.ts` since this is not a package policy concern
// and is based entirely on the package contents
function _applyIndexPrivileges(packageDataStream, stream) {
  var _packageDataStream$el, _packageDataStream$el2, _packageDataStream$el3, _packageDataStream$el4, _appContextService$ge2;
  const streamOut = (0, _lodash.cloneDeep)(stream);
  if (packageDataStream !== null && packageDataStream !== void 0 && (_packageDataStream$el = packageDataStream.elasticsearch) !== null && _packageDataStream$el !== void 0 && _packageDataStream$el.dynamic_dataset) {
    var _streamOut$data_strea;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea !== void 0 ? _streamOut$data_strea : {};
    streamOut.data_stream.elasticsearch.dynamic_dataset = packageDataStream.elasticsearch.dynamic_dataset;
  }
  if (packageDataStream !== null && packageDataStream !== void 0 && (_packageDataStream$el2 = packageDataStream.elasticsearch) !== null && _packageDataStream$el2 !== void 0 && _packageDataStream$el2.dynamic_namespace) {
    var _streamOut$data_strea2;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea2 = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea2 !== void 0 ? _streamOut$data_strea2 : {};
    streamOut.data_stream.elasticsearch.dynamic_namespace = packageDataStream.elasticsearch.dynamic_namespace;
  }
  const indexPrivileges = packageDataStream === null || packageDataStream === void 0 ? void 0 : (_packageDataStream$el3 = packageDataStream.elasticsearch) === null || _packageDataStream$el3 === void 0 ? void 0 : (_packageDataStream$el4 = _packageDataStream$el3.privileges) === null || _packageDataStream$el4 === void 0 ? void 0 : _packageDataStream$el4.indices;
  if (!(indexPrivileges !== null && indexPrivileges !== void 0 && indexPrivileges.length)) {
    return streamOut;
  }
  const isServerless = (_appContextService$ge2 = _.appContextService.getCloud()) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.isServerlessEnabled;
  const DATA_STREAM_ALLOWED_INDEX_PRIVILEGES = new Set(['auto_configure', 'create_doc', 'maintenance', 'monitor', 'read', ...(isServerless ? [] : ['read_cross_cluster'])]);
  const [valid, invalid] = (0, _lodash.partition)(indexPrivileges, permission => DATA_STREAM_ALLOWED_INDEX_PRIVILEGES.has(permission));
  if (invalid.length) {
    _.appContextService.getLogger().warn(`Ignoring invalid or forbidden index privilege(s) in "${stream.id}" data stream: ${invalid}`);
  }
  if (valid.length) {
    var _streamOut$data_strea3;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea3 = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea3 !== void 0 ? _streamOut$data_strea3 : {};
    streamOut.data_stream.elasticsearch.privileges = {
      indices: valid
    };
  }
  return streamOut;
}
function _compilePackageStream(pkgInfo, vars, input, streamIn, assetsMap) {
  let stream = streamIn;
  if (!stream.enabled) {
    return {
      ...stream,
      compiled_stream: undefined
    };
  }
  const packageDataStreams = (0, _services.getNormalizedDataStreams)(pkgInfo);
  if (!packageDataStreams) {
    throw new _errors.StreamNotFoundError('Stream template not found, no data streams');
  }
  const packageDataStream = packageDataStreams.find(pkgDataStream => pkgDataStream.dataset === stream.data_stream.dataset);
  if (!packageDataStream) {
    throw new _errors.StreamNotFoundError(`Stream template not found, unable to find dataset ${stream.data_stream.dataset}`);
  }
  stream = _applyIndexPrivileges(packageDataStream, streamIn);
  const streamFromPkg = (packageDataStream.streams || []).find(pkgStream => pkgStream.input === input.type);
  if (!streamFromPkg) {
    throw new _errors.StreamNotFoundError(`Stream template not found, unable to find stream for input ${input.type}`);
  }
  if (!streamFromPkg.template_path) {
    throw new _errors.StreamNotFoundError(`Stream template path not found for dataset ${stream.data_stream.dataset}`);
  }
  const datasetPath = packageDataStream.path;
  const [pkgStreamTemplate] = (0, _assets.getAssetsDataFromAssetsMap)(pkgInfo, assetsMap, path => path.endsWith(streamFromPkg.template_path), datasetPath);
  if (!pkgStreamTemplate || !pkgStreamTemplate.buffer) {
    throw new _errors.StreamNotFoundError(`Unable to load stream template ${streamFromPkg.template_path} for dataset ${stream.data_stream.dataset}`);
  }
  const yaml = (0, _agent.compileTemplate)(
  // Populate template variables from package-, input-, and stream-level vars
  Object.assign({}, vars, input.vars, stream.vars), pkgStreamTemplate.buffer.toString());
  stream.compiled_stream = yaml;
  return {
    ...stream
  };
}
function enforceFrozenInputs(oldInputs, newInputs, force = false) {
  const resultInputs = [...newInputs];
  for (const input of resultInputs) {
    const oldInput = oldInputs.find(i => i.type === input.type);
    if (oldInput !== null && oldInput !== void 0 && oldInput.keep_enabled) input.enabled = oldInput.enabled;
    if (input.vars && oldInput !== null && oldInput !== void 0 && oldInput.vars) {
      input.vars = _enforceFrozenVars(oldInput.vars, input.vars, force);
    }
    if (input.streams && oldInput !== null && oldInput !== void 0 && oldInput.streams) {
      for (const stream of input.streams) {
        const oldStream = oldInput.streams.find(s => s.id === stream.id);
        if (oldStream !== null && oldStream !== void 0 && oldStream.keep_enabled) stream.enabled = oldStream.enabled;
        if (stream.vars && oldStream !== null && oldStream !== void 0 && oldStream.vars) {
          stream.vars = _enforceFrozenVars(oldStream.vars, stream.vars, force);
        }
      }
    }
  }
  return resultInputs;
}
function _enforceFrozenVars(oldVars, newVars, force = false) {
  const resultVars = {};
  for (const [key, val] of Object.entries(newVars)) {
    var _oldVars$key;
    if ((_oldVars$key = oldVars[key]) !== null && _oldVars$key !== void 0 && _oldVars$key.frozen) {
      if (force) {
        resultVars[key] = val;
      } else if (!(0, _lodash.isEqual)(oldVars[key].value, val.value) || oldVars[key].type !== val.type) {
        throw new _errors.PackagePolicyValidationError(`${key} is a frozen variable and cannot be modified`);
      } else {
        resultVars[key] = oldVars[key];
      }
    } else {
      resultVars[key] = val;
    }
  }
  for (const [key, val] of Object.entries(oldVars)) {
    if (!newVars[key] && val.frozen) {
      resultVars[key] = val;
    }
  }
  return resultVars;
}
const packagePolicyService = exports.packagePolicyService = new PackagePolicyClientImpl();
async function getPackageInfoForPackagePolicies(packagePolicies, soClient, ignoreMissing) {
  const pkgInfoMap = new Map();
  packagePolicies.forEach(({
    package: pkg
  }) => {
    if (pkg) {
      pkgInfoMap.set(`${pkg.name}-${pkg.version}`, pkg);
    }
  });
  const resultMap = new Map();
  await (0, _pMap.default)(pkgInfoMap.keys(), async pkgKey => {
    const pkgInfo = pkgInfoMap.get(pkgKey);
    if (pkgInfo) {
      try {
        const pkgInfoData = await (0, _packages.getPackageInfo)({
          savedObjectsClient: soClient,
          pkgName: pkgInfo.name,
          pkgVersion: pkgInfo.version,
          prerelease: true
        });
        resultMap.set(pkgKey, pkgInfoData);
      } catch (error) {
        if (!ignoreMissing) {
          throw error;
        }
      }
    }
  });
  return resultMap;
}
function updatePackageInputs(basePackagePolicy, packageInfo, inputsUpdated, dryRun) {
  var _packageInfo$policy_t;
  if (!inputsUpdated) return basePackagePolicy;
  const availablePolicyTemplates = (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 ? _packageInfo$policy_t : [];
  const limitedPackage = (0, _services.isPackageLimited)(packageInfo);
  const inputs = [...basePackagePolicy.inputs.filter(input => {
    var _policyTemplate$input, _policyTemplate$input2;
    if (!input.policy_template) {
      return true;
    }
    const policyTemplate = availablePolicyTemplates.find(({
      name
    }) => name === input.policy_template);

    // Ignore any policy templates removed in the new package version
    if (!policyTemplate) {
      return false;
    }

    // Ignore any inputs removed from this policy template in the new package version
    const policyTemplateStillIncludesInput = (0, _services.isInputOnlyPolicyTemplate)(policyTemplate) ? policyTemplate.input === input.type : (_policyTemplate$input = (_policyTemplate$input2 = policyTemplate.inputs) === null || _policyTemplate$input2 === void 0 ? void 0 : _policyTemplate$input2.some(policyTemplateInput => policyTemplateInput.type === input.type)) !== null && _policyTemplate$input !== void 0 ? _policyTemplate$input : false;
    return policyTemplateStillIncludesInput;
  })];
  for (const update of inputsUpdated) {
    let originalInput;
    if (update.policy_template) {
      // If the updated value defines a policy template, try to find an original input
      // with the same policy template value
      const matchingInput = inputs.find(i => i.type === update.type && i.policy_template === update.policy_template);

      // If we didn't find an input with the same policy template, try to look for one
      // with the same type, but with an undefined policy template. This ensures we catch
      // cases where we're upgrading an older policy from before policy template was
      // reliably define on package policy inputs.
      originalInput = matchingInput || inputs.find(i => i.type === update.type && !i.policy_template);
    } else {
      // For inputs that don't specify a policy template, just grab the first input
      // that matches its `type`
      originalInput = inputs.find(i => i.type === update.type);
    }

    // If there's no corresponding input on the original package policy, just
    // take the override value from the new package as-is. This case typically
    // occurs when inputs or package policy templates are added/removed between versions.
    if (originalInput === undefined) {
      // Do not enable new inputs for limited packages
      if (limitedPackage) {
        update.enabled = false;
      }
      inputs.push(update);
      continue;
    }

    // For flags like this, we only want to override the original value if it was set
    // as `undefined` in the original object. An explicit true/false value should be
    // persisted from the original object to the result after the override process is complete.
    if (!limitedPackage && originalInput.enabled === undefined && update.enabled !== undefined) {
      originalInput.enabled = update.enabled;
    }
    if (originalInput.keep_enabled === undefined && update.keep_enabled !== undefined) {
      originalInput.keep_enabled = update.keep_enabled;
    }

    // `policy_template` should always be defined, so if we have an older policy here we need
    // to ensure we set it
    if (originalInput.policy_template === undefined && update.policy_template !== undefined) {
      originalInput.policy_template = update.policy_template;
    }
    if (update.vars || originalInput.vars) {
      const indexOfInput = inputs.indexOf(originalInput);
      const mergedVars = deepMergeVars(originalInput, update, true);
      const filteredVars = removeStaleVars(mergedVars, update);
      inputs[indexOfInput] = filteredVars;
      originalInput = inputs[indexOfInput];
    }
    if (update.streams) {
      var _originalInput;
      const isInputPkgUpdate = packageInfo.type === 'input' && update.streams.length === 1 && ((_originalInput = originalInput) === null || _originalInput === void 0 ? void 0 : _originalInput.streams.length) === 1;
      for (const stream of update.streams) {
        var _originalInput2, _originalStream;
        let originalStream = (_originalInput2 = originalInput) === null || _originalInput2 === void 0 ? void 0 : _originalInput2.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);

        // this handles the input only pkg case where the new stream cannot have a dataset name
        // so will never match. Input only packages only ever have one stream.
        if (!originalStream && isInputPkgUpdate) {
          var _originalInput3, _originalInput3$strea, _originalInput4;
          originalStream = {
            ...update.streams[0],
            id: (_originalInput3 = originalInput) === null || _originalInput3 === void 0 ? void 0 : (_originalInput3$strea = _originalInput3.streams[0]) === null || _originalInput3$strea === void 0 ? void 0 : _originalInput3$strea.id,
            vars: (_originalInput4 = originalInput) === null || _originalInput4 === void 0 ? void 0 : _originalInput4.streams[0].vars
          };
        }
        if (originalStream === undefined) {
          originalInput.streams.push(stream);
          continue;
        }
        if (((_originalStream = originalStream) === null || _originalStream === void 0 ? void 0 : _originalStream.enabled) === undefined) {
          originalStream.enabled = stream.enabled;
        }
        if (stream.vars || originalStream.vars) {
          // streams wont match for input pkgs
          const indexOfStream = isInputPkgUpdate ? 0 : originalInput.streams.indexOf(originalStream);
          const mergedVars = deepMergeVars(originalStream, stream, true);
          const filteredVars = removeStaleVars(mergedVars, stream);
          originalInput.streams[indexOfStream] = filteredVars;
          originalStream = originalInput.streams[indexOfStream];
        }
      }
    }

    // Filter all stream that have been removed from the input
    originalInput.streams = originalInput.streams.filter(originalStream => {
      var _update$streams$some, _update$streams;
      return (_update$streams$some = (_update$streams = update.streams) === null || _update$streams === void 0 ? void 0 : _update$streams.some(s => s.data_stream.dataset === originalStream.data_stream.dataset)) !== null && _update$streams$some !== void 0 ? _update$streams$some : false;
    });
  }
  const vars = getUpdatedGlobalVars(packageInfo, basePackagePolicy);
  const resultingPackagePolicy = {
    ...basePackagePolicy,
    inputs,
    vars
  };
  const validationResults = (0, _services.validatePackagePolicy)(resultingPackagePolicy, packageInfo, _jsYaml.load);
  if ((0, _services.validationHasErrors)(validationResults)) {
    const responseFormattedValidationErrors = Object.entries((0, _std.getFlattenedObject)(validationResults)).map(([key, value]) => {
      try {
        const message = !!value ? JSON.stringify(value) : value;
        return {
          key,
          message
        };
      } catch (e) {
        return {
          key,
          message: value
        };
      }
    }).filter(({
      message
    }) => !!message);
    if (responseFormattedValidationErrors.length) {
      if (dryRun) {
        return {
          ...resultingPackagePolicy,
          errors: responseFormattedValidationErrors
        };
      }
      throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.packagePolicyInvalidError', {
        defaultMessage: 'Package policy is invalid: {errors}',
        values: {
          errors: responseFormattedValidationErrors.map(({
            key,
            message
          }) => `${key}: ${message}`).join('\n')
        }
      }));
    }
  }
  return resultingPackagePolicy;
}
function preconfigurePackageInputs(basePackagePolicy, packageInfo, preconfiguredInputs) {
  if (!preconfiguredInputs) return basePackagePolicy;
  const inputs = [...basePackagePolicy.inputs];
  for (const preconfiguredInput of preconfiguredInputs) {
    // Preconfiguration does not currently support multiple policy templates, so overrides will have an undefined
    // policy template, so we only match on `type` in that case.
    let originalInput = preconfiguredInput.policy_template ? inputs.find(i => i.type === preconfiguredInput.type && i.policy_template === preconfiguredInput.policy_template) : inputs.find(i => i.type === preconfiguredInput.type);

    // If the input do not exist skip
    if (originalInput === undefined) {
      continue;
    }
    if (preconfiguredInput.enabled !== undefined) {
      originalInput.enabled = preconfiguredInput.enabled;
    }
    if (preconfiguredInput.keep_enabled !== undefined) {
      originalInput.keep_enabled = preconfiguredInput.keep_enabled;
    }
    if (preconfiguredInput.vars) {
      const indexOfInput = inputs.indexOf(originalInput);
      inputs[indexOfInput] = deepMergeVars(originalInput, preconfiguredInput);
      originalInput = inputs[indexOfInput];
    }
    if (preconfiguredInput.streams) {
      for (const stream of preconfiguredInput.streams) {
        var _originalInput5;
        let originalStream = (_originalInput5 = originalInput) === null || _originalInput5 === void 0 ? void 0 : _originalInput5.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);
        if (originalStream === undefined) {
          continue;
        }
        if (stream.enabled !== undefined) {
          originalStream.enabled = stream.enabled;
        }
        if (stream.vars) {
          const indexOfStream = originalInput.streams.indexOf(originalStream);
          originalInput.streams[indexOfStream] = deepMergeVars(originalStream, stream);
          originalStream = originalInput.streams[indexOfStream];
        }
      }
    }
  }
  const resultingPackagePolicy = {
    ...basePackagePolicy,
    inputs
  };
  validatePackagePolicyOrThrow(resultingPackagePolicy, packageInfo);
  return resultingPackagePolicy;
}

// input only packages cannot have their namespace or dataset modified
function _validateRestrictedFieldsNotModifiedOrThrow(opts) {
  const {
    pkgInfo,
    oldPackagePolicy,
    packagePolicyUpdate
  } = opts;
  if (pkgInfo.type !== 'input') return;
  const {
    namespace,
    inputs
  } = packagePolicyUpdate;
  if (namespace && namespace !== oldPackagePolicy.namespace) {
    throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.updatePackagePolicy.namespaceCannotBeModified', {
      defaultMessage: 'Package policy namespace cannot be modified for input only packages, please create a new package policy.'
    }));
  }
  if (inputs) {
    for (const input of inputs) {
      const oldInput = oldPackagePolicy.inputs.find(i => i.type === input.type);
      if (oldInput) {
        for (const stream of input.streams || []) {
          var _oldStream$vars, _oldStream$vars$DATAS, _stream$vars, _stream$vars$DATASET_, _oldStream$vars2, _oldStream$vars$DATA_, _stream$vars3, _stream$vars3$DATA_ST;
          const oldStream = oldInput.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);
          if (oldStream && oldStream !== null && oldStream !== void 0 && (_oldStream$vars = oldStream.vars) !== null && _oldStream$vars !== void 0 && _oldStream$vars[_constants2.DATASET_VAR_NAME] && (oldStream === null || oldStream === void 0 ? void 0 : (_oldStream$vars$DATAS = oldStream.vars[_constants2.DATASET_VAR_NAME]) === null || _oldStream$vars$DATAS === void 0 ? void 0 : _oldStream$vars$DATAS.value) !== (stream === null || stream === void 0 ? void 0 : (_stream$vars = stream.vars) === null || _stream$vars === void 0 ? void 0 : (_stream$vars$DATASET_ = _stream$vars[_constants2.DATASET_VAR_NAME]) === null || _stream$vars$DATASET_ === void 0 ? void 0 : _stream$vars$DATASET_.value)) {
            // seeing this error in dev? Package policy must be called with prepareInputPackagePolicyDataset function first in UI code
            _.appContextService.getLogger().debug(() => {
              var _stream$vars2, _stream$vars2$DATASET;
              return `Rejecting package policy update due to dataset change, old val '${oldStream.vars[_constants2.DATASET_VAR_NAME].value}, new val '${JSON.stringify(stream === null || stream === void 0 ? void 0 : (_stream$vars2 = stream.vars) === null || _stream$vars2 === void 0 ? void 0 : (_stream$vars2$DATASET = _stream$vars2[_constants2.DATASET_VAR_NAME]) === null || _stream$vars2$DATASET === void 0 ? void 0 : _stream$vars2$DATASET.value)}'`;
            });
            throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.updatePackagePolicy.datasetCannotBeModified', {
              defaultMessage: 'Package policy dataset cannot be modified for input only packages, please create a new package policy.'
            }));
          }
          if (oldStream && oldStream !== null && oldStream !== void 0 && (_oldStream$vars2 = oldStream.vars) !== null && _oldStream$vars2 !== void 0 && _oldStream$vars2[_constants2.DATA_STREAM_TYPE_VAR_NAME] && (oldStream === null || oldStream === void 0 ? void 0 : (_oldStream$vars$DATA_ = oldStream.vars[_constants2.DATA_STREAM_TYPE_VAR_NAME]) === null || _oldStream$vars$DATA_ === void 0 ? void 0 : _oldStream$vars$DATA_.value) !== (stream === null || stream === void 0 ? void 0 : (_stream$vars3 = stream.vars) === null || _stream$vars3 === void 0 ? void 0 : (_stream$vars3$DATA_ST = _stream$vars3[_constants2.DATA_STREAM_TYPE_VAR_NAME]) === null || _stream$vars3$DATA_ST === void 0 ? void 0 : _stream$vars3$DATA_ST.value)) {
            // seeing this error in dev? Package policy must be called with prepareInputPackagePolicyDataset function first in UI code
            _.appContextService.getLogger().debug(() => {
              var _stream$vars4, _stream$vars4$DATA_ST;
              return `Rejecting package policy update due to data stream type change, old val '${oldStream.vars[_constants2.DATA_STREAM_TYPE_VAR_NAME].value}, new val '${JSON.stringify(stream === null || stream === void 0 ? void 0 : (_stream$vars4 = stream.vars) === null || _stream$vars4 === void 0 ? void 0 : (_stream$vars4$DATA_ST = _stream$vars4[_constants2.DATA_STREAM_TYPE_VAR_NAME]) === null || _stream$vars4$DATA_ST === void 0 ? void 0 : _stream$vars4$DATA_ST.value)}'`;
            });
            throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.updatePackagePolicy.datasetCannotBeModified', {
              defaultMessage: 'Package policy data stream type cannot be modified for input only packages, please create a new package policy.'
            }));
          }
        }
      }
    }
  }
}
function validateReusableIntegrationsAndSpaceAwareness(packagePolicy, agentPolicies) {
  var _packagePolicy$policy2;
  if (((_packagePolicy$policy2 = packagePolicy.policy_ids.length) !== null && _packagePolicy$policy2 !== void 0 ? _packagePolicy$policy2 : 0) <= 1) {
    return;
  }
  for (const agentPolicy of agentPolicies) {
    var _agentPolicy$space_id, _agentPolicy$space_id2;
    if (((_agentPolicy$space_id = agentPolicy === null || agentPolicy === void 0 ? void 0 : (_agentPolicy$space_id2 = agentPolicy.space_ids) === null || _agentPolicy$space_id2 === void 0 ? void 0 : _agentPolicy$space_id2.length) !== null && _agentPolicy$space_id !== void 0 ? _agentPolicy$space_id : 0) > 1) {
      throw new _errors.FleetError('Reusable integration policies cannot be used with agent policies belonging to multiple spaces.');
    }
  }
}
function validateIsNotHostedPolicy(agentPolicy, force = false, errorMessage) {
  const isManagedPolicyWithoutServerlessSupport = agentPolicy.is_managed && !force;
  if (isManagedPolicyWithoutServerlessSupport) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(errorMessage !== null && errorMessage !== void 0 ? errorMessage : `Cannot update integrations of hosted agent policy ${agentPolicy.id}`);
  }
}
function sendUpdatePackagePolicyTelemetryEvent(soClient, updatedPkgPolicies, oldPackagePolicies) {
  updatedPkgPolicies.forEach(updatedPkgPolicy => {
    if (updatedPkgPolicy.package) {
      var _oldPkgPolicy$package;
      const {
        name,
        version
      } = updatedPkgPolicy.package;
      const oldPkgPolicy = oldPackagePolicies.find(packagePolicy => packagePolicy.id === updatedPkgPolicy.id);
      const oldVersion = oldPkgPolicy === null || oldPkgPolicy === void 0 ? void 0 : (_oldPkgPolicy$package = oldPkgPolicy.package) === null || _oldPkgPolicy$package === void 0 ? void 0 : _oldPkgPolicy$package.version;
      if (oldVersion && oldVersion !== version) {
        const upgradeTelemetry = {
          packageName: name,
          currentVersion: oldVersion,
          newVersion: version,
          status: 'success',
          eventType: 'package-policy-upgrade'
        };
        (0, _upgrade_sender.sendTelemetryEvents)(_.appContextService.getLogger(), _.appContextService.getTelemetryEventsSender(), upgradeTelemetry);
        _.appContextService.getLogger().info(`Package policy upgraded successfully`);
        _.appContextService.getLogger().debug(() => JSON.stringify(upgradeTelemetry));
      }
    }
  });
}
function deepMergeVars(original, override, keepOriginalValue = false) {
  if (!override.vars) {
    return original;
  }
  if (!original.vars) {
    original.vars = {
      ...override.vars
    };
  }
  const result = {
    ...original
  };
  const overrideVars = Array.isArray(override.vars) ? override.vars : Object.entries(override.vars).map(([key, rest]) => ({
    name: key,
    ...rest
  }));
  for (const {
    name,
    ...overrideVal
  } of overrideVars) {
    const originalVar = original.vars[name];
    result.vars[name] = {
      ...originalVar,
      ...overrideVal
    };

    // Ensure that any value from the original object is persisted on the newly merged resulting object,
    // even if we merge other data about the given variable
    if (keepOriginalValue && (originalVar === null || originalVar === void 0 ? void 0 : originalVar.value) !== undefined) {
      result.vars[name].value = originalVar.value;
    }
  }
  return result;
}
function getUpdatedGlobalVars(packageInfo, packagePolicy) {
  if (!packageInfo.vars) {
    return undefined;
  }
  const packageInfoVars = packageInfo.vars.reduce(_services.varsReducer, {});
  const result = deepMergeVars(packagePolicy, {
    vars: packageInfoVars
  }, true);
  return removeStaleVars(result, {
    vars: packageInfoVars
  }).vars;
}
function removeStaleVars(currentWithVars, expectedVars) {
  if (!currentWithVars.vars) {
    return currentWithVars;
  }
  if (!expectedVars.vars) {
    return {
      ...currentWithVars,
      vars: {}
    };
  }
  const filteredVars = Object.entries(currentWithVars.vars).reduce((acc, [key, val]) => {
    if (key in expectedVars.vars) {
      acc[key] = val;
    }
    return acc;
  }, {});
  return {
    ...currentWithVars,
    vars: filteredVars
  };
}
async function requireUniqueName(soClient, packagePolicy, id, spaceIds = []) {
  var _existingPoliciesWith, _existingPoliciesWith2;
  const savedObjectType = await getPackagePolicySavedObjectType();
  const isMultiSpace = spaceIds.length > 1;
  const baseSearchParams = {
    perPage: _constants2.SO_SEARCH_LIMIT,
    kuery: `${savedObjectType}.name:"${packagePolicy.name}"`
  };
  let existingPoliciesWithName;
  if (isMultiSpace) {
    existingPoliciesWithName = await packagePolicyService.list(_.appContextService.getInternalUserSOClientWithoutSpaceExtension(), {
      ...baseSearchParams,
      spaceId: '*'
    });
  } else {
    existingPoliciesWithName = await packagePolicyService.list(soClient, baseSearchParams);
  }
  const policiesToCheck = id ?
  // Check that the name does not exist already but exclude the current package policy
  (((_existingPoliciesWith = existingPoliciesWithName) === null || _existingPoliciesWith === void 0 ? void 0 : _existingPoliciesWith.items) || []).filter(p => {
    var _p$spaceIds;
    return p.id !== id && (isMultiSpace ? ((_p$spaceIds = p.spaceIds) !== null && _p$spaceIds !== void 0 ? _p$spaceIds : []).some(spaceId => spaceIds.includes(spaceId)) : true);
  }) : (_existingPoliciesWith2 = existingPoliciesWithName) === null || _existingPoliciesWith2 === void 0 ? void 0 : _existingPoliciesWith2.items;
  if (policiesToCheck.length > 0) {
    throw new _errors.PackagePolicyNameExistsError(`An integration policy with the name ${packagePolicy.name} already exists. Please rename it or choose a different name.`);
  }
}