"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkGetAgentPoliciesHandler = exports.GetListAgentPolicyOutputsHandler = exports.GetAgentPolicyOutputsHandler = void 0;
exports.checkAgentPoliciesAllPrivilegesForSpaces = checkAgentPoliciesAllPrivilegesForSpaces;
exports.getOneAgentPolicyHandler = exports.getK8sManifest = exports.getFullAgentPolicy = exports.getAutoUpgradeAgentsStatusHandler = exports.getAgentPoliciesHandler = exports.downloadK8sManifest = exports.downloadFullAgentPolicy = exports.deleteAgentPoliciesHandler = exports.createAgentPolicyHandler = exports.createAgentAndPackagePoliciesHandler = exports.copyAgentPolicyHandler = void 0;
exports.populateAssignedAgentsCount = populateAssignedAgentsCount;
exports.updateAgentPolicyHandler = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _jsYaml = require("js-yaml");
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _http_authorization_header = require("../../../common/http_authorization_header");
var _services = require("../../../common/services");
var _services2 = require("../../services");
var _agents = require("../../services/agents");
var _constants2 = require("../../constants");
var _errors = require("../../errors");
var _agent_policy_create = require("../../services/agent_policy_create");
var _agent_policy = require("../../services/spaces/agent_policy");
var _simplified_package_policy_helper = require("../../../common/services/simplified_package_policy_helper");
var _api_privileges = require("../../constants/api_privileges");
var _handlers = require("../package_policy/handlers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function populateAssignedAgentsCount(agentClient, agentPolicies) {
  await (0, _pMap.default)(agentPolicies, agentPolicy => {
    const totalAgents = agentClient.listAgents({
      showInactive: true,
      perPage: 0,
      page: 1,
      kuery: `${_constants2.AGENTS_PREFIX}.policy_id:"${agentPolicy.id}"`
    }).then(({
      total
    }) => agentPolicy.agents = total);
    const unprivilegedAgents = agentClient.listAgents({
      showInactive: true,
      perPage: 0,
      page: 1,
      kuery: `${_constants2.AGENTS_PREFIX}.policy_id:"${agentPolicy.id}" and ${_constants2.UNPRIVILEGED_AGENT_KUERY}`
    }).then(({
      total
    }) => agentPolicy.unprivileged_agents = total);
    const fipsAgents = agentClient.listAgents({
      showInactive: true,
      perPage: 0,
      page: 1,
      kuery: `${_constants2.AGENTS_PREFIX}.policy_id:"${agentPolicy.id}" and ${_constants.FIPS_AGENT_KUERY}`
    }).then(({
      total
    }) => agentPolicy.fips_agents = total);
    return Promise.all([totalAgents, unprivilegedAgents, fipsAgents]);
  }, {
    concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_10
  });
}
function sanitizeItemForReadAgentOnly(item) {
  return {
    id: item.id,
    name: item.name,
    description: item.description,
    revision: item.revision,
    namespace: item.namespace,
    is_managed: item.is_managed,
    is_protected: item.is_protected,
    status: item.status,
    updated_at: item.updated_at,
    updated_by: item.updated_by,
    has_fleet_server: item.has_fleet_server,
    monitoring_enabled: item.monitoring_enabled,
    package_policies: []
  };
}
async function checkAgentPoliciesAllPrivilegesForSpaces(request, context, spaceIds) {
  const security = _services2.appContextService.getSecurity();
  const spaces = await (await context.fleet).getAllSpaces();
  const allSpaceId = spaces.map(s => s.id);
  const res = await security.authz.checkPrivilegesWithRequest(request).atSpaces(allSpaceId, {
    kibana: [security.authz.actions.api.get(`fleet-agent-policies-all`)]
  });
  return allSpaceId.filter(id => {
    var _res$privileges$kiban, _res$privileges$kiban2;
    return (_res$privileges$kiban = (_res$privileges$kiban2 = res.privileges.kibana.find(privilege => privilege.resource === id)) === null || _res$privileges$kiban2 === void 0 ? void 0 : _res$privileges$kiban2.authorized) !== null && _res$privileges$kiban !== void 0 ? _res$privileges$kiban : false;
  });
}
const getAgentPoliciesHandler = async (context, request, response) => {
  var _request$authzResult, _request$authzResult2;
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const authzFleetReadAgentPolicies = ((_request$authzResult = request.authzResult) === null || _request$authzResult === void 0 ? void 0 : _request$authzResult[_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ]) === true;
  const authzFleetAgentRead = ((_request$authzResult2 = request.authzResult) === null || _request$authzResult2 === void 0 ? void 0 : _request$authzResult2[_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]) === true;
  const soClient = authzFleetReadAgentPolicies || authzFleetAgentRead ? coreContext.savedObjects.client : fleetContext.internalSoClient;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const {
    full: withPackagePolicies = false,
    noAgentCount,
    withAgentCount,
    format,
    ...restOfQuery
  } = request.query;
  if (!authzFleetReadAgentPolicies && withPackagePolicies) {
    throw new _errors.FleetUnauthorizedError('full query parameter require agent policies read permissions');
  }
  const agentPoliciesResponse = await _services2.agentPolicyService.list(soClient, {
    withPackagePolicies,
    esClient,
    ...restOfQuery
  });
  let {
    items
  } = agentPoliciesResponse;
  const {
    total,
    page,
    perPage
  } = agentPoliciesResponse;
  if (authzFleetAgentRead && (noAgentCount === false || withAgentCount)) {
    await populateAssignedAgentsCount(fleetContext.agentClient.asCurrentUser, items);
  }
  if (!authzFleetReadAgentPolicies) {
    items = items.map(sanitizeItemForReadAgentOnly);
  } else if (withPackagePolicies && format === _constants.inputsFormat.Simplified) {
    items.map(item => {
      if ((0, _lodash.isEmpty)(item.package_policies)) {
        return item;
      }
      return {
        ...item,
        package_policies: item.package_policies.map(packagePolicy => (0, _simplified_package_policy_helper.packagePolicyToSimplifiedPackagePolicy)(packagePolicy))
      };
    });
  }
  const body = {
    items,
    total,
    page,
    perPage
  };
  return response.ok({
    body
  });
};
exports.getAgentPoliciesHandler = getAgentPoliciesHandler;
const bulkGetAgentPoliciesHandler = async (context, request, response) => {
  try {
    var _request$authzResult3, _request$authzResult4;
    const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
    const authzFleetReadAgentPolicies = ((_request$authzResult3 = request.authzResult) === null || _request$authzResult3 === void 0 ? void 0 : _request$authzResult3[_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ]) === true;
    const authzFleetAgentRead = ((_request$authzResult4 = request.authzResult) === null || _request$authzResult4 === void 0 ? void 0 : _request$authzResult4[_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]) === true;
    const soClient = authzFleetReadAgentPolicies || authzFleetAgentRead ? coreContext.savedObjects.client : fleetContext.internalSoClient;
    const {
      full: withPackagePolicies = false,
      ignoreMissing = false,
      ids
    } = request.body;
    if (!authzFleetReadAgentPolicies && withPackagePolicies) {
      throw new _errors.FleetUnauthorizedError('full query parameter require agent policies read permissions');
    }
    let items = await _services2.agentPolicyService.getByIds(soClient, ids, {
      withPackagePolicies,
      ignoreMissing
    });
    if (!authzFleetReadAgentPolicies) {
      items = items.map(sanitizeItemForReadAgentOnly);
    } else if (withPackagePolicies && request.query.format === _constants.inputsFormat.Simplified) {
      items.map(item => {
        if ((0, _lodash.isEmpty)(item.package_policies)) {
          return item;
        }
        return {
          ...item,
          package_policies: item.package_policies.map(packagePolicy => (0, _simplified_package_policy_helper.packagePolicyToSimplifiedPackagePolicy)(packagePolicy))
        };
      });
    }
    const body = {
      items
    };
    if (authzFleetAgentRead) {
      await populateAssignedAgentsCount(fleetContext.agentClient.asCurrentUser, items);
    }
    return response.ok({
      body
    });
  } catch (error) {
    if (error instanceof _errors.AgentPolicyNotFoundError) {
      return response.notFound({
        body: {
          message: error.message
        }
      });
    }
    throw error;
  }
};
exports.bulkGetAgentPoliciesHandler = bulkGetAgentPoliciesHandler;
const getOneAgentPolicyHandler = async (context, request, response) => {
  const [coreContext, fleetContext] = await Promise.all([context.core, context.fleet]);
  const soClient = coreContext.savedObjects.client;
  const agentPolicy = await _services2.agentPolicyService.get(soClient, request.params.agentPolicyId);
  if (agentPolicy) {
    if (fleetContext.authz.fleet.readAgents) {
      await populateAssignedAgentsCount(fleetContext.agentClient.asCurrentUser, [agentPolicy]);
    }
    let item = agentPolicy;
    if (!fleetContext.authz.fleet.readAgentPolicies) {
      item = sanitizeItemForReadAgentOnly(agentPolicy);
    } else if (request.query.format === _constants.inputsFormat.Simplified && !(0, _lodash.isEmpty)(agentPolicy.package_policies)) {
      item = {
        ...agentPolicy,
        package_policies: agentPolicy.package_policies.map(packagePolicy => (0, _simplified_package_policy_helper.packagePolicyToSimplifiedPackagePolicy)(packagePolicy))
      };
    }
    const body = {
      item
    };
    return response.ok({
      body
    });
  } else {
    return response.customError({
      statusCode: 404,
      body: {
        message: 'Agent policy not found'
      }
    });
  }
};
exports.getOneAgentPolicyHandler = getOneAgentPolicyHandler;
const getAutoUpgradeAgentsStatusHandler = async (context, request, response) => {
  const [_, fleetContext] = await Promise.all([context.core, context.fleet]);
  const agentClient = fleetContext.agentClient.asCurrentUser;
  if (!_services2.licenseService.isEnterprise()) {
    throw new _errors.FleetUnauthorizedError('Auto-upgrade agents feature requires at least Enterprise license');
  }
  const body = await (0, _agents.getAutoUpgradeAgentsStatus)(agentClient, request.params.agentPolicyId);
  return response.ok({
    body
  });
};
exports.getAutoUpgradeAgentsStatusHandler = getAutoUpgradeAgentsStatusHandler;
const createAgentPolicyHandler = async (context, request, response) => {
  var _request$query$sys_mo;
  const coreContext = await context.core;
  const fleetContext = await context.fleet;
  const soClient = fleetContext.internalSoClient;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const user = _services2.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  const withSysMonitoring = (_request$query$sys_mo = request.query.sys_monitoring) !== null && _request$query$sys_mo !== void 0 ? _request$query$sys_mo : false;
  const monitoringEnabled = request.body.monitoring_enabled;
  const logger = _services2.appContextService.getLogger().get('httpCreateAgentPolicyHandler');
  const {
    has_fleet_server: hasFleetServer,
    force,
    ...newPolicy
  } = request.body;
  const spaceId = fleetContext.spaceId;
  const authorizationHeader = _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request, user === null || user === void 0 ? void 0 : user.username);
  const {
    space_ids: spaceIds
  } = request.body;
  logger.debug(`Creating agent policy [${newPolicy.name}]`);
  try {
    let authorizedSpaces;
    if (spaceIds !== null && spaceIds !== void 0 && spaceIds.length) {
      logger.debug(`Checking privileges for spaces [${spaceIds.join(', ')}] `);
      authorizedSpaces = await checkAgentPoliciesAllPrivilegesForSpaces(request, context, spaceIds);
      for (const requestedSpaceId of spaceIds) {
        if (!authorizedSpaces.includes(requestedSpaceId)) {
          throw new _errors.FleetError(`No enough permissions to create policies in space ${requestedSpaceId}`);
        }
      }
    }
    const agentPolicy = await (0, _agent_policy_create.createAgentPolicyWithPackages)({
      soClient,
      esClient,
      agentPolicyService: _services2.agentPolicyService,
      newPolicy,
      hasFleetServer,
      withSysMonitoring,
      monitoringEnabled,
      spaceId,
      user,
      authorizationHeader,
      force
    });
    const body = {
      item: agentPolicy
    };
    // Update spaces if there is more than one space ID assigned to that policy or if there the space that policy is created is different than the current space
    if (spaceIds && authorizedSpaces && (spaceIds.length > 1 || (spaceIds.length === 0 && spaceIds[0]) !== spaceId)) {
      await (0, _agent_policy.updateAgentPolicySpaces)({
        agentPolicy: {
          ...agentPolicy,
          space_ids: spaceIds
        },
        currentSpaceId: spaceId,
        authorizedSpaces,
        options: {
          force
        }
      });
    }
    return response.ok({
      body
    });
  } catch (error) {
    if (error.statusCode) {
      return response.customError({
        statusCode: error.statusCode,
        body: {
          message: error.message
        }
      });
    }
    throw error;
  }
};
exports.createAgentPolicyHandler = createAgentPolicyHandler;
const createAgentAndPackagePoliciesHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const logger = _services2.appContextService.getLogger().get('httpCreateAgentAndPackagePoliciesHandler');
  logger.debug('Creating agent and package policies');

  // Try to create the agent policy
  const {
    package_policies: packagePolicies,
    ...agentPolicyWithoutPackagePolicies
  } = request.body;
  const agentPolicyRequest = {
    ...request,
    body: agentPolicyWithoutPackagePolicies,
    query: request.query
  };
  const agentPolicyResult = await createAgentPolicyHandler(context, agentPolicyRequest, response);
  const createdAgentPolicy = agentPolicyResult.options.body.item;
  const createdPackagePolicyIds = [];
  if (agentPolicyRequest.body.id && agentPolicyRequest.body.id !== createdAgentPolicy.id) {
    logger.warn(`Agent policy created with id ${createdAgentPolicy.id} instead of requested id ${agentPolicyRequest.body.id}`);
  }

  // Try to create the package policies
  try {
    for (const packagePolicy of packagePolicies) {
      // Extract the original agent policy ID from the request in order to replace it with the created agent policy ID
      const {
        policy_id: agentPolicyId,
        policy_ids: agentPolicyIds,
        ...restOfPackagePolicy
      } = packagePolicy;

      // Warn if the requested agent policy ID does not match the created agent policy ID
      if (agentPolicyId && agentPolicyId !== createdAgentPolicy.id) {
        logger.warn(`Creating package policy with agent policy ID ${createdAgentPolicy.id} instead of requested id ${agentPolicyId}`);
      }
      if (agentPolicyIds && agentPolicyIds.length > 0 && (!agentPolicyIds.includes(createdAgentPolicy.id) || agentPolicyIds.length > 1)) {
        logger.warn(`Creating package policy with agent policy ID ${createdAgentPolicy.id} instead of requested id(s) ${agentPolicyIds.join(',')}`);
      }
      const packagePolicyRequest = {
        ...request,
        body: {
          ...restOfPackagePolicy,
          policy_ids: [createdAgentPolicy.id]
        },
        query: request.query
      };
      const packagePolicyResult = await (0, _handlers.createPackagePolicyHandler)(context, packagePolicyRequest, response);
      createdPackagePolicyIds.push(packagePolicyResult.options.body.item.id);
    }

    // Return the created agent policy with full package policy details
    return getOneAgentPolicyHandler(context, {
      ...request,
      body: {},
      params: {
        agentPolicyId: createdAgentPolicy.id
      }
    }, response);
  } catch (e) {
    // If there is an error creating package policies, delete any created package policy
    // and the parent agent policy
    if (createdPackagePolicyIds.length > 0) {
      await _services2.packagePolicyService.delete(coreContext.savedObjects.client, coreContext.elasticsearch.client.asInternalUser, createdPackagePolicyIds, {
        force: true,
        skipUnassignFromAgentPolicies: true
      });
    }
    if (createdAgentPolicy) {
      await _services2.agentPolicyService.delete(coreContext.savedObjects.client, coreContext.elasticsearch.client.asInternalUser, createdAgentPolicy.id, {
        force: true
      });
    }

    // Rethrow
    throw e;
  }
};
exports.createAgentAndPackagePoliciesHandler = createAgentAndPackagePoliciesHandler;
const updateAgentPolicyHandler = async (context, request, response) => {
  const logger = _services2.appContextService.getLogger().get('httpUpdateAgentPolicyHandler');
  const coreContext = await context.core;
  const fleetContext = await context.fleet;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const user = _services2.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  const {
    force,
    bumpRevision,
    ...data
  } = request.body;
  const spaceIds = data.space_ids;
  let spaceId = fleetContext.spaceId;
  logger.debug(`updating policy [${request.params.agentPolicyId}] in space [${spaceId}]`);
  try {
    const authzFleetAgentsAll = fleetContext.authz.fleet.allAgents;
    const requestSpaceId = spaceId;
    if (spaceIds !== null && spaceIds !== void 0 && spaceIds.length) {
      const authorizedSpaces = await checkAgentPoliciesAllPrivilegesForSpaces(request, context, spaceIds);
      await (0, _agent_policy.updateAgentPolicySpaces)({
        agentPolicy: {
          ...data,
          id: request.params.agentPolicyId
        },
        currentSpaceId: spaceId,
        authorizedSpaces,
        options: {
          force
        }
      });
      spaceId = spaceIds[0];
      logger.debug(`spaceId now set to [${spaceId}] for updating agent policy [${request.params.agentPolicyId}]`);
    }
    const agentPolicy = await _services2.agentPolicyService.update(_services2.appContextService.getInternalUserSOClientForSpaceId(spaceId), esClient, request.params.agentPolicyId, data, {
      force,
      bumpRevision,
      user,
      spaceId,
      requestSpaceId,
      isRequiredVersionsAuthorized: authzFleetAgentsAll
    });
    let item = agentPolicy;
    if (request.query.format === _constants.inputsFormat.Simplified && !(0, _lodash.isEmpty)(agentPolicy.package_policies)) {
      item = {
        ...agentPolicy,
        package_policies: agentPolicy.package_policies.map(packagePolicy => (0, _simplified_package_policy_helper.packagePolicyToSimplifiedPackagePolicy)(packagePolicy))
      };
    }
    const body = {
      item
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.statusCode) {
      return response.customError({
        statusCode: error.statusCode,
        body: {
          message: error.message
        }
      });
    }
    throw error;
  }
};
exports.updateAgentPolicyHandler = updateAgentPolicyHandler;
const copyAgentPolicyHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const user = _services2.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  try {
    const agentPolicy = await _services2.agentPolicyService.copy(soClient, esClient, request.params.agentPolicyId, request.body, {
      user
    });
    let item = agentPolicy;
    if (request.query.format === _constants.inputsFormat.Simplified && !(0, _lodash.isEmpty)(agentPolicy.package_policies)) {
      item = {
        ...agentPolicy,
        package_policies: agentPolicy.package_policies.map(packagePolicy => (0, _simplified_package_policy_helper.packagePolicyToSimplifiedPackagePolicy)(packagePolicy))
      };
    }
    const body = {
      item
    };
    return response.ok({
      body
    });
  } catch (error) {
    throw error;
  }
};
exports.copyAgentPolicyHandler = copyAgentPolicyHandler;
const deleteAgentPoliciesHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const user = _services2.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  const body = await _services2.agentPolicyService.delete(soClient, esClient, request.body.agentPolicyId, {
    user,
    force: request.body.force
  });
  return response.ok({
    body
  });
};
exports.deleteAgentPoliciesHandler = deleteAgentPoliciesHandler;
const getFullAgentPolicy = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const soClient = fleetContext.internalSoClient;
  if (request.query.kubernetes === true) {
    const agentVersion = await fleetContext.agentClient.asInternalUser.getLatestAgentAvailableVersion();
    const fullAgentConfigMap = await _services2.agentPolicyService.getFullAgentConfigMap(soClient, request.params.agentPolicyId, agentVersion, {
      standalone: request.query.standalone === true
    });
    if (fullAgentConfigMap) {
      const body = {
        item: fullAgentConfigMap
      };
      return response.ok({
        body
      });
    } else {
      return response.customError({
        statusCode: 404,
        body: {
          message: 'Agent config map not found'
        }
      });
    }
  } else {
    const fullAgentPolicy = await _services2.agentPolicyService.getFullAgentPolicy(soClient, request.params.agentPolicyId, {
      standalone: request.query.standalone === true
    });
    if (fullAgentPolicy) {
      const body = {
        item: fullAgentPolicy
      };
      return response.ok({
        body
      });
    } else {
      return response.customError({
        statusCode: 404,
        body: {
          message: 'Agent policy not found'
        }
      });
    }
  }
};
exports.getFullAgentPolicy = getFullAgentPolicy;
const downloadFullAgentPolicy = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const soClient = fleetContext.internalSoClient;
  const {
    params: {
      agentPolicyId
    }
  } = request;
  if (request.query.kubernetes === true) {
    const agentVersion = await fleetContext.agentClient.asInternalUser.getLatestAgentAvailableVersion();
    const fullAgentConfigMap = await _services2.agentPolicyService.getFullAgentConfigMap(soClient, request.params.agentPolicyId, agentVersion, {
      standalone: request.query.standalone === true
    });
    if (fullAgentConfigMap) {
      const body = fullAgentConfigMap;
      const headers = {
        'content-type': 'text/x-yaml',
        'content-disposition': `attachment; filename="elastic-agent-standalone-kubernetes.yml"`
      };
      return response.ok({
        body,
        headers
      });
    } else {
      return response.customError({
        statusCode: 404,
        body: {
          message: 'Agent config map not found'
        }
      });
    }
  } else {
    const fullAgentPolicy = await _services2.agentPolicyService.getFullAgentPolicy(soClient, agentPolicyId, {
      standalone: request.query.standalone === true
    });
    if (fullAgentPolicy) {
      const body = (0, _services.fullAgentPolicyToYaml)(fullAgentPolicy, _jsYaml.dump);
      const headers = {
        'content-type': 'text/x-yaml',
        'content-disposition': `attachment; filename="elastic-agent.yml"`
      };
      return response.ok({
        body,
        headers
      });
    } else {
      return response.customError({
        statusCode: 404,
        body: {
          message: 'Agent policy not found'
        }
      });
    }
  }
};
exports.downloadFullAgentPolicy = downloadFullAgentPolicy;
const getK8sManifest = async (context, request, response) => {
  var _request$query$fleetS, _request$query$enrolT;
  const fleetServer = (_request$query$fleetS = request.query.fleetServer) !== null && _request$query$fleetS !== void 0 ? _request$query$fleetS : '';
  const token = (_request$query$enrolT = request.query.enrolToken) !== null && _request$query$enrolT !== void 0 ? _request$query$enrolT : '';
  const agentVersion = await (0, _agents.getLatestAvailableAgentVersion)();
  const fullAgentManifest = await _services2.agentPolicyService.getFullAgentManifest(fleetServer, token, agentVersion);
  if (fullAgentManifest) {
    const body = {
      item: fullAgentManifest
    };
    return response.ok({
      body
    });
  } else {
    return response.customError({
      statusCode: 404,
      body: {
        message: 'Agent manifest not found'
      }
    });
  }
};
exports.getK8sManifest = getK8sManifest;
const downloadK8sManifest = async (context, request, response) => {
  var _request$query$fleetS2, _request$query$enrolT2;
  const fleetServer = (_request$query$fleetS2 = request.query.fleetServer) !== null && _request$query$fleetS2 !== void 0 ? _request$query$fleetS2 : '';
  const token = (_request$query$enrolT2 = request.query.enrolToken) !== null && _request$query$enrolT2 !== void 0 ? _request$query$enrolT2 : '';
  const agentVersion = await (0, _agents.getLatestAvailableAgentVersion)();
  const fullAgentManifest = await _services2.agentPolicyService.getFullAgentManifest(fleetServer, token, agentVersion);
  if (fullAgentManifest) {
    const body = fullAgentManifest;
    const headers = {
      'content-type': 'text/x-yaml',
      'content-disposition': `attachment; filename="elastic-agent-managed-kubernetes.yml"`
    };
    return response.ok({
      body,
      headers
    });
  } else {
    return response.customError({
      statusCode: 404,
      body: {
        message: 'Agent manifest not found'
      }
    });
  }
};
exports.downloadK8sManifest = downloadK8sManifest;
const GetAgentPolicyOutputsHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const agentPolicy = await _services2.agentPolicyService.get(soClient, request.params.agentPolicyId);
  if (!agentPolicy) {
    return response.customError({
      statusCode: 404,
      body: {
        message: 'Agent policy not found'
      }
    });
  }
  const outputs = await _services2.agentPolicyService.getAllOutputsForPolicy(soClient, agentPolicy);
  const body = {
    item: outputs
  };
  return response.ok({
    body
  });
};
exports.GetAgentPolicyOutputsHandler = GetAgentPolicyOutputsHandler;
const GetListAgentPolicyOutputsHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const {
    ids
  } = request.body;
  if (!ids) {
    return response.ok({
      body: {
        items: []
      }
    });
  }
  const agentPolicies = await _services2.agentPolicyService.getByIds(soClient, ids, {
    withPackagePolicies: true
  });
  const outputsList = await _services2.agentPolicyService.listAllOutputsForPolicies(soClient, agentPolicies);
  const body = {
    items: outputsList
  };
  return response.ok({
    body
  });
};
exports.GetListAgentPolicyOutputsHandler = GetListAgentPolicyOutputsHandler;