"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAPIRoutes = registerAPIRoutes;
var _configSchema = require("@kbn/config-schema");
var _content_management = require("../../common/content_management");
var _constants = require("./constants");
var _v = require("../content_management/v1");
var _schema = require("../content_management/v1/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const commonRouteConfig = {
  // This route is in development and not yet intended for public use.
  access: 'internal',
  /**
   * `enableQueryVersion` is a temporary solution for testing internal endpoints.
   * Requests to these internal endpoints from Kibana Dev Tools or external clients
   * should include the ?apiVersion=1 query parameter.
   * This will be removed when the API is finalized and moved to a stable version.
   */
  enableQueryVersion: true,
  description: 'This functionality is in technical preview and may be changed or removed in a future release. Elastic will work to fix any issues, but features in technical preview are not subject to the support SLA of official GA features.',
  options: {
    tags: ['oas-tag:Dashboards'],
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      enabled: false,
      reason: 'Relies on Content Client for authorization'
    }
  }
};
const formatResult = item => {
  const {
    id,
    type,
    attributes,
    createdAt,
    updatedAt,
    createdBy,
    updatedBy,
    error,
    managed,
    version,
    ...rest
  } = item;
  return {
    id,
    type,
    data: {
      ...attributes,
      ...rest
    },
    meta: {
      createdAt,
      updatedAt,
      createdBy,
      updatedBy,
      error,
      managed,
      version
    }
  };
};
function registerAPIRoutes({
  http,
  contentManagement,
  restCounter,
  logger
}) {
  const {
    versioned: versionedRouter
  } = http.createRouter();

  // Create API route
  const createRoute = versionedRouter.post({
    path: `${_constants.PUBLIC_API_PATH}/{id?}`,
    summary: 'Create a dashboard',
    ...commonRouteConfig
  });
  createRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'A unique identifier for the dashboard.'
            }
          }))
        }),
        body: _schema.dashboardCreateRequestAttributesSchema
      },
      response: {
        200: {
          body: () => _schema.dashboardAPICreateResultSchema
        }
      }
    }
  }, async (ctx, req, res) => {
    const {
      id
    } = req.params;
    const {
      references,
      spaces: initialNamespaces,
      ...attributes
    } = req.body;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    let result;
    try {
      ({
        result
      } = await client.create(attributes, {
        id,
        references,
        initialNamespaces
      }));
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 409) {
        return res.conflict({
          body: {
            message: `A dashboard with saved object ID ${id} already exists.`
          }
        });
      }
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest({
        body: e
      });
    }
    const formattedResult = formatResult(result.item);
    const response = {
      ...formattedResult,
      meta: {
        ...formattedResult.meta,
        ...result.meta
      }
    };
    return res.ok({
      body: response
    });
  });

  // Update API route

  const updateRoute = versionedRouter.put({
    path: `${_constants.PUBLIC_API_PATH}/{id}`,
    summary: `Update an existing dashboard`,
    ...commonRouteConfig
  });
  updateRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'A unique identifier for the dashboard.'
            }
          })
        }),
        body: _schema.dashboardAttributesSchemaRequest
      },
      response: {
        200: {
          body: () => _schema.dashboardUpdateResultSchema
        }
      }
    }
  }, async (ctx, req, res) => {
    const {
      references,
      ...attributes
    } = req.body;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    let result;
    try {
      ({
        result
      } = await client.update(req.params.id, attributes, {
        references
      }));
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 404) {
        return res.notFound({
          body: {
            message: `A dashboard with saved object ID ${req.params.id} was not found.`
          }
        });
      }
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest({
        body: e.output.payload
      });
    }
    const formattedResult = formatResult(result.item);
    return res.ok({
      body: {
        ...formattedResult,
        meta: {
          ...formattedResult.meta,
          ...result.meta
        }
      }
    });
  });

  // List API route
  const listRoute = versionedRouter.get({
    path: `${_constants.PUBLIC_API_PATH}`,
    summary: `Get a list of dashboards`,
    ...commonRouteConfig
  });
  listRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        query: _configSchema.schema.object({
          page: _configSchema.schema.number({
            meta: {
              description: 'The page number to return. Default is "1".'
            },
            min: 1,
            defaultValue: 1
          }),
          perPage: _configSchema.schema.number({
            meta: {
              description: 'The number of dashboards to display on each page (max 1000). Default is "20".'
            },
            defaultValue: 20,
            min: 1,
            max: 1000
          })
        })
      },
      response: {
        200: {
          body: () => _schema.dashboardListResultAPISchema
        }
      }
    }
  }, async (ctx, req, res) => {
    const {
      page,
      perPage: limit
    } = req.query;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    let result;
    try {
      // TODO add filtering
      ({
        result
      } = await client.search({
        cursor: page.toString(),
        limit
      }, {
        fields: ['title', 'description', 'timeRestore']
      }));
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest();
    }
    const body = {
      items: result.hits.map(formatResult),
      total: result.pagination.total
    };
    return res.ok({
      body
    });
  });

  // Get API route
  const getRoute = versionedRouter.get({
    path: `${_constants.PUBLIC_API_PATH}/{id}`,
    summary: `Get a dashboard`,
    ...commonRouteConfig
  });
  getRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'A unique identifier for the dashboard.'
            }
          })
        })
      },
      response: {
        200: {
          body: () => _v.dashboardAPIGetResultSchema
        }
      }
    }
  }, async (ctx, req, res) => {
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    let result;
    try {
      ({
        result
      } = await client.get(req.params.id));
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 404) {
        return res.notFound({
          body: {
            message: `A dashboard with saved object ID ${req.params.id}] was not found.`
          }
        });
      }
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest(e.message);
    }
    const formattedResult = formatResult(result.item);
    return res.ok({
      body: {
        ...formattedResult,
        meta: {
          ...formattedResult.meta,
          ...result.meta
        }
      }
    });
  });

  // Delete API route
  const deleteRoute = versionedRouter.delete({
    path: `${_constants.PUBLIC_API_PATH}/{id}`,
    summary: `Delete a dashboard`,
    ...commonRouteConfig
  });
  deleteRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'A unique identifier for the dashboard.'
            }
          })
        })
      }
    }
  }, async (ctx, req, res) => {
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    try {
      await client.delete(req.params.id);
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 404) {
        return res.notFound({
          body: {
            message: `A dashboard with saved object ID ${req.params.id} was not found.`
          }
        });
      }
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest();
    }
    return res.ok();
  });
}