"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ProductFeaturesRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Registry for managing pricing product features.
 * Provides methods to register, retrieve, and manage features that are available in specific pricing tiers.
 *
 * @public
 */
class ProductFeaturesRegistry {
  /**
   * Creates a new ProductFeaturesRegistry instance.
   *
   * @param initialFeatures - Optional initial set of features to populate the registry
   */
  constructor(initialFeatures = {}) {
    /**
     * Internal storage for registered product features.
     * @internal
     */
    (0, _defineProperty2.default)(this, "productFeatures", void 0);
    this.productFeatures = new Map(Object.entries(initialFeatures));
  }

  /**
   * Retrieves a product feature by its ID.
   *
   * @param featureId - The ID of the feature to retrieve
   * @returns The product feature if found, undefined otherwise
   */
  get(featureId) {
    return this.productFeatures.get(featureId);
  }

  /**
   * Registers a new product feature in the registry.
   * Throws an error if a feature with the same ID is already registered.
   *
   * @param feature - The product feature to register
   * @throws Error if a feature with the same ID is already registered
   */
  register(feature) {
    if (this.productFeatures.has(feature.id)) {
      throw new Error(`A product feature with id "${feature.id}" is already registered, please change id or check whether is the same feature.`);
    }
    this.productFeatures.set(feature.id, feature);
  }

  /**
   * Converts the registry to a plain JavaScript object.
   *
   * @returns A record mapping feature IDs to their corresponding feature objects
   */
  asObject() {
    return Object.fromEntries(this.productFeatures);
  }
}
exports.ProductFeaturesRegistry = ProductFeaturesRegistry;