"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.gatherInfo = gatherInfo;
var _promises = _interopRequireDefault(require("fs/promises"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const CONTROL_GROUP_RE = new RegExp('\\d+:([^:]+):(/.*)');
const CONTROLLER_SEPARATOR_RE = ',';
const PROC_SELF_CGROUP_FILE = '/proc/self/cgroup';

/**
 * Check whether the contents of /proc/self/cgroup indicate that we are running in a cgroup v2
 *
 * @note cgroup v2 is always in the format "0::<PATH>". See https://www.kernel.org/doc/Documentation/cgroup-v2.txt.
 */
function isCgroups2(procSelfLines) {
  return procSelfLines.length === 1 && procSelfLines[0].trim().startsWith('0::');
}
async function readProcSelf() {
  const data = (await _promises.default.readFile(PROC_SELF_CGROUP_FILE)).toString();
  return data.split(/\n/).filter(line => line.trim().length > 0);
}
async function gatherInfo() {
  const lines = await readProcSelf();
  if (isCgroups2(lines)) {
    // eslint-disable-next-line prettier/prettier
    const [/* '0' */ /* '' */,, path] = lines[0].trim().split(':');
    return {
      v2: true,
      path
    };
  }
  const data = lines.reduce((acc, line) => {
    const matches = line.match(CONTROL_GROUP_RE);
    if (matches !== null) {
      const controllers = matches[1].split(CONTROLLER_SEPARATOR_RE);
      controllers.forEach(controller => {
        acc[controller] = matches[2];
      });
    }
    return acc;
  }, {});
  return {
    data,
    v2: false
  };
}