/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch.index.codec.vectors;

import org.apache.lucene.codecs.KnnVectorsFormat;
import org.apache.lucene.codecs.hnsw.FlatVectorsFormat;
import org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat;
import org.apache.lucene.search.TaskExecutor;
import org.apache.lucene.util.hnsw.HnswGraph;

import java.util.concurrent.ExecutorService;

import static org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat.DEFAULT_BEAM_WIDTH;
import static org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat.DEFAULT_MAX_CONN;
import static org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat.DEFAULT_NUM_MERGE_WORKER;
import static org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat.MAXIMUM_BEAM_WIDTH;
import static org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsFormat.MAXIMUM_MAX_CONN;
import static org.elasticsearch.index.mapper.vectors.DenseVectorFieldMapper.MAX_DIMS_COUNT;

public abstract class AbstractHnswVectorsFormat extends KnnVectorsFormat {

    /**
     * Controls how many of the nearest neighbor candidates are connected to the new node. Defaults to
     * {@link Lucene99HnswVectorsFormat#DEFAULT_MAX_CONN}. See {@link HnswGraph} for more details.
     */
    protected final int maxConn;

    /**
     * The number of candidate neighbors to track while searching the graph for each newly inserted
     * node. Defaults to {@link Lucene99HnswVectorsFormat#DEFAULT_BEAM_WIDTH}. See {@link HnswGraph}
     * for details.
     */
    protected final int beamWidth;

    protected final int numMergeWorkers;
    protected final TaskExecutor mergeExec;

    /** Constructs a format using default graph construction parameters */
    protected AbstractHnswVectorsFormat(String name) {
        this(name, DEFAULT_MAX_CONN, DEFAULT_BEAM_WIDTH, DEFAULT_NUM_MERGE_WORKER, null);
    }

    /**
     * Constructs a format using the given graph construction parameters.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     */
    protected AbstractHnswVectorsFormat(String name, int maxConn, int beamWidth) {
        this(name, maxConn, beamWidth, DEFAULT_NUM_MERGE_WORKER, null);
    }

    /**
     * Constructs a format using the given graph construction parameters and scalar quantization.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     * @param numMergeWorkers number of workers (threads) that will be used when doing merge. If
     *     larger than 1, a non-null {@link ExecutorService} must be passed as mergeExec
     * @param mergeExec the {@link ExecutorService} that will be used by ALL vector writers that are
     *     generated by this format to do the merge
     */
    protected AbstractHnswVectorsFormat(String name, int maxConn, int beamWidth, int numMergeWorkers, ExecutorService mergeExec) {
        super(name);
        if (maxConn <= 0 || maxConn > MAXIMUM_MAX_CONN) {
            throw new IllegalArgumentException(
                "maxConn must be positive and less than or equal to " + MAXIMUM_MAX_CONN + "; maxConn=" + maxConn
            );
        }
        if (beamWidth <= 0 || beamWidth > MAXIMUM_BEAM_WIDTH) {
            throw new IllegalArgumentException(
                "beamWidth must be positive and less than or equal to " + MAXIMUM_BEAM_WIDTH + "; beamWidth=" + beamWidth
            );
        }
        this.maxConn = maxConn;
        this.beamWidth = beamWidth;
        if (numMergeWorkers == 1 && mergeExec != null) {
            throw new IllegalArgumentException("No executor service is needed as we'll use single thread to merge");
        }
        this.numMergeWorkers = numMergeWorkers;
        if (mergeExec != null) {
            this.mergeExec = new TaskExecutor(mergeExec);
        } else {
            this.mergeExec = null;
        }
    }

    protected abstract FlatVectorsFormat flatVectorsFormat();

    @Override
    public int getMaxDimensions(String fieldName) {
        return MAX_DIMS_COUNT;
    }

    @Override
    public String toString() {
        return getName()
            + "(name="
            + getName()
            + ", maxConn="
            + maxConn
            + ", beamWidth="
            + beamWidth
            + ", flatVectorFormat="
            + flatVectorsFormat()
            + ")";
    }
}
