/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
package org.elasticsearch.cluster.coordination;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;

import org.elasticsearch.cli.ExitCodes;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.core.FixForMultiProject;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.PersistedClusterStateService;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

public class RemoveCustomsCommand extends ElasticsearchNodeCommand {

    static final String CUSTOMS_REMOVED_MSG = "Customs were successfully removed from the cluster state";
    static final String CONFIRMATION_MSG = DELIMITER
        + "\n"
        + "You should only run this tool if you have broken custom metadata in the\n"
        + "cluster state that prevents the cluster state from being loaded.\n"
        + "This tool can cause data loss and its use should be your last resort.\n"
        + "\n"
        + "Do you want to proceed?\n";

    private final OptionSpec<String> arguments;

    public RemoveCustomsCommand() {
        super("Removes custom metadata from the cluster state");
        arguments = parser.nonOptions("custom metadata names");
    }

    @Override
    protected void processDataPaths(Terminal terminal, Path[] dataPaths, OptionSet options, Environment env) throws IOException,
        UserException {
        final List<String> customsToRemove = arguments.values(options);
        if (customsToRemove.isEmpty()) {
            throw new UserException(ExitCodes.USAGE, "Must supply at least one custom metadata name to remove");
        }

        final PersistedClusterStateService persistedClusterStateService = createPersistedClusterStateService(env.settings(), dataPaths);

        terminal.println(Terminal.Verbosity.VERBOSE, "Loading cluster state");
        final Tuple<Long, ClusterState> termAndClusterState = loadTermAndClusterState(persistedClusterStateService, env);
        final ClusterState oldClusterState = termAndClusterState.v2();
        terminal.println(
            Terminal.Verbosity.VERBOSE,
            "cluster scoped custom metadata names: " + oldClusterState.metadata().customs().keySet()
        );
        terminal.println(
            Terminal.Verbosity.VERBOSE,
            "project scoped custom metadata names: " + oldClusterState.metadata().getProject().customs().keySet()
        );
        final Metadata.Builder metadataBuilder = Metadata.builder(oldClusterState.metadata());
        @FixForMultiProject
        final ProjectMetadata.Builder projectBuilder = metadataBuilder.getProject(ProjectId.DEFAULT);
        for (String customToRemove : customsToRemove) {
            @FixForMultiProject
            boolean matched = false;
            // TODO[MultiProject]: Should we add a scope flag to the command, or just iterate through both maps?
            for (String customKey : oldClusterState.metadata().customs().keySet()) {
                if (Regex.simpleMatch(customToRemove, customKey)) {
                    metadataBuilder.removeCustom(customKey);
                    if (matched == false) {
                        terminal.println("The following customs will be removed:");
                    }
                    matched = true;
                    terminal.println(customKey);
                }
            }
            for (String customKey : oldClusterState.metadata().getProject().customs().keySet()) {
                if (Regex.simpleMatch(customToRemove, customKey)) {
                    projectBuilder.removeCustom(customKey);
                    if (matched == false) {
                        terminal.println("The following customs will be removed:");
                    }
                    matched = true;
                    terminal.println(customKey);
                }
            }
            if (matched == false) {
                throw new UserException(ExitCodes.USAGE, "No custom metadata matching [" + customToRemove + "] were found on this node");
            }
        }
        final ClusterState newClusterState = ClusterState.builder(oldClusterState).metadata(metadataBuilder.build()).build();
        terminal.println(
            Terminal.Verbosity.VERBOSE,
            "[old cluster state = " + oldClusterState + ", new cluster state = " + newClusterState + "]"
        );

        confirm(terminal, CONFIRMATION_MSG);

        try (PersistedClusterStateService.Writer writer = persistedClusterStateService.createWriter()) {
            writer.writeFullStateAndCommit(termAndClusterState.v1(), newClusterState);
        }

        terminal.println(CUSTOMS_REMOVED_MSG);
    }
}
