/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.IntConsumer;
import java.util.function.IntSupplier;
import java.util.function.Predicate;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.RoutingMissingException;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.routing.Murmur3HashFunction;
import org.elasticsearch.common.ParsingException;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.util.ByteUtils;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentParserUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.features.NodeFeature;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.mapper.TimeSeriesRoutingHashFieldMapper;
import org.elasticsearch.transport.Transports;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;

public abstract class IndexRouting {
    static final NodeFeature LOGSB_ROUTE_ON_SORT_FIELDS = new NodeFeature("routing.logsb_route_on_sort_fields");
    protected final String indexName;
    private final int routingNumShards;
    private final int routingFactor;

    public static IndexRouting fromIndexMetadata(IndexMetadata metadata) {
        if (!metadata.getRoutingPaths().isEmpty()) {
            return new ExtractFromSource(metadata);
        }
        if (metadata.isRoutingPartitionedIndex()) {
            return new Partitioned(metadata);
        }
        return new Unpartitioned(metadata);
    }

    private IndexRouting(IndexMetadata metadata) {
        this.indexName = metadata.getIndex().getName();
        this.routingNumShards = metadata.getRoutingNumShards();
        this.routingFactor = metadata.getRoutingFactor();
    }

    public void preProcess(IndexRequest indexRequest) {
    }

    public void postProcess(IndexRequest indexRequest) {
    }

    public abstract int indexShard(String var1, @Nullable String var2, XContentType var3, BytesReference var4);

    public abstract int updateShard(String var1, @Nullable String var2);

    public abstract int deleteShard(String var1, @Nullable String var2);

    public abstract int getShard(String var1, @Nullable String var2);

    public abstract void collectSearchShards(String var1, IntConsumer var2);

    protected final int hashToShardId(int hash) {
        return Math.floorMod(hash, this.routingNumShards) / this.routingFactor;
    }

    private static int effectiveRoutingToHash(String effectiveRouting) {
        return Murmur3HashFunction.hash(effectiveRouting);
    }

    public void checkIndexSplitAllowed() {
    }

    public static class ExtractFromSource
    extends IndexRouting {
        private final Predicate<String> isRoutingPath;
        private final XContentParserConfiguration parserConfig;
        private final IndexMode indexMode;
        private final boolean trackTimeSeriesRoutingHash;
        private final boolean addIdWithRoutingHash;
        private int hash = Integer.MAX_VALUE;

        ExtractFromSource(IndexMetadata metadata) {
            super(metadata);
            if (metadata.isRoutingPartitionedIndex()) {
                throw new IllegalArgumentException("routing_partition_size is incompatible with routing_path");
            }
            this.indexMode = metadata.getIndexMode();
            this.trackTimeSeriesRoutingHash = this.indexMode == IndexMode.TIME_SERIES && metadata.getCreationVersion().onOrAfter(IndexVersions.TIME_SERIES_ROUTING_HASH_IN_ID);
            this.addIdWithRoutingHash = this.indexMode == IndexMode.LOGSDB;
            List<String> routingPaths = metadata.getRoutingPaths();
            this.isRoutingPath = Regex.simpleMatcher((String[])routingPaths.toArray(String[]::new));
            this.parserConfig = XContentParserConfiguration.EMPTY.withFiltering(null, Set.copyOf(routingPaths), null, true);
        }

        public boolean matchesField(String fieldName) {
            return this.isRoutingPath.test(fieldName);
        }

        @Override
        public void postProcess(IndexRequest indexRequest) {
            if (this.trackTimeSeriesRoutingHash) {
                indexRequest.routing(TimeSeriesRoutingHashFieldMapper.encode(this.hash));
            } else if (this.addIdWithRoutingHash) {
                assert (this.hash != Integer.MAX_VALUE);
                indexRequest.autoGenerateTimeBasedId(OptionalInt.of(this.hash));
            }
        }

        @Override
        public int indexShard(String id, @Nullable String routing, XContentType sourceType, BytesReference source) {
            assert (Transports.assertNotTransportThread("parsing the _source can get slow"));
            this.checkNoRouting(routing);
            this.hash = this.hashSource(sourceType, source).buildHash(ExtractFromSource::defaultOnEmpty);
            return this.hashToShardId(this.hash);
        }

        public String createId(XContentType sourceType, BytesReference source, byte[] suffix) {
            return this.hashSource(sourceType, source).createId(suffix, ExtractFromSource::defaultOnEmpty);
        }

        public String createId(Map<String, Object> flat, byte[] suffix) {
            Builder b = this.builder();
            for (Map.Entry<String, Object> e : flat.entrySet()) {
                if (!this.isRoutingPath.test(e.getKey())) continue;
                Iterator iterator = e.getValue();
                if (iterator instanceof List) {
                    List listValue = (List)((Object)iterator);
                    for (Object v : listValue) {
                        b.addHash(e.getKey(), new BytesRef(v.toString()));
                    }
                    continue;
                }
                b.addHash(e.getKey(), new BytesRef(e.getValue().toString()));
            }
            return b.createId(suffix, ExtractFromSource::defaultOnEmpty);
        }

        private static int defaultOnEmpty() {
            throw new IllegalArgumentException("Error extracting routing: source didn't contain any routing fields");
        }

        public Builder builder() {
            return new Builder();
        }

        private Builder hashSource(XContentType sourceType, BytesReference source) {
            Builder b = this.builder();
            try (XContentParser parser = XContentHelper.createParserNotCompressed(this.parserConfig, source, sourceType);){
                parser.nextToken();
                if (parser.currentToken() == null) {
                    throw new IllegalArgumentException("Error extracting routing: source didn't contain any routing fields");
                }
                parser.nextToken();
                b.extractObject(null, parser);
                XContentParserUtils.ensureExpectedToken(null, parser.nextToken(), parser);
            }
            catch (IOException | ParsingException e) {
                throw new IllegalArgumentException("Error extracting routing: " + e.getMessage(), e);
            }
            return b;
        }

        private static int hash(BytesRef ref) {
            return StringHelper.murmurhash3_x86_32(ref, 0);
        }

        @Override
        public int updateShard(String id, @Nullable String routing) {
            throw new IllegalArgumentException(this.error("update"));
        }

        @Override
        public int deleteShard(String id, @Nullable String routing) {
            this.checkNoRouting(routing);
            return this.idToHash(id);
        }

        @Override
        public int getShard(String id, @Nullable String routing) {
            this.checkNoRouting(routing);
            return this.idToHash(id);
        }

        private void checkNoRouting(@Nullable String routing) {
            if (routing != null) {
                throw new IllegalArgumentException(this.error("specifying routing"));
            }
        }

        private int idToHash(String id) {
            byte[] idBytes;
            try {
                idBytes = Base64.getUrlDecoder().decode(id);
            }
            catch (IllegalArgumentException e) {
                throw new ResourceNotFoundException("invalid id [{}] for index [{}] in " + this.indexMode.getName() + " mode", id, this.indexName);
            }
            if (idBytes.length < 4) {
                throw new ResourceNotFoundException("invalid id [{}] for index [{}] in " + this.indexMode.getName() + " mode", id, this.indexName);
            }
            return this.hashToShardId(ByteUtils.readIntLE(idBytes, this.addIdWithRoutingHash ? idBytes.length - 9 : 0));
        }

        @Override
        public void checkIndexSplitAllowed() {
            throw new IllegalArgumentException(this.error("index-split"));
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            throw new IllegalArgumentException(this.error("searching with a specified routing"));
        }

        private String error(String operation) {
            return operation + " is not supported because the destination index [" + this.indexName + "] is in " + this.indexMode.getName() + " mode";
        }

        public class Builder {
            private final List<NameAndHash> hashes = new ArrayList<NameAndHash>();

            public void addMatching(String fieldName, BytesRef string) {
                if (ExtractFromSource.this.isRoutingPath.test(fieldName)) {
                    this.addHash(fieldName, string);
                }
            }

            public String createId(byte[] suffix, IntSupplier onEmpty) {
                byte[] idBytes = new byte[4 + suffix.length];
                ByteUtils.writeIntLE(this.buildHash(onEmpty), idBytes, 0);
                System.arraycopy(suffix, 0, idBytes, 4, suffix.length);
                return Strings.BASE_64_NO_PADDING_URL_ENCODER.encodeToString(idBytes);
            }

            private void extractObject(@Nullable String path, XContentParser source) throws IOException {
                while (source.currentToken() != XContentParser.Token.END_OBJECT) {
                    XContentParserUtils.ensureExpectedToken(XContentParser.Token.FIELD_NAME, source.currentToken(), source);
                    String fieldName = source.currentName();
                    String subPath = path == null ? fieldName : path + "." + fieldName;
                    source.nextToken();
                    this.extractItem(subPath, source);
                }
            }

            private void extractArray(@Nullable String path, XContentParser source) throws IOException {
                while (source.currentToken() != XContentParser.Token.END_ARRAY) {
                    XContentParserUtils.expectValueToken(source.currentToken(), source);
                    this.extractItem(path, source);
                }
            }

            private void extractItem(String path, XContentParser source) throws IOException {
                switch (source.currentToken()) {
                    case START_OBJECT: {
                        source.nextToken();
                        this.extractObject(path, source);
                        source.nextToken();
                        break;
                    }
                    case VALUE_STRING: 
                    case VALUE_NUMBER: 
                    case VALUE_BOOLEAN: {
                        this.addHash(path, new BytesRef(source.text()));
                        source.nextToken();
                        break;
                    }
                    case START_ARRAY: {
                        source.nextToken();
                        this.extractArray(path, source);
                        source.nextToken();
                        break;
                    }
                    case VALUE_NULL: {
                        source.nextToken();
                        break;
                    }
                    default: {
                        throw new ParsingException(source.getTokenLocation(), "Cannot extract routing path due to unexpected token [{}]", new Object[]{source.currentToken()});
                    }
                }
            }

            private void addHash(String path, BytesRef value) {
                this.hashes.add(new NameAndHash(new BytesRef(path), ExtractFromSource.hash(value), this.hashes.size()));
            }

            private int buildHash(IntSupplier onEmpty) {
                if (this.hashes.isEmpty()) {
                    return onEmpty.getAsInt();
                }
                Collections.sort(this.hashes);
                int hash = 0;
                for (NameAndHash nah : this.hashes) {
                    hash = 31 * hash + (ExtractFromSource.hash(nah.name) ^ nah.hash);
                }
                return hash;
            }
        }
    }

    private static class Partitioned
    extends IdAndRoutingOnly {
        private final int routingPartitionSize;

        Partitioned(IndexMetadata metadata) {
            super(metadata);
            this.routingPartitionSize = metadata.getRoutingPartitionSize();
        }

        @Override
        protected int shardId(String id, @Nullable String routing) {
            if (routing == null) {
                throw new IllegalArgumentException("A routing value is required for gets from a partitioned index");
            }
            int offset = Math.floorMod(IndexRouting.effectiveRoutingToHash(id), this.routingPartitionSize);
            return this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing) + offset);
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            int hash = IndexRouting.effectiveRoutingToHash(routing);
            for (int i = 0; i < this.routingPartitionSize; ++i) {
                consumer.accept(this.hashToShardId(hash + i));
            }
        }
    }

    private static class Unpartitioned
    extends IdAndRoutingOnly {
        Unpartitioned(IndexMetadata metadata) {
            super(metadata);
        }

        @Override
        protected int shardId(String id, @Nullable String routing) {
            return this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing == null ? id : routing));
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            consumer.accept(this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing)));
        }
    }

    private record NameAndHash(BytesRef name, int hash, int order) implements Comparable<NameAndHash>
    {
        @Override
        public int compareTo(NameAndHash o) {
            int i = this.name.compareTo(o.name);
            if (i != 0) {
                return i;
            }
            return Integer.compare(this.order, o.order);
        }
    }

    private static abstract class IdAndRoutingOnly
    extends IndexRouting {
        private final boolean routingRequired;
        private final IndexVersion creationVersion;
        private final IndexMode indexMode;

        IdAndRoutingOnly(IndexMetadata metadata) {
            super(metadata);
            this.creationVersion = metadata.getCreationVersion();
            MappingMetadata mapping = metadata.mapping();
            this.routingRequired = mapping == null ? false : mapping.routingRequired();
            this.indexMode = metadata.getIndexMode();
        }

        protected abstract int shardId(String var1, @Nullable String var2);

        @Override
        public void preProcess(IndexRequest indexRequest) {
            String id = indexRequest.id();
            if (id == null) {
                if (IdAndRoutingOnly.shouldUseTimeBasedId(this.indexMode, this.creationVersion)) {
                    indexRequest.autoGenerateTimeBasedId();
                } else {
                    indexRequest.autoGenerateId();
                }
            } else if (id.isEmpty()) {
                throw new IllegalArgumentException("if _id is specified it must not be empty");
            }
        }

        private static boolean shouldUseTimeBasedId(IndexMode indexMode, IndexVersion creationVersion) {
            return indexMode == IndexMode.LOGSDB && IdAndRoutingOnly.isNewIndexVersion(creationVersion);
        }

        private static boolean isNewIndexVersion(IndexVersion creationVersion) {
            return creationVersion.between(IndexVersions.TIME_BASED_K_ORDERED_DOC_ID_BACKPORT, IndexVersions.UPGRADE_TO_LUCENE_10_0_0) || creationVersion.onOrAfter(IndexVersions.TIME_BASED_K_ORDERED_DOC_ID);
        }

        @Override
        public int indexShard(String id, @Nullable String routing, XContentType sourceType, BytesReference source) {
            if (id == null) {
                throw new IllegalStateException("id is required and should have been set by process");
            }
            this.checkRoutingRequired(id, routing);
            return this.shardId(id, routing);
        }

        @Override
        public int updateShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            return this.shardId(id, routing);
        }

        @Override
        public int deleteShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            return this.shardId(id, routing);
        }

        @Override
        public int getShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            return this.shardId(id, routing);
        }

        private void checkRoutingRequired(String id, @Nullable String routing) {
            if (this.routingRequired && routing == null) {
                throw new RoutingMissingException(this.indexName, id);
            }
        }
    }
}

