"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runV2Migration = void 0;
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _semver = _interopRequireWildcard(require("semver"));
var _lodash = require("lodash");
var _core = require("./core");
var _kibana_migrator_utils = require("./kibana_migrator_utils");
var _run_resilient_migrator = require("./run_resilient_migrator");
var _migrate_raw_docs = require("./core/migrate_raw_docs");
var _get_index_details = require("./core/get_index_details");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const runV2Migration = async options => {
  var _indexDetails, _indexDetails$mapping, _indexDetails2, _indexDetails2$mappin, _indexDetails2$mappin2;
  const mainIndex = options.kibanaIndexPrefix;
  let indexDetails;
  try {
    // try to find out if `.kibana index already exists, and get some information from it
    indexDetails = await (0, _get_index_details.getIndexDetails)(options.elasticsearchClient, mainIndex);
  } catch (error) {
    var _error$meta;
    if (((_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) === 404) {
      options.logger.debug(`The ${mainIndex} index do NOT exist. Assuming this is a fresh deployment`);
    } else {
      options.logger.fatal(`Cannot query the meta information on the ${mainIndex} saved object index`);
      throw error;
    }
  }

  // if the .kibana index exists, ensure previous Kibana version is >= 8.18.0
  if (options.kibanaVersionCheck && (_indexDetails = indexDetails) !== null && _indexDetails !== void 0 && _indexDetails.aliases) {
    // .kibana index exists and should have version aliases
    const previousKibanaVersion = (0, _get_index_details.extractVersionFromKibanaIndexAliases)(indexDetails.aliases);
    if (!previousKibanaVersion) {
      throw new Error(`Cannot determine Kibana version from the ${mainIndex} aliases [${indexDetails.aliases}]. If you are running a Kibana version <= 7.11.0, please upgrade to 8.18.0 or 8.19.0 before upgrading to 9.x series`);
    }
    if (new _semver.SemVer(options.kibanaVersionCheck).compare(previousKibanaVersion) === 1) {
      const currentMajor = new _semver.SemVer(options.kibanaVersion).major;
      throw new Error(`Kibana ${previousKibanaVersion} deployment detected. Please upgrade to Kibana ${options.kibanaVersionCheck} or newer before upgrading to ${currentMajor}.x series.`);
    }
  }
  const indexMap = (0, _core.createIndexMap)({
    kibanaIndexName: options.kibanaIndexPrefix,
    indexMap: options.mappingProperties,
    registry: options.typeRegistry
  });
  options.logger.debug('Applying registered migrations for the following saved object types:');
  Object.entries(options.documentMigrator.getMigrationVersion()).sort(([t1, v1], [t2, v2]) => {
    return _semver.default.compare(v1, v2);
  }).forEach(([type, migrationVersion]) => {
    options.logger.debug(`migrationVersion: ${migrationVersion} saved object type: ${type}`);
  });

  // build a indexTypesMap from the info present in the typeRegistry, e.g.:
  // {
  //   '.kibana': ['typeA', 'typeB', ...]
  //   '.kibana_task_manager': ['task', ...]
  //   '.kibana_cases': ['typeC', 'typeD', ...]
  //   ...
  // }
  const indexTypesMap = (0, _kibana_migrator_utils.indexMapToIndexTypesMap)(indexMap);

  // compare indexTypesMap with the one present (or not) in the .kibana index meta
  // and check if some SO types have been moved to different indices
  const indicesWithRelocatingTypes = indexDetails ?
  // the .kibana index exists, we might have to relocate some SO to different indices
  (0, _kibana_migrator_utils.getIndicesInvolvedInRelocation)((_indexDetails$mapping = (_indexDetails2 = indexDetails) === null || _indexDetails2 === void 0 ? void 0 : (_indexDetails2$mappin = _indexDetails2.mappings) === null || _indexDetails2$mappin === void 0 ? void 0 : (_indexDetails2$mappin2 = _indexDetails2$mappin._meta) === null || _indexDetails2$mappin2 === void 0 ? void 0 : _indexDetails2$mappin2.indexTypesMap) !== null && _indexDetails$mapping !== void 0 ? _indexDetails$mapping : options.defaultIndexTypesMap, indexTypesMap) :
  // this is a fresh deployment, no indices involved in a relocation
  [];

  // we create synchronization objects (synchronization points) for each of the
  // migrators involved in relocations, aka each of the migrators that will:
  // A) reindex some documents TO other indices
  // B) receive some documents FROM other indices
  // C) both
  const readyToReindexWaitGroupMap = (0, _kibana_migrator_utils.createWaitGroupMap)(indicesWithRelocatingTypes);
  const doneReindexingWaitGroupMap = (0, _kibana_migrator_utils.createWaitGroupMap)(indicesWithRelocatingTypes);
  const updateAliasesWaitGroupMap = (0, _kibana_migrator_utils.createWaitGroupMap)(indicesWithRelocatingTypes);

  // build a list of all migrators that must be started
  const migratorIndices = new Set(Object.keys(indexMap));
  // the types in indices involved in relocation might not have mappings in the current mappings anymore
  // but if their SOs must be relocated to another index, we still need a migrator to do the job
  indicesWithRelocatingTypes.forEach(index => migratorIndices.add(index));

  // we will store model versions instead of hashes (to be FIPS compliant)
  const appVersions = (0, _coreSavedObjectsBaseServerInternal.getVirtualVersionMap)({
    types: options.typeRegistry.getAllTypes(),
    useModelVersionsOnly: true
  });
  const migrators = Array.from(migratorIndices).map((indexName, i) => {
    return {
      migrate: () => {
        var _indexMap$indexName$t, _indexMap$indexName, _indexMap$indexName2;
        const readyToReindex = readyToReindexWaitGroupMap[indexName];
        const doneReindexing = doneReindexingWaitGroupMap[indexName];
        const updateRelocationAliases = updateAliasesWaitGroupMap[indexName];
        // check if this migrator's index is involved in some document redistribution
        const mustRelocateDocuments = indicesWithRelocatingTypes.includes(indexName);

        // a migrator's index might no longer have any associated types to it
        const typeDefinitions = (_indexMap$indexName$t = (_indexMap$indexName = indexMap[indexName]) === null || _indexMap$indexName === void 0 ? void 0 : _indexMap$indexName.typeMappings) !== null && _indexMap$indexName$t !== void 0 ? _indexMap$indexName$t : {};
        const indexTypes = Object.keys(typeDefinitions);
        // store only the model versions of SO types that belong to the index
        const mappingVersions = (0, _lodash.pick)(appVersions, indexTypes);
        const _meta = {
          indexTypesMap,
          mappingVersions
        };
        return (0, _run_resilient_migrator.runResilientMigrator)({
          client: options.elasticsearchClient,
          kibanaVersion: options.kibanaVersion,
          mustRelocateDocuments,
          indexTypes,
          indexTypesMap,
          hashToVersionMap: options.hashToVersionMap,
          waitForMigrationCompletion: options.waitForMigrationCompletion,
          targetIndexMappings: (0, _core.buildActiveMappings)(typeDefinitions, _meta),
          logger: options.logger,
          preMigrationScript: (_indexMap$indexName2 = indexMap[indexName]) === null || _indexMap$indexName2 === void 0 ? void 0 : _indexMap$indexName2.script,
          readyToReindex,
          doneReindexing,
          updateRelocationAliases,
          transformRawDocs: rawDocs => (0, _migrate_raw_docs.migrateRawDocsSafely)({
            serializer: options.serializer,
            migrateDoc: options.documentMigrator.migrateAndConvert,
            rawDocs
          }),
          coreMigrationVersionPerType: options.documentMigrator.getMigrationVersion({
            includeDeferred: false,
            migrationType: 'core'
          }),
          migrationVersionPerType: options.documentMigrator.getMigrationVersion({
            includeDeferred: false
          }),
          indexPrefix: indexName,
          migrationsConfig: options.migrationConfig,
          typeRegistry: options.typeRegistry,
          docLinks: options.docLinks,
          esCapabilities: options.esCapabilities
        });
      }
    };
  });
  return Promise.all(migrators.map(migrator => migrator.migrate()));
};
exports.runV2Migration = runV2Migration;