"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertGraphEvents = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _backlog = require("./backlog");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatResearchGoalReasoning = researchGoal => {
  return `${researchGoal.reasoning}\n\nDefining the research goal as: "${researchGoal.researchGoal}"`;
};
const formatReflectionResult = reflection => {
  let formatted = `${reflection.reasoning}\n\nThe current information are ${reflection.isSufficient ? '*sufficient*' : '*insufficient*'}`;
  if (reflection.nextQuestions.length > 0) {
    formatted += `\n\nThe following questions should be followed up on: ${reflection.nextQuestions.map(question => ` - ${question}`).join(', ')}`;
  }
  return formatted;
};
const convertGraphEvents = ({
  graphName,
  toolIdMapping
}) => {
  return streamEvents$ => {
    const messageId = (0, _uuid.v4)();
    return streamEvents$.pipe((0, _rxjs.mergeMap)(event => {
      if (!(0, _langchain.matchGraphName)(event, graphName)) {
        return _rxjs.EMPTY;
      }

      // clarification response text chunks
      if ((0, _langchain.matchEvent)(event, 'on_chat_model_stream') && (0, _langchain.hasTag)(event, 'researcher-ask-for-clarification')) {
        const chunk = event.data.chunk;
        const textContent = (0, _langchain.extractTextContent)(chunk);
        if (textContent) {
          const messageChunkEvent = (0, _langchain.createTextChunkEvent)(chunk, {
            defaultMessageId: messageId
          });
          return (0, _rxjs.of)(messageChunkEvent);
        }
      }

      // clarification response message
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, 'identify_research_goal')) {
        const {
          generatedAnswer
        } = event.data.output;
        if (generatedAnswer) {
          const messageEvent = (0, _langchain.createMessageEvent)(generatedAnswer);
          return (0, _rxjs.of)(messageEvent);
        }
      }

      // research goal reasoning events
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, 'identify_research_goal')) {
        const {
          backlog
        } = event.data.output;
        const researchGoalResult = (0, _backlog.firstResearchGoalResult)(backlog);
        const reasoningEvent = (0, _langchain.createReasoningEvent)(formatResearchGoalReasoning(researchGoalResult));
        return (0, _rxjs.of)(reasoningEvent);
      }

      // answer step text chunks
      if ((0, _langchain.matchEvent)(event, 'on_chat_model_stream') && (0, _langchain.hasTag)(event, 'researcher-answer')) {
        const chunk = event.data.chunk;
        const messageChunkEvent = (0, _langchain.createTextChunkEvent)(chunk, {
          defaultMessageId: messageId
        });
        return (0, _rxjs.of)(messageChunkEvent);
      }

      // answer step response message
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, 'answer')) {
        const {
          generatedAnswer
        } = event.data.output;
        const messageEvent = (0, _langchain.createMessageEvent)(generatedAnswer);
        return (0, _rxjs.of)(messageEvent);
      }

      // reasoning events for reflection step
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, 'reflection')) {
        const {
          backlog
        } = event.data.output;
        const reflectionResult = (0, _backlog.lastReflectionResult)(backlog);
        const reasoningEvent = (0, _langchain.createReasoningEvent)(formatReflectionResult(reflectionResult));
        return (0, _rxjs.of)(reasoningEvent);
      }
      return _rxjs.EMPTY;
    }));
  };
};
exports.convertGraphEvents = convertGraphEvents;