"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateObjectDiff = calculateObjectDiff;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isAllUndefined(obj) {
  return Array.isArray(obj) && obj.every(value => value === undefined);
}

/**
 * Compares two JSON objects and calculates the added and removed properties, including nested properties.
 * @param oldObj - The base object.
 * @param newObj - The comparison object.
 * @returns An object containing added and removed properties.
 */
function calculateObjectDiff(oldObj, newObj) {
  const added = {};
  const removed = {};
  const updated = {};
  if (!newObj) return {
    added,
    removed,
    updated
  };
  function diffRecursive(base, compare, addedMap, removedMap, updatedMap) {
    for (const key in compare) {
      if (!(key in base)) {
        addedMap[key] = compare[key];
      } else if ((0, _lodash.isPlainObject)(base[key]) && (0, _lodash.isPlainObject)(compare[key])) {
        addedMap[key] = {};
        removedMap[key] = {};
        updatedMap[key] = {};
        diffRecursive(base[key], compare[key], addedMap[key], removedMap[key], updatedMap[key]);
        if ((0, _lodash.isEmpty)(addedMap[key])) delete addedMap[key];
        if ((0, _lodash.isEmpty)(removedMap[key])) delete removedMap[key];
      } else if (Array.isArray(base[key]) && Array.isArray(compare[key])) {
        addedMap[key] = [];
        removedMap[key] = [];
        updatedMap[key] = [];
        diffRecursive(base[key], compare[key], addedMap[key], removedMap[key], updatedMap[key]);
        if (isAllUndefined(addedMap[key])) delete addedMap[key];
        if (isAllUndefined(removedMap[key])) delete removedMap[key];
        if (isAllUndefined(updatedMap[key])) delete updatedMap[key];
      } else if (base[key] !== compare[key]) {
        updatedMap[key] = compare[key];
      }
    }
    for (const key in base) {
      if (!(key in compare)) {
        removedMap[key] = base[key];
      }
    }
  }
  diffRecursive(oldObj, newObj, added, removed, updated);
  return {
    added,
    removed,
    updated
  };
}