"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.computeLocationExtends = computeLocationExtends;
exports.createAstBaseItem = createAstBaseItem;
exports.createBinaryExpression = void 0;
exports.createColumn = createColumn;
exports.createColumnStar = createColumnStar;
exports.createCommand = void 0;
exports.createError = createError;
exports.createFakeMultiplyLiteral = createFakeMultiplyLiteral;
exports.createFunction = createFunction;
exports.createList = exports.createInlineCast = exports.createIdentifierOrParam = exports.createIdentifier = exports.createFunctionCall = void 0;
exports.createLiteral = createLiteral;
exports.createLiteralString = createLiteralString;
exports.createNumericLiteral = void 0;
exports.createOption = createOption;
exports.createParserFieldsFromTerminalNode = exports.createParserFields = exports.createParam = void 0;
exports.createTimeUnit = createTimeUnit;
exports.createUnknownItem = createUnknownItem;
exports.nonNullable = nonNullable;
exports.sanitizeIdentifierString = sanitizeIdentifierString;
exports.textExistsAndIsValid = textExistsAndIsValid;
exports.visitSource = visitSource;
exports.wrapIdentifierAsArray = wrapIdentifierAsArray;
var _esql_parser = require("../antlr/esql_parser");
var _builder = require("../builder");
var _pretty_print = require("../pretty_print");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * In case of changes in the grammar, this script should be updated: esql_update_ast_script.js
 */

function nonNullable(v) {
  return v != null;
}
function createAstBaseItem(name, ctx) {
  return {
    name,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
const createParserFields = ctx => ({
  text: ctx.getText(),
  location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
  incomplete: Boolean(ctx.exception)
});
exports.createParserFields = createParserFields;
const createParserFieldsFromTerminalNode = node => {
  const text = node.getText();
  const symbol = node.symbol;
  const fields = {
    text,
    location: (0, _helpers.getPosition)(symbol, symbol),
    incomplete: false
  };
  return fields;
};
exports.createParserFieldsFromTerminalNode = createParserFieldsFromTerminalNode;
const createCommand = (name, ctx, partial) => {
  const command = _builder.Builder.command({
    name,
    args: []
  }, createParserFields(ctx));
  if (partial) {
    Object.assign(command, partial);
  }
  return command;
};
exports.createCommand = createCommand;
const createInlineCast = (ctx, value) => _builder.Builder.expression.inlineCast({
  castType: ctx.dataType().getText().toLowerCase(),
  value
}, createParserFields(ctx));
exports.createInlineCast = createInlineCast;
const createList = (ctx, values) => _builder.Builder.expression.list.literal({
  values
}, createParserFields(ctx));
exports.createList = createList;
const createNumericLiteral = (ctx, literalType) => _builder.Builder.expression.literal.numeric({
  value: Number(ctx.getText()),
  literalType
}, createParserFields(ctx));
exports.createNumericLiteral = createNumericLiteral;
function createFakeMultiplyLiteral(ctx, literalType) {
  return {
    type: 'literal',
    literalType,
    text: ctx.getText(),
    name: ctx.getText(),
    value: ctx.PLUS() ? 1 : -1,
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createLiteralString(ctx) {
  var _ctx$QUOTED_STRING$ge, _ctx$QUOTED_STRING;
  const quotedString = (_ctx$QUOTED_STRING$ge = (_ctx$QUOTED_STRING = ctx.QUOTED_STRING()) === null || _ctx$QUOTED_STRING === void 0 ? void 0 : _ctx$QUOTED_STRING.getText()) !== null && _ctx$QUOTED_STRING$ge !== void 0 ? _ctx$QUOTED_STRING$ge : '""';
  const isTripleQuoted = quotedString.startsWith('"""') && quotedString.endsWith('"""');
  let valueUnquoted = isTripleQuoted ? quotedString.slice(3, -3) : quotedString.slice(1, -1);
  if (!isTripleQuoted) {
    valueUnquoted = valueUnquoted.replace(/\\"/g, '"').replace(/\\r/g, '\r').replace(/\\n/g, '\n').replace(/\\t/g, '\t').replace(/\\\\/g, '\\');
  }
  return _builder.Builder.expression.literal.string(valueUnquoted, {
    name: quotedString
  }, createParserFields(ctx));
}
function isMissingText(text) {
  return /<missing /.test(text);
}
function textExistsAndIsValid(text) {
  return !!(text && !isMissingText(text));
}
function createLiteral(type, node) {
  if (!node) {
    return {
      type: 'literal',
      name: 'unknown',
      text: 'unknown',
      value: 'unknown',
      literalType: type,
      location: {
        min: 0,
        max: 0
      },
      incomplete: false
    };
  }
  const text = node.getText();
  const partialLiteral = {
    type: 'literal',
    text,
    name: text,
    location: (0, _helpers.getPosition)(node.symbol),
    incomplete: isMissingText(text)
  };
  if (type === 'double' || type === 'integer') {
    return {
      ...partialLiteral,
      literalType: type,
      value: Number(text),
      paramType: 'number'
    };
  } else if (type === 'param') {
    throw new Error('Should never happen');
  }
  return {
    ...partialLiteral,
    literalType: type,
    value: text
  };
}
function createTimeUnit(ctx) {
  return {
    type: 'timeInterval',
    quantity: Number(ctx.integerValue().INTEGER_LITERAL().getText()),
    unit: ctx.UNQUOTED_IDENTIFIER().symbol.text,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    name: `${ctx.integerValue().INTEGER_LITERAL().getText()} ${ctx.UNQUOTED_IDENTIFIER().symbol.text}`,
    incomplete: Boolean(ctx.exception)
  };
}
function createFunction(name, ctx, customPosition, subtype, args = [], incomplete) {
  const node = {
    type: 'function',
    name,
    text: ctx.getText(),
    location: customPosition !== null && customPosition !== void 0 ? customPosition : (0, _helpers.getPosition)(ctx.start, ctx.stop),
    args,
    incomplete: Boolean(ctx.exception) || !!incomplete
  };
  if (subtype) {
    node.subtype = subtype;
  }
  return node;
}
const createFunctionCall = ctx => {
  const functionExpressionCtx = ctx.functionExpression();
  const functionName = functionExpressionCtx.functionName();
  const node = {
    type: 'function',
    subtype: 'variadic-call',
    name: functionName.getText().toLowerCase(),
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception)
  };
  const identifierOrParameter = functionName.identifierOrParameter();
  if (identifierOrParameter instanceof _esql_parser.IdentifierOrParameterContext) {
    const operator = createIdentifierOrParam(identifierOrParameter);
    if (operator) {
      node.operator = operator;
    }
  }
  return node;
};
exports.createFunctionCall = createFunctionCall;
const createBinaryExpression = (operator, ctx, args) => {
  const node = _builder.Builder.expression.func.binary(operator, args, {}, {
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  });
  return node;
};
exports.createBinaryExpression = createBinaryExpression;
const createIdentifierOrParam = ctx => {
  var _ctx$parameter;
  const identifier = ctx.identifier();
  if (identifier) {
    return createIdentifier(identifier);
  }
  const parameter = (_ctx$parameter = ctx.parameter()) !== null && _ctx$parameter !== void 0 ? _ctx$parameter : ctx.doubleParameter();
  if (parameter) {
    return createParam(parameter);
  }
};
exports.createIdentifierOrParam = createIdentifierOrParam;
const createIdentifier = identifier => {
  const text = identifier.getText();
  const name = (0, _helpers.parseIdentifier)(text);
  return _builder.Builder.identifier({
    name
  }, createParserFields(identifier));
};
exports.createIdentifier = createIdentifier;
const createParam = ctx => {
  if (ctx instanceof _esql_parser.InputParamContext || ctx instanceof _esql_parser.InputDoubleParamsContext) {
    const isDoubleParam = ctx instanceof _esql_parser.InputDoubleParamsContext;
    const paramKind = isDoubleParam ? '??' : '?';
    return _builder.Builder.param.unnamed(createParserFields(ctx), {
      paramKind
    });
  } else if (ctx instanceof _esql_parser.InputNamedOrPositionalParamContext || ctx instanceof _esql_parser.InputNamedOrPositionalDoubleParamsContext) {
    const isDoubleParam = ctx instanceof _esql_parser.InputNamedOrPositionalDoubleParamsContext;
    const paramKind = isDoubleParam ? '??' : '?';
    const text = ctx.getText();
    const value = text.slice(isDoubleParam ? 2 : 1);
    const valueAsNumber = Number(value);
    const isPositional = String(valueAsNumber) === value;
    const parserFields = createParserFields(ctx);
    if (isPositional) {
      return _builder.Builder.param.positional({
        paramKind,
        value: valueAsNumber
      }, parserFields);
    } else {
      return _builder.Builder.param.named({
        paramKind,
        value
      }, parserFields);
    }
  }
};
exports.createParam = createParam;
function walkFunctionStructure(args, initialLocation, prop, getNextItemIndex) {
  let nextArg = args[getNextItemIndex(args)];
  const location = {
    ...initialLocation
  };
  while (Array.isArray(nextArg) || nextArg) {
    if (Array.isArray(nextArg)) {
      nextArg = nextArg[getNextItemIndex(nextArg)];
    } else {
      location[prop] = Math[prop](location[prop], nextArg.location[prop]);
      if (nextArg.type === 'function') {
        nextArg = nextArg.args[getNextItemIndex(nextArg.args)];
      } else {
        nextArg = undefined;
      }
    }
  }
  return location[prop];
}
function computeLocationExtends(fn) {
  const location = fn.location;
  if (fn.args) {
    // get min location navigating in depth keeping the left/first arg
    location.min = walkFunctionStructure(fn.args, location, 'min', () => 0);
    // get max location navigating in depth keeping the right/last arg
    location.max = walkFunctionStructure(fn.args, location, 'max', args => args.length - 1);
    // in case of empty array as last arg, bump the max location by 3 chars (empty brackets)
    if (Array.isArray(fn.args[fn.args.length - 1]) && !fn.args[fn.args.length - 1].length) {
      location.max += 3;
    }
  }
  return location;
}

// Note: do not import esql_parser or bundle size will grow up by ~500 kb
/**
 * Do not touch this piece of code as it is auto-generated by a script
 */

/* SCRIPT_MARKER_START */
function getQuotedText(ctx) {
  return [27 /* esql_parser.QUOTED_STRING */, 69 /* esql_parser.QUOTED_IDENTIFIER */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
function getUnquotedText(ctx) {
  return [68 /* esql_parser.UNQUOTED_IDENTIFIER */, 77 /* esql_parser.UNQUOTED_SOURCE */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
/* SCRIPT_MARKER_END */

function isQuoted(text) {
  return text && /^(`)/.test(text);
}

/**
 * Follow a similar logic to the ES one:
 * * remove backticks at the beginning and at the end
 * * remove double backticks
 */
function safeBackticksRemoval(text) {
  return (text === null || text === void 0 ? void 0 : text.replace(_constants.TICKS_REGEX, '').replace(_constants.DOUBLE_TICKS_REGEX, _constants.SINGLE_BACKTICK)) || '';
}
function sanitizeSourceString(ctx) {
  const contextText = ctx.getText();
  // If wrapped by triple quote, remove
  if (contextText.startsWith(`"""`) && contextText.endsWith(`"""`)) {
    return contextText.replace(/\"\"\"/g, '');
  }
  // If wrapped by single quote, remove
  if (contextText.startsWith(`"`) && contextText.endsWith(`"`)) {
    return contextText.slice(1, -1);
  }
  return contextText;
}
function sanitizeIdentifierString(ctx) {
  var _getUnquotedText, _getQuotedText;
  const result = ((_getUnquotedText = getUnquotedText(ctx)) === null || _getUnquotedText === void 0 ? void 0 : _getUnquotedText.getText()) || safeBackticksRemoval((_getQuotedText = getQuotedText(ctx)) === null || _getQuotedText === void 0 ? void 0 : _getQuotedText.getText()) || safeBackticksRemoval(ctx.getText()); // for some reason some quoted text is not detected correctly by the parser
  // TODO - understand why <missing null> is now returned as the match text for the FROM command
  return result === '<missing null>' ? '' : result;
}
function wrapIdentifierAsArray(identifierCtx) {
  return Array.isArray(identifierCtx) ? identifierCtx : [identifierCtx];
}
const visitQuotedString = ctx => {
  const unquotedCtx = ctx.UNQUOTED_SOURCE();
  const valueUnquoted = unquotedCtx.getText();
  const quotedString = _pretty_print.LeafPrinter.string({
    valueUnquoted
  });
  return _builder.Builder.expression.literal.string(valueUnquoted, {
    name: quotedString,
    unquoted: true
  }, createParserFieldsFromTerminalNode(unquotedCtx));
};
const visitUnquotedOrQuotedString = ctx => {
  const unquotedCtx = ctx.UNQUOTED_SOURCE();
  if (unquotedCtx) {
    const valueUnquoted = unquotedCtx.getText();
    const quotedString = _pretty_print.LeafPrinter.string({
      valueUnquoted
    });
    return _builder.Builder.expression.literal.string(valueUnquoted, {
      name: quotedString,
      unquoted: true
    }, createParserFieldsFromTerminalNode(unquotedCtx));
  }
  return createLiteralString(ctx);
};
function visitSource(ctx, type = 'index') {
  const text = sanitizeSourceString(ctx);
  let prefix;
  let index;
  let selector;
  if (ctx instanceof _esql_parser.IndexPatternContext) {
    const clusterStringCtx = ctx.clusterString();
    const unquotedIndexString = ctx.unquotedIndexString();
    const indexStringCtx = ctx.indexString();
    const selectorStringCtx = ctx.selectorString();
    if (clusterStringCtx) {
      prefix = visitQuotedString(clusterStringCtx);
    }
    if (unquotedIndexString) {
      index = visitQuotedString(unquotedIndexString);
    }
    if (indexStringCtx) {
      index = visitUnquotedOrQuotedString(indexStringCtx);
    }
    if (selectorStringCtx) {
      selector = visitQuotedString(selectorStringCtx);
    }
  }
  return _builder.Builder.expression.source.node({
    sourceType: type,
    prefix,
    index,
    selector,
    name: text
  }, {
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === ''),
    text: ctx === null || ctx === void 0 ? void 0 : ctx.getText()
  });
}
function createColumnStar(ctx) {
  const text = ctx.getText();
  const parserFields = {
    text,
    location: (0, _helpers.getPosition)(ctx.symbol),
    incomplete: ctx.getText() === '',
    quoted: false
  };
  const node = _builder.Builder.expression.column({
    args: [_builder.Builder.identifier({
      name: '*'
    }, parserFields)]
  }, parserFields);
  node.name = text;
  return node;
}
function createColumn(ctx) {
  const args = [];
  if (ctx instanceof _esql_parser.QualifiedNamePatternContext) {
    const list = ctx.identifierPattern_list();
    for (const identifierPattern of list) {
      const ID_PATTERN = identifierPattern.ID_PATTERN();
      if (ID_PATTERN) {
        const name = (0, _helpers.parseIdentifier)(ID_PATTERN.getText());
        const node = _builder.Builder.identifier({
          name
        }, createParserFields(identifierPattern));
        args.push(node);
      } else {
        const parameter = createParam(identifierPattern.parameter());
        if (parameter) {
          args.push(parameter);
        }
      }
    }
  } else if (ctx instanceof _esql_parser.QualifiedNameContext) {
    const list = ctx.identifierOrParameter_list();
    for (const item of list) {
      if (item instanceof _esql_parser.IdentifierOrParameterContext) {
        const node = createIdentifierOrParam(item);
        if (node) {
          args.push(node);
        }
      }
    }
  } else {
    const name = sanitizeIdentifierString(ctx);
    const node = _builder.Builder.identifier({
      name
    }, createParserFields(ctx));
    args.push(node);
  }
  const text = sanitizeIdentifierString(ctx);
  const hasQuotes = Boolean(getQuotedText(ctx) || isQuoted(ctx.getText()));
  const column = _builder.Builder.expression.column({
    args
  }, {
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === '')
  });
  column.name = text;
  column.quoted = hasQuotes;
  return column;
}
function createOption(name, ctx) {
  var _ctx$children;
  return {
    type: 'option',
    name,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception || ((_ctx$children = ctx.children) === null || _ctx$children === void 0 ? void 0 : _ctx$children.some(c => {
      // @ts-expect-error not exposed in type but exists see https://github.com/antlr/antlr4/blob/v4.11.1/runtime/JavaScript/src/antlr4/tree/ErrorNodeImpl.js#L19
      return Boolean(c.isErrorNode);
    })))
  };
}
function createUnknownItem(ctx) {
  return {
    type: 'unknown',
    name: 'unknown',
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createError(exception) {
  const token = exception.offendingToken;
  return {
    type: 'error',
    text: `SyntaxError: ${exception.message}`,
    location: (0, _helpers.getPosition)(token)
  };
}