"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkAndLoadIntegration = checkAndLoadIntegration;
var _validate_custom_component_template = require("./validate_custom_component_template");
var _get_integrations = require("../../integrations/get_integrations");
var _component_template_name = require("../../../../common/utils/component_template_name");
var _services = require("../../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The function works on 2 conditions:
// 1. It checks if integration name is present in meta field response of the datastream.
// If yes, it considers it to be an integration. No further checks
// 2. If not, then it does the various checks
async function checkAndLoadIntegration({
  esClient,
  packageClient,
  logger,
  dataStream
}) {
  var _dataStreamInfo$_meta, _dataStreamInfo$_meta2, _dataStreamInfo$_meta3;
  const [dataStreamInfo] = await _services.dataStreamService.getMatchingDataStreams(esClient, dataStream);
  const indexTemplate = dataStreamInfo === null || dataStreamInfo === void 0 ? void 0 : dataStreamInfo.template;
  const isManaged = dataStreamInfo === null || dataStreamInfo === void 0 ? void 0 : (_dataStreamInfo$_meta = dataStreamInfo._meta) === null || _dataStreamInfo$_meta === void 0 ? void 0 : _dataStreamInfo$_meta.managed;
  const integrationNameFromDataStream = dataStreamInfo === null || dataStreamInfo === void 0 ? void 0 : (_dataStreamInfo$_meta2 = dataStreamInfo._meta) === null || _dataStreamInfo$_meta2 === void 0 ? void 0 : (_dataStreamInfo$_meta3 = _dataStreamInfo$_meta2.package) === null || _dataStreamInfo$_meta3 === void 0 ? void 0 : _dataStreamInfo$_meta3.name;

  // Index template must be present and isManaged should be true or
  // integration name should be present
  // Else it's not an integration
  if ((!indexTemplate || !isManaged) && !integrationNameFromDataStream) {
    return {
      isIntegration: false,
      areAssetsAvailable: false
    };
  }

  // If integration name is present, then we find and return the integration
  if (integrationNameFromDataStream) {
    try {
      const integrationDetailsMatchingDataStream = await (0, _get_integrations.getIntegration)({
        packageClient,
        logger,
        packageName: integrationNameFromDataStream
      });
      if (integrationDetailsMatchingDataStream) {
        return {
          isIntegration: true,
          integration: integrationDetailsMatchingDataStream,
          areAssetsAvailable: true
        };
      }
    } catch (e) {
      // This should ideally not happen. As integration name is present in Data stream
      // meta response but the integration itself is not found
      // Worst case i could think of is, may be the integration is deleted from the
      // system at a later point of time
      return {
        isIntegration: false,
        areAssetsAvailable: false
      };
    }
  }

  // cleaning the index template name as some have @template suffix
  const indexTemplateNameWithoutSuffix = (0, _component_template_name.getComponentTemplatePrefixFromIndexTemplate)(indexTemplate);

  // Check if index template name has both type and dataset part
  const isDedicatedComponentTemplate = indexTemplateNameWithoutSuffix.split('-').length === 2;

  // If only 1 part exists, then it's not a dedicated index template
  // Data stream name must starts with this index template, then it's a dedicated index template else not
  if (!isDedicatedComponentTemplate || !dataStream.startsWith(indexTemplateNameWithoutSuffix)) {
    return {
      isIntegration: false,
      areAssetsAvailable: false
    };
  }
  const isValidCustomComponentTemplate = await (0, _validate_custom_component_template.validateCustomComponentTemplate)({
    esClient,
    indexTemplateName: indexTemplate
  });
  if (!isValidCustomComponentTemplate) {
    return {
      isIntegration: false,
      areAssetsAvailable: false
    };
  }
  const datasetName = indexTemplateNameWithoutSuffix.split('-')[1];
  const allIntegrations = await (0, _get_integrations.getIntegrations)({
    packageClient,
    logger
  });
  const integrationFromDataset = allIntegrations.find(integration => {
    var _integration$datasets;
    return datasetName in ((_integration$datasets = integration === null || integration === void 0 ? void 0 : integration.datasets) !== null && _integration$datasets !== void 0 ? _integration$datasets : {});
  });
  if (integrationFromDataset) {
    return {
      isIntegration: true,
      integration: integrationFromDataset,
      areAssetsAvailable: true
    };
  }

  // Since the logic reached the last statement, it means it passed all checks for assets being available
  return {
    isIntegration: false,
    areAssetsAvailable: true
  };
}