"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServiceMap = getServiceMap;
var _lodash = require("lodash");
var _service_map = require("../../../common/service_map");
var _with_apm_span = require("../../utils/with_apm_span");
var _get_trace_sample_ids = require("./get_trace_sample_ids");
var _get_service_anomalies = require("./get_service_anomalies");
var _get_service_stats = require("./get_service_stats");
var _get_service_map_from_trace_ids = require("./get_service_map_from_trace_ids");
var _fetch_exit_span_samples = require("./fetch_exit_span_samples");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getConnectionData({
  config,
  apmEventClient,
  serviceName,
  environment,
  start,
  end,
  serviceGroupKuery,
  kuery,
  logger,
  useV2 = false
}) {
  return (0, _with_apm_span.withApmSpan)('get_service_map_connections', async () => {
    logger.debug('Getting trace sample IDs');
    const {
      traceIds
    } = await (0, _get_trace_sample_ids.getTraceSampleIds)({
      config,
      apmEventClient,
      serviceName,
      environment,
      start,
      end,
      serviceGroupKuery,
      kuery
    });
    logger.debug(`Found ${traceIds.length} traces to inspect`);
    if (useV2) {
      if (traceIds.length === 0) {
        return {
          spans: [],
          tracesCount: 0
        };
      }
      const spans = await (0, _with_apm_span.withApmSpan)('get_service_map_exit_spans_and_transactions_from_traces', () => (0, _fetch_exit_span_samples.fetchExitSpanSamplesFromTraceIds)({
        apmEventClient,
        traceIds,
        start,
        end
      }));
      return {
        tracesCount: traceIds.length,
        spans
      };
    }
    if (!traceIds.length) {
      return {
        connections: [],
        discoveredServices: [],
        tracesCount: 0
      };
    }
    const chunkedResponses = await (0, _with_apm_span.withApmSpan)('get_service_paths_from_all_trace_ids', () => {
      const chunks = (0, _lodash.chunk)(traceIds, config.serviceMapMaxTracesPerRequest);
      logger.debug(`Executing scripted metric agg (${chunks.length} chunks)`);
      return Promise.all(chunks.map(traceIdsChunk => (0, _get_service_map_from_trace_ids.getServiceMapFromTraceIds)({
        apmEventClient,
        traceIds: traceIdsChunk,
        start,
        end,
        terminateAfter: config.serviceMapTerminateAfter,
        serviceMapMaxAllowableBytes: config.serviceMapMaxAllowableBytes,
        numOfRequests: chunks.length,
        logger
      })));
    });
    logger.debug('Received chunk responses');
    const mergedResponses = chunkedResponses.reduce((prev, current) => {
      return {
        connections: prev.connections.concat(current.connections),
        discoveredServices: prev.discoveredServices.concat(current.discoveredServices)
      };
    });
    logger.debug('Merged responses');
    return {
      connections: mergedResponses.connections,
      discoveredServices: mergedResponses.discoveredServices,
      tracesCount: traceIds.length
    };
  });
}
function getServiceMap(options) {
  return (0, _with_apm_span.withApmSpan)('get_service_map', async () => {
    const {
      logger
    } = options;
    const anomaliesPromise = (0, _get_service_anomalies.getServiceAnomalies)(options

    // always catch error to avoid breaking service maps if there is a problem with ML
    ).catch(error => {
      logger.debug(`Unable to retrieve anomalies for service maps.`, {
        error
      });
      return _get_service_anomalies.DEFAULT_ANOMALIES;
    });
    const [connectionData, servicesData, anomalies] = await Promise.all([getConnectionData(options), (0, _get_service_stats.getServiceStats)(options), anomaliesPromise]);
    logger.debug('Received and parsed all responses');
    if ('spans' in connectionData) {
      return {
        spans: connectionData.spans,
        tracesCount: connectionData.tracesCount,
        servicesData,
        anomalies
      };
    }
    const serviceMapNodes = (0, _service_map.getServiceMapNodes)({
      connections: connectionData.connections,
      exitSpanDestinations: connectionData.discoveredServices,
      servicesData,
      anomalies
    });
    return {
      ...serviceMapNodes,
      tracesCount: connectionData.tracesCount,
      nodesCount: serviceMapNodes.nodesCount
    };
  });
}