"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TRANSIENT_NAVIGATION_WINDOW_MS = exports.KibanaErrorService = exports.DEFAULT_MAX_ERROR_DURATION_MS = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _telemetry_events = require("../../lib/telemetry_events");
var _throw_if_error = require("../ui/throw_if_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MATCH_CHUNK_LOAD_ERROR = /ChunkLoadError/;

// Maximum duration to track for error component rendering
const DEFAULT_MAX_ERROR_DURATION_MS = exports.DEFAULT_MAX_ERROR_DURATION_MS = 10 * 1000; // 10 seconds

// Time window to capture transient navigation after an error
// Navigation within this window suggests the error was transient and may not have been seen by users
const TRANSIENT_NAVIGATION_WINDOW_MS = exports.TRANSIENT_NAVIGATION_WINDOW_MS = 250;

// To keep track of errors that are enqueued for later reporting

/**
 * Kibana Error Boundary Services: Error Service
 * Each Error Boundary tracks an instance of this class
 * @internal
 */
class KibanaErrorService {
  constructor(deps) {
    (0, _defineProperty2.default)(this, "analytics", void 0);
    (0, _defineProperty2.default)(this, "enqueuedErrors", new Map());
    this.analytics = deps.analytics;
  }

  /**
   * Determines if the error fallback UI should appear as an apologetic but promising "Refresh" button,
   * or treated with "danger" coloring and include a detailed error message.
   */
  getIsFatal(error) {
    const isChunkLoadError = MATCH_CHUNK_LOAD_ERROR.test(error.name);
    return !isChunkLoadError; // "ChunkLoadError" is recoverable by refreshing the page
  }

  /**
   * Derive the name of the component that threw the error
   */
  getErrorComponentName(errorInfo) {
    var _errorInfo$componentS;
    let errorComponentName = null;
    const stackLines = errorInfo === null || errorInfo === void 0 ? void 0 : (_errorInfo$componentS = errorInfo.componentStack) === null || _errorInfo$componentS === void 0 ? void 0 : _errorInfo$componentS.split('\n');
    const errorIndicator = /^    at (\S+).*/;
    if (stackLines) {
      let i = 0;
      while (i < stackLines.length) {
        // scan the stack trace text
        if (stackLines[i].match(errorIndicator)) {
          // extract the name of the bad component
          errorComponentName = stackLines[i].replace(errorIndicator, '$1');
          // If the component is the utility for throwing errors, skip
          if (errorComponentName && errorComponentName !== _throw_if_error.ThrowIfError.name) {
            break;
          }
        }
        i++;
      }
    }
    return errorComponentName;
  }
  getCurrentPathname() {
    return typeof window !== 'undefined' ? window.location.pathname : '';
  }
  getAnalyticsReference() {
    return this.analytics;
  }

  /**
   * Enqueues an error to be reported later with timing and navigation information
   * @param error The error that was thrown
   * @param errorInfo React error info containing component stack
   * @returns An ID for the enqueued error that can be used to commit it later
   */
  enqueueError(error, errorInfo) {
    const isFatal = this.getIsFatal(error);
    const id = `${Date.now()}-${Math.random().toString(36).substring(2, 9)}`;

    // Create the enqueued error object
    const enqueuedError = {
      id,
      error,
      errorInfo,
      isFatal,
      name: this.getErrorComponentName(errorInfo),
      isReported: false,
      hasTransientNavigation: false,
      transientNavigationDetermined: false,
      initialPathname: this.getCurrentPathname(),
      enqueuedAt: Date.now() // Set the enqueued timestamp
    };
    this.enqueuedErrors.set(id, enqueuedError);

    // Set up single timer for transient navigation check
    if (isFatal) {
      enqueuedError.timeoutId = setTimeout(() => {
        this.handleTransientNavigationCheck(id);
      }, TRANSIENT_NAVIGATION_WINDOW_MS);
    }
    return enqueuedError;
  }

  /**
   * Handles the transient navigation check after TRANSIENT_NAVIGATION_WINDOW_MS
   * @private
   */
  handleTransientNavigationCheck(errorId) {
    const enqueuedError = this.enqueuedErrors.get(errorId);
    if (!enqueuedError || enqueuedError.isReported) {
      return;
    }

    // Check for transient navigation
    const currentPathname = this.getCurrentPathname();
    enqueuedError.hasTransientNavigation = enqueuedError.initialPathname !== currentPathname;
    enqueuedError.transientNavigationDetermined = true;

    // If external commit was already requested, commit immediately
    if (enqueuedError.committedAt) {
      this.reportError(enqueuedError);
    } else {
      // Otherwise, set up timer for the remaining duration until max timeout
      const remainingTime = DEFAULT_MAX_ERROR_DURATION_MS - TRANSIENT_NAVIGATION_WINDOW_MS;
      enqueuedError.timeoutId = setTimeout(() => {
        this.reportError(enqueuedError);
      }, remainingTime);
    }
  }

  /**
   * Commits an error, ensuring transient navigation has been determined
   * @param errorId The ID of the enqueued error
   * @returns The error object or null if not found or already committed
   */
  commitError(errorId) {
    var _enqueuedError$commit;
    const enqueuedError = this.enqueuedErrors.get(errorId);
    if (!enqueuedError || enqueuedError.isReported) {
      return null;
    }

    // Mark the commit timestamp
    enqueuedError.committedAt = (_enqueuedError$commit = enqueuedError.committedAt) !== null && _enqueuedError$commit !== void 0 ? _enqueuedError$commit : Date.now();

    // If transient navigation hasn't been determined yet, just mark the request and return null
    // The handleTransientNavigationCheck will call reportError when ready
    if (!enqueuedError.transientNavigationDetermined) {
      return null;
    }

    // Transient navigation is already determined, commit immediately
    return this.reportError(enqueuedError);
  }

  /**
   * Actually reports the error telemetry
   * @private
   */
  reportError(enqueuedError) {
    if (enqueuedError.isReported) {
      return {
        error: enqueuedError.error,
        errorInfo: enqueuedError.errorInfo,
        isFatal: enqueuedError.isFatal,
        name: enqueuedError.name
      };
    }

    // Mark the error as reported
    enqueuedError.isReported = true;
    if (enqueuedError.timeoutId) {
      // Mark the error as committed
      enqueuedError.timeoutId = undefined;
    }
    try {
      if (enqueuedError.isFatal && this.analytics) {
        var _enqueuedError$errorI, _enqueuedError$commit2;
        this.analytics.reportEvent(_telemetry_events.REACT_FATAL_ERROR_EVENT_TYPE, {
          component_name: enqueuedError.name,
          component_stack: ((_enqueuedError$errorI = enqueuedError.errorInfo) === null || _enqueuedError$errorI === void 0 ? void 0 : _enqueuedError$errorI.componentStack) || '',
          error_message: enqueuedError.error.toString(),
          error_stack: typeof enqueuedError.error.stack === 'string' ? enqueuedError.error.stack : '',
          component_render_min_duration_ms: ((_enqueuedError$commit2 = enqueuedError.committedAt) !== null && _enqueuedError$commit2 !== void 0 ? _enqueuedError$commit2 : Date.now()) - enqueuedError.enqueuedAt,
          has_transient_navigation: enqueuedError.hasTransientNavigation
        });
      }
    } catch (e) {
      // eslint-disable-next-line no-console
      console.error(e);
    }
    return {
      error: enqueuedError.error,
      errorInfo: enqueuedError.errorInfo,
      isFatal: enqueuedError.isFatal,
      name: enqueuedError.name
    };
  }
}
exports.KibanaErrorService = KibanaErrorService;