"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExceptionsPreMultiListFindHandler = void 0;
var _errors = require("../validators/errors");
var _validators = require("../validators");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getExceptionsPreMultiListFindHandler = endpointAppContextService => {
  return async function ({
    data,
    context: {
      request
    }
  }) {
    if (!data.namespaceType.includes('agnostic')) {
      return data;
    }
    let isEndpointArtifact = false;
    if (data.listId.some(id => _validators.TrustedAppValidator.isTrustedApp({
      listId: id
    }))) {
      // validate Trusted application
      isEndpointArtifact = true;
      await new _validators.TrustedAppValidator(endpointAppContextService, request).validatePreMultiListFind();
    } else if (data.listId.some(listId => _validators.HostIsolationExceptionsValidator.isHostIsolationException({
      listId
    }))) {
      // Validate Host Isolation Exceptions
      isEndpointArtifact = true;
      await new _validators.HostIsolationExceptionsValidator(endpointAppContextService, request).validatePreMultiListFind();
    } else if (data.listId.some(listId => _validators.EventFilterValidator.isEventFilter({
      listId
    }))) {
      // Event Filters Exceptions
      isEndpointArtifact = true;
      await new _validators.EventFilterValidator(endpointAppContextService, request).validatePreMultiListFind();
    } else if (data.listId.some(id => _validators.BlocklistValidator.isBlocklist({
      listId: id
    }))) {
      // validate Blocklist
      isEndpointArtifact = true;
      await new _validators.BlocklistValidator(endpointAppContextService, request).validatePreMultiListFind();
    } else if (data.listId.some(id => _validators.EndpointExceptionsValidator.isEndpointException({
      listId: id
    }))) {
      // Validate Endpoint Exceptions
      isEndpointArtifact = true;
      await new _validators.EndpointExceptionsValidator(endpointAppContextService, request).validatePreMultiListFind();
    }
    if (isEndpointArtifact) {
      if (!request) {
        throw new _errors.EndpointArtifactExceptionValidationError(`Missing HTTP Request object`);
      }
      await (0, _utils.setFindRequestFilterScopeToActiveSpace)(endpointAppContextService, request, data);
    }
    return data;
  };
};
exports.getExceptionsPreMultiListFindHandler = getExceptionsPreMultiListFindHandler;