"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startPrivilegeMonitoringTask = exports.scheduleNow = exports.removePrivilegeMonitoringTask = exports.registerPrivilegeMonitoringTask = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../constants");
var _state = require("./state");
var _api_key = require("../auth/api_key");
var _privilege_monitoring_data_client = require("../privilege_monitoring_data_client");
var _helpers = require("../../risk_score/tasks/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTaskName = () => _constants.TYPE;
const getTaskId = namespace => `${_constants.TYPE}:${namespace}:${_constants.VERSION}`;
const registerPrivilegeMonitoringTask = ({
  getStartServices,
  logger,
  auditLogger,
  telemetry,
  taskManager,
  kibanaVersion,
  experimentalFeatures
}) => {
  if (!taskManager) {
    logger.info('[Privilege Monitoring]  Task Manager is unavailable; skipping privilege monitoring task registration.');
    return;
  }
  const getPrivilegedUserMonitoringDataClient = async namespace => {
    const [core, {
      taskManager: taskManagerStart,
      security,
      encryptedSavedObjects
    }] = await getStartServices();
    const apiKeyManager = (0, _api_key.getApiKeyManager)({
      core,
      logger,
      security,
      encryptedSavedObjects,
      namespace
    });
    const client = await apiKeyManager.getClient();
    if (!client) {
      logger.error('[Privilege Monitoring] Unable to create Elasticsearch client from API key.');
      return undefined;
    }
    const soClient = (0, _helpers.buildScopedInternalSavedObjectsClientUnsafe)({
      coreStart: core,
      namespace
    });
    return new _privilege_monitoring_data_client.PrivilegeMonitoringDataClient({
      logger,
      clusterClient: client.clusterClient,
      namespace,
      soClient,
      taskManager: taskManagerStart,
      auditLogger,
      kibanaVersion,
      telemetry,
      apiKeyManager
    });
  };
  taskManager.registerTaskDefinitions({
    [getTaskName()]: {
      title: 'Entity Analytics Privilege Monitoring',
      timeout: _constants.TIMEOUT,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: createPrivilegeMonitoringTaskRunnerFactory({
        logger,
        telemetry,
        experimentalFeatures,
        getPrivilegedUserMonitoringDataClient
      })
    }
  });
};
exports.registerPrivilegeMonitoringTask = registerPrivilegeMonitoringTask;
const createPrivilegeMonitoringTaskRunnerFactory = deps => ({
  taskInstance
}) => {
  let cancelled = false;
  const isCancelled = () => cancelled;
  return {
    run: async () => runPrivilegeMonitoringTask({
      isCancelled,
      logger: deps.logger,
      telemetry: deps.telemetry,
      taskInstance,
      experimentalFeatures: deps.experimentalFeatures,
      getPrivilegedUserMonitoringDataClient: deps.getPrivilegedUserMonitoringDataClient
    }),
    cancel: async () => {
      cancelled = true;
    }
  };
};
const runPrivilegeMonitoringTask = async ({
  isCancelled,
  logger,
  telemetry,
  taskInstance,
  experimentalFeatures,
  getPrivilegedUserMonitoringDataClient
}) => {
  const state = taskInstance.state;
  const taskStartTime = (0, _moment.default)().utc().toISOString();
  const updatedState = {
    lastExecutionTimestamp: taskStartTime,
    namespace: state.namespace,
    runs: state.runs + 1
  };
  if (isCancelled()) {
    logger.info('[Privilege Monitoring] Task was cancelled.');
    return {
      state: updatedState
    };
  }
  try {
    logger.info('[Privilege Monitoring] Running privilege monitoring task');
    const dataClient = await getPrivilegedUserMonitoringDataClient(state.namespace);
    if (!dataClient) {
      logger.error('[Privilege Monitoring] error creating data client.');
      throw Error('No data client was found');
    }
    await dataClient.plainIndexSync();
  } catch (e) {
    logger.error(`[Privilege Monitoring] Error running privilege monitoring task: ${e.message}`);
  }
  logger.info('[Privilege Monitoring] Finished running privilege monitoring task');
  return {
    state: updatedState
  };
};
const startPrivilegeMonitoringTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  try {
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: getTaskName(),
      scope: _constants.SCOPE,
      schedule: {
        interval: _constants.INTERVAL
      },
      state: {
        ..._state.defaultState,
        namespace
      },
      params: {
        version: _constants.VERSION
      }
    });
    logger.info(`Scheduling privilege monitoring task with id ${taskId}`);
  } catch (e) {
    logger.warn(`[Privilege Monitoring]  [task ${taskId}]: error scheduling task, received ${e.message}`);
    throw e;
  }
};
exports.startPrivilegeMonitoringTask = startPrivilegeMonitoringTask;
const removePrivilegeMonitoringTask = async ({
  taskManager,
  namespace,
  logger
}) => {
  const taskId = getTaskId(namespace);
  try {
    await taskManager.removeIfExists(taskId);
    logger.info(`Removed privilege monitoring task with id ${taskId}`);
  } catch (e) {
    logger.warn(`[Privilege Monitoring]  [task ${taskId}]: error removing task, received ${e.message}`);
    throw e;
  }
};
exports.removePrivilegeMonitoringTask = removePrivilegeMonitoringTask;
const scheduleNow = async ({
  logger,
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  logger.info('[Privilege Monitoring] Attempting to schedule task to run now');
  try {
    return taskManager.runSoon(taskId);
  } catch (e) {
    logger.warn(`[task ${taskId}]: error scheduling task now, received ${e.message}`);
    throw e;
  }
};
exports.scheduleNow = scheduleNow;