"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upgradePrebuiltRule = void 0;
var _convert_alerting_rule_to_rule_response = require("../converters/convert_alerting_rule_to_rule_response");
var _convert_rule_response_to_alerting_rule = require("../converters/convert_rule_response_to_alerting_rule");
var _apply_rule_update = require("../mergers/apply_rule_update");
var _utils = require("../utils");
var _create_rule = require("./create_rule");
var _get_rule_by_rule_id = require("./get_rule_by_rule_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const upgradePrebuiltRule = async ({
  actionsClient,
  rulesClient,
  ruleAsset,
  mlAuthz,
  prebuiltRuleAssetClient
}) => {
  await (0, _utils.validateMlAuth)(mlAuthz, ruleAsset.type);
  const existingRule = await (0, _get_rule_by_rule_id.getRuleByRuleId)({
    rulesClient,
    ruleId: ruleAsset.rule_id
  });
  if (!existingRule) {
    throw new _utils.ClientError(`Failed to find rule ${ruleAsset.rule_id}`, 500);
  }
  if (ruleAsset.type !== existingRule.type) {
    // If we're trying to change the type of a prepackaged rule, we need to delete the old one
    // and replace it with the new rule, keeping the enabled setting, actions, throttle, id,
    // and exception lists from the old rule
    await rulesClient.delete({
      id: existingRule.id
    });
    const createdRule = await (0, _create_rule.createRule)({
      actionsClient,
      rulesClient,
      mlAuthz,
      rule: {
        ...ruleAsset,
        immutable: true,
        enabled: existingRule.enabled,
        exceptions_list: existingRule.exceptions_list,
        actions: existingRule.actions,
        timeline_id: existingRule.timeline_id,
        timeline_title: existingRule.timeline_title
      },
      id: existingRule.id
    });
    return createdRule;
  }

  // Else, recreate the rule from scratch with the passed payload.
  const updatedRule = await (0, _apply_rule_update.applyRuleUpdate)({
    prebuiltRuleAssetClient,
    existingRule,
    ruleUpdate: ruleAsset
  });

  // We want to preserve existing actions from existing rule on upgrade
  if (existingRule.actions.length) {
    updatedRule.actions = existingRule.actions;
  }
  const updatedRuleWithMergedExceptions = (0, _utils.mergeExceptionLists)(updatedRule, existingRule);
  const updatedInternalRule = await rulesClient.update({
    id: existingRule.id,
    data: (0, _convert_rule_response_to_alerting_rule.convertRuleResponseToAlertingRule)(updatedRuleWithMergedExceptions, actionsClient)
  });
  return (0, _convert_alerting_rule_to_rule_response.convertAlertingRuleToRuleResponse)(updatedInternalRule);
};
exports.upgradePrebuiltRule = upgradePrebuiltRule;