"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CompleteExternalActionsTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _task = require("@kbn/task-manager-plugin/server/task");
var _errors = require("../../services/actions/clients/errors");
var _utils = require("../../utils");
var _stringify = require("../../utils/stringify");
var _constants = require("../../../../common/endpoint/service/response_actions/constants");
var _queue_processor = require("../../utils/queue_processor");
var _constants2 = require("../../../../common/endpoint/constants");
var _complete_external_actions_task = require("./complete_external_actions_task");
var _fetch_space_ids_with_maybe_pending_actions = require("../../services/actions/utils/fetch_space_ids_with_maybe_pending_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A task manager runner responsible for checking the status of and completing pending actions
 * that were sent to 3rd party EDR systems.
 */
class CompleteExternalActionsTaskRunner {
  constructor(endpointContextServices, esClient, nextRunInterval = '60s', taskInstanceId, taskType) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "updatesQueue", void 0);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "errors", []);
    this.endpointContextServices = endpointContextServices;
    this.esClient = esClient;
    this.nextRunInterval = nextRunInterval;
    this.taskInstanceId = taskInstanceId;
    this.taskType = taskType;
    this.log = this.endpointContextServices.createLogger(
    // Adding a unique identifier to the end of the class name to help identify log entries related to this run
    `${this.constructor.name}.${Math.random().toString(32).substring(2, 8)}`);
    this.updatesQueue = new _queue_processor.QueueProcessor({
      batchHandler: this.queueBatchProcessor.bind(this),
      batchSize: 50,
      logger: this.log
    });
  }
  get taskId() {
    return `${_complete_external_actions_task.COMPLETE_EXTERNAL_RESPONSE_ACTIONS_TASK_TYPE}-${_complete_external_actions_task.COMPLETE_EXTERNAL_RESPONSE_ACTIONS_TASK_VERSION}`;
  }
  async queueBatchProcessor({
    batch,
    data
  }) {
    const operations = [];
    for (const actionResponseDoc of data) {
      operations.push({
        create: {
          _index: _constants2.ENDPOINT_ACTION_RESPONSES_INDEX
        }
      }, actionResponseDoc);
    }
    const bulkResponse = await this.esClient.bulk({
      index: _constants2.ENDPOINT_ACTION_RESPONSES_INDEX,
      operations
    }).catch(_utils.catchAndWrapError);
    if (bulkResponse.errors) {
      this.errors.push(`Batch [${batch}] processing of [${data.length}] items generated the following errors:\n${(0, _stringify.stringify)(bulkResponse)}`);
    }
  }
  getNextRunDate() {
    const nextRun = new Date();
    const nextRunInterval = this.nextRunInterval;
    if (nextRunInterval.endsWith('s')) {
      const seconds = parseInt(nextRunInterval.slice(0, -1), 10);
      nextRun.setSeconds(nextRun.getSeconds() + seconds);
    } else if (nextRunInterval.endsWith('m')) {
      const minutes = parseInt(nextRunInterval.slice(0, -1), 10);
      nextRun.setMinutes(nextRun.getMinutes() + minutes);
    } else {
      this.log.error(`Invalid task interval: ${nextRunInterval}`);
      return;
    }
    return nextRun;
  }
  async run() {
    if (this.taskInstanceId !== this.taskId) {
      this.log.info(`Outdated task version. Got [${this.taskInstanceId}] from task instance. Current version is [${this.taskId}]`);
      return (0, _task.getDeleteTaskRunResult)();
    }
    this.log.debug(`Started: Checking status of external response actions. Task Type:[${this.taskType}] Task Id:[${this.taskId}]`);
    this.abortController = new AbortController();

    // If license is not `enterprise`, then exit. Support for external response actions is a
    // Enterprise level feature.
    if (!this.endpointContextServices.getLicenseService().isEnterprise()) {
      this.abortController.abort(`License not Enterprise!`);
      this.log.debug(`Exiting: Run aborted due to license not being Enterprise`);
      return;
    }
    const allPendingActionsProcessing = [];

    // Collect update needed to complete response actions
    for (const agentType of _constants.RESPONSE_ACTION_AGENT_TYPE) {
      if (agentType !== 'endpoint') {
        // If run was aborted, then stop looping through
        if (this.abortController.signal.aborted) {
          break;
        }
        const spaceIds = await (0, _fetch_space_ids_with_maybe_pending_actions.fetchSpaceIdsWithMaybePendingActions)(this.endpointContextServices, agentType);
        if (this.abortController.signal.aborted) {
          break;
        }
        for (const spaceId of spaceIds) {
          this.log.debug(() => `Processing pending actions for [${agentType}] in space [${spaceId}]`);
          const agentTypeActionsClient = this.endpointContextServices.getInternalResponseActionsClient({
            agentType,
            spaceId,
            taskType: this.taskType,
            taskId: this.taskInstanceId
          });
          allPendingActionsProcessing.push(agentTypeActionsClient.processPendingActions({
            abortSignal: this.abortController.signal,
            addToQueue: this.updatesQueue.addToQueue.bind(this.updatesQueue)
          }).catch(err => {
            // ignore errors due to connector not being configured - no point in logging errors if a customer
            // is not using response actions for the given agent type
            if (err instanceof _errors.ResponseActionsConnectorNotConfiguredError) {
              this.log.debug(`Skipping agentType [${agentType}] for space [${spaceId}]: No stack connector configured for this agent type`);
              return;
            }
            this.errors.push(err.stack);
          }));
        }
      }
    }
    await Promise.all(allPendingActionsProcessing);
    await this.updatesQueue.complete();
    this.abortController.abort(`Run complete.`);
    if (this.errors.length) {
      this.log.error(`${this.errors.length} errors were encountered while running task:\n${this.errors.join('\n----')}`);
    }
    this.log.debug(`Completed: Checking status of external response actions`);
    return {
      state: {},
      runAt: this.getNextRunDate()
    };
  }
  async cancel() {
    if (!this.abortController.signal.aborted) {
      this.abortController.abort('Task runner canceled!');

      // Sleep 2 seconds to give an opportunity for the abort signal to be processed
      await new Promise(r => setTimeout(r, 2000));

      // Wait for remainder of updates to be written to ES
      await this.updatesQueue.complete();
    }
  }
}
exports.CompleteExternalActionsTaskRunner = CompleteExternalActionsTaskRunner;