"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignalsMigrationDeprecationsInfo = void 0;
var _i18n = require("@kbn/i18n");
var _constants = require("../../common/constants");
var _get_non_migrated_signals_info = require("../lib/detection_engine/migrations/get_non_migrated_signals_info");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const constructMigrationApiCall = (space, range) => `GET <kibana host>:<port>${space === 'default' ? '' : `/s/${space}`}${_constants.DETECTION_ENGINE_SIGNALS_MIGRATION_STATUS_URL}?from=${range}`;
const getSignalsMigrationDeprecationsInfo = async (ctx, config, logger, docLinks) => {
  const esClient = ctx.esClient.asInternalUser;
  const {
    isMigrationRequired,
    spaces
  } = await (0, _get_non_migrated_signals_info.getNonMigratedSignalsInfo)({
    esClient,
    signalsIndex: config.signalsIndex,
    logger
  });
  // Deprecation API requires time range to be part of request (https://www.elastic.co/guide/en/security/current/signals-migration-api.html#migration-1)
  // Return the earliest date, so it would capture the oldest possible signals
  const fromRange = new Date(0).toISOString();
  if (isMigrationRequired) {
    return [{
      deprecationType: 'feature',
      title: _i18n.i18n.translate('xpack.securitySolution.deprecations.signalsMigrationTitle', {
        defaultMessage: 'Found not migrated detection alerts'
      }),
      level: 'warning',
      message: _i18n.i18n.translate('xpack.securitySolution.deprecations.signalsMigrationMessage', {
        defaultMessage: `After upgrading Kibana, the latest Elastic Security features will be available for any newly generated detection alerts. However, in order to enable new features for existing detection alerts, migration may be necessary.`
      }),
      documentationUrl: docLinks.links.securitySolution.signalsMigrationApi,
      correctiveActions: {
        manualSteps: [_i18n.i18n.translate('xpack.securitySolution.deprecations.migrateIndexIlmPolicy.signalsMigrationManualStepOne', {
          defaultMessage: `Visit "Learn more" link for instructions how to migrate detection alerts. Migrate indices for each space.`
        }), _i18n.i18n.translate('xpack.securitySolution.deprecations.migrateIndexIlmPolicy.signalsMigrationManualStepTwo', {
          defaultMessage: 'Spaces with at least one non-migrated signals index: {spaces}.',
          values: {
            spaces: spaces.join(', ')
          }
        }), _i18n.i18n.translate('xpack.securitySolution.deprecations.migrateIndexIlmPolicy.signalsMigrationManualStepFour', {
          defaultMessage: 'Example of migration API calls:'
        }), ...spaces.map(space => constructMigrationApiCall(space, fromRange))]
      }
    }];
  }
  return [];
};
exports.getSignalsMigrationDeprecationsInfo = getSignalsMigrationDeprecationsInfo;