"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TITLE_CONVERSATION_FUNCTION_NAME = void 0;
exports.getGeneratedTitle = getGeneratedTitle;
exports.getTitleSystemMessage = void 0;
var _rxjs = require("rxjs");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TITLE_CONVERSATION_FUNCTION_NAME = exports.TITLE_CONVERSATION_FUNCTION_NAME = 'title_conversation';
const getTitleSystemMessage = scopes => `You are a helpful assistant for ${scopes.includes('observability') ? 'Elastic Observability' : 'Elasticsearch'}. Assume the following message is the start of a conversation between you and a user; give this conversation a title based on the content below. DO NOT UNDER ANY CIRCUMSTANCES wrap this title in single or double quotes. This title is shown in a list of conversations to the user, so title it for the user, not for you.`;
exports.getTitleSystemMessage = getTitleSystemMessage;
function getGeneratedTitle({
  messages,
  chat,
  logger,
  scopes
}) {
  return (0, _rxjs.from)(chat('generate_title', {
    systemMessage: getTitleSystemMessage(scopes),
    messages: [{
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common.MessageRole.User,
        content: messages.reduce((acc, curr) => {
          return `${acc} ${curr.message.role}: ${curr.message.content}`;
        }, 'Generate a title, using the title_conversation_function, based on the following conversation:\n\n')
      }
    }],
    functions: [{
      name: TITLE_CONVERSATION_FUNCTION_NAME,
      description: 'Use this function to title the conversation. Do not wrap the title in quotes',
      parameters: {
        type: 'object',
        properties: {
          title: {
            type: 'string'
          }
        },
        required: ['title']
      }
    }],
    functionCall: TITLE_CONVERSATION_FUNCTION_NAME,
    stream: false
  })).pipe((0, _rxjs.mergeMap)(response => {
    let title = (response.toolCalls[0].function.name ? response.toolCalls[0].function.arguments.title : response.content) || '';

    // This captures a string enclosed in single or double quotes.
    // It extracts the string content without the quotes.
    // Example matches:
    // - "Hello, World!" => Captures: Hello, World!
    // - 'Another Example' => Captures: Another Example
    // - JustTextWithoutQuotes => Captures: JustTextWithoutQuotes
    title = title.replace(/^"(.*)"$/g, '$1').replace(/^'(.*)'$/g, '$1');
    return (0, _rxjs.of)(title);
  }), (0, _rxjs.tap)(event => {
    if (typeof event === 'string') {
      logger.debug(`Generated title: ${event}`);
    }
  }), (0, _rxjs.catchError)(error => {
    logger.error(`Error generating title`);
    logger.error(error);
    // TODO: i18n
    return (0, _rxjs.of)('New conversation');
  }));
}