"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsSearchAPIRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _boom = require("@hapi/boom");
var _content_management = require("../../../../common/content_management");
var _constants = require("../../constants");
var _v = require("../../../content_management/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsSearchAPIRoute = (router, {
  contentManagement
}) => {
  const searchRoute = router.get({
    path: `${_constants.PUBLIC_API_PATH}/visualizations`,
    access: _constants.PUBLIC_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Search Lens visualizations',
    description: 'Get list of Lens visualizations.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  searchRoute.addVersion({
    version: _constants.PUBLIC_API_VERSION,
    validate: {
      request: {
        query: _configSchema.schema.object({
          query: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'The text to search for Lens visualizations'
            }
          })),
          page: _configSchema.schema.number({
            meta: {
              description: 'Specifies the current page number of the paginated result.'
            },
            min: 1,
            defaultValue: 1
          }),
          perPage: _configSchema.schema.number({
            meta: {
              description: 'Maximum number of Lens visualizations included in a single response'
            },
            defaultValue: 20,
            min: 1,
            max: 1000
          })
        })
      },
      response: {
        200: {
          body: () => _configSchema.schema.arrayOf(_v.lensSavedObjectSchema),
          description: 'Ok'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    let result;
    const {
      query,
      page,
      perPage: limit
    } = req.query;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _constants.PUBLIC_API_CONTENT_MANAGEMENT_VERSION);
    try {
      ({
        result
      } = await client.search({
        text: query,
        cursor: page.toString(),
        limit
      }, {
        searchFields: ['title', 'description']
      }));
    } catch (error) {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 403) {
        return res.forbidden();
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
    return res.ok({
      body: result.hits
    });
  });
};
exports.registerLensVisualizationsSearchAPIRoute = registerLensVisualizationsSearchAPIRoute;