"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaSavedObjectTypeMapping = void 0;
exports.createDefaultIndexPatterns = createDefaultIndexPatterns;
exports.createSavedObjectKibanaAsset = createSavedObjectKibanaAsset;
exports.deleteKibanaAssetsAndReferencesForSpace = deleteKibanaAssetsAndReferencesForSpace;
exports.installKibanaAssets = installKibanaAssets;
exports.installKibanaAssetsAndReferences = installKibanaAssetsAndReferences;
exports.installKibanaAssetsAndReferencesMultispace = installKibanaAssetsAndReferencesMultispace;
exports.installKibanaSavedObjects = installKibanaSavedObjects;
exports.installManagedIndexPattern = installManagedIndexPattern;
exports.isKibanaAssetType = void 0;
exports.toAssetReference = toAssetReference;
var _promises = require("timers/promises");
var _utils = require("@kbn/utils");
var _lodash = require("lodash");
var _archive = require("../../archive");
var _types = require("../../../../types");
var _install = require("../index_pattern/install");
var _install2 = require("../../packages/install");
var _remove = require("../../packages/remove");
var _errors = require("../../../../errors");
var _utils2 = require("../../packages/utils");
var _ = require("../../..");
var _tag_assets = require("./tag_assets");
var _saved_objects = require("./saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ASSETS_TO_INSTALL_IN_PARALLEL = 200;
const formatImportErrorsForLog = errors => JSON.stringify(errors.map(({
  type,
  id,
  error
}) => ({
  type,
  id,
  error
})) // discard other fields
);
const validKibanaAssetTypes = new Set(Object.values(_types.KibanaAssetType));
// KibanaSavedObjectTypes are used to ensure saved objects being created for a given
// KibanaAssetType have the correct type
const KibanaSavedObjectTypeMapping = exports.KibanaSavedObjectTypeMapping = {
  [_types.KibanaAssetType.dashboard]: _types.KibanaSavedObjectType.dashboard,
  [_types.KibanaAssetType.indexPattern]: _types.KibanaSavedObjectType.indexPattern,
  [_types.KibanaAssetType.map]: _types.KibanaSavedObjectType.map,
  [_types.KibanaAssetType.search]: _types.KibanaSavedObjectType.search,
  [_types.KibanaAssetType.visualization]: _types.KibanaSavedObjectType.visualization,
  [_types.KibanaAssetType.lens]: _types.KibanaSavedObjectType.lens,
  [_types.KibanaAssetType.mlModule]: _types.KibanaSavedObjectType.mlModule,
  [_types.KibanaAssetType.securityAIPrompt]: _types.KibanaSavedObjectType.securityAIPrompt,
  [_types.KibanaAssetType.securityRule]: _types.KibanaSavedObjectType.securityRule,
  [_types.KibanaAssetType.cloudSecurityPostureRuleTemplate]: _types.KibanaSavedObjectType.cloudSecurityPostureRuleTemplate,
  [_types.KibanaAssetType.tag]: _types.KibanaSavedObjectType.tag,
  [_types.KibanaAssetType.osqueryPackAsset]: _types.KibanaSavedObjectType.osqueryPackAsset,
  [_types.KibanaAssetType.osquerySavedQuery]: _types.KibanaSavedObjectType.osquerySavedQuery
};
const AssetFilters = {
  [_types.KibanaAssetType.indexPattern]: removeReservedIndexPatterns
};
function createSavedObjectKibanaAsset(asset) {
  // convert that to an object
  const so = {
    type: asset.type,
    id: asset.id,
    attributes: asset.attributes,
    references: asset.references || []
  };

  // migrating deprecated migrationVersion to typeMigrationVersion
  if (asset.migrationVersion && asset.migrationVersion[asset.type]) {
    so.typeMigrationVersion = asset.migrationVersion[asset.type];
  }
  if (asset.coreMigrationVersion) {
    so.coreMigrationVersion = asset.coreMigrationVersion;
  }
  if (asset.typeMigrationVersion) {
    so.typeMigrationVersion = asset.typeMigrationVersion;
  }
  return so;
}
async function installKibanaAssets(options) {
  const {
    kibanaAssetsArchiveIterator,
    savedObjectsClient,
    savedObjectsImporter,
    logger
  } = options;
  let assetsToInstall = [];
  let res = [];
  const installManagedIndexPatternOnce = (0, _lodash.once)(() => installManagedIndexPattern({
    savedObjectsClient,
    savedObjectsImporter
  }));
  async function flushAssetsToInstall() {
    await installManagedIndexPatternOnce();
    const installedAssets = await installKibanaSavedObjects({
      logger,
      savedObjectsImporter,
      kibanaAssets: assetsToInstall,
      assetsChunkSize: MAX_ASSETS_TO_INSTALL_IN_PARALLEL
    });
    assetsToInstall = [];
    res = [...res, ...installedAssets];
  }
  await kibanaAssetsArchiveIterator(async ({
    assetType,
    asset
  }) => {
    const assetFilter = AssetFilters[assetType];
    if (assetFilter) {
      assetsToInstall = [...assetsToInstall, ...assetFilter([asset])];
    } else {
      assetsToInstall.push(asset);
    }
    if (assetsToInstall.length >= MAX_ASSETS_TO_INSTALL_IN_PARALLEL) {
      await flushAssetsToInstall();
    }
  });
  if (assetsToInstall.length) {
    await flushAssetsToInstall();
  }
  return res;
}
async function installManagedIndexPattern({
  savedObjectsClient,
  savedObjectsImporter
}) {
  var _appContextService$ge;
  if (((_appContextService$ge = _.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.enableManagedLogsAndMetricsDataviews) === true) {
    await createDefaultIndexPatterns(savedObjectsImporter);
    await (0, _install.makeManagedIndexPatternsGlobal)(savedObjectsClient);
  }
}
async function createDefaultIndexPatterns(savedObjectsImporter) {
  // Create index patterns separately with `overwrite: false` to prevent blowing away users' runtime fields.
  // These don't get retried on conflict, because we expect that they exist once an integration has been installed.
  const indexPatternSavedObjects = (0, _install.getIndexPatternSavedObjects)();
  await savedObjectsImporter.import({
    overwrite: false,
    readStream: (0, _utils.createListStream)(indexPatternSavedObjects),
    createNewCopies: false,
    refresh: false,
    managed: true
  });
}
async function installKibanaAssetsAndReferencesMultispace({
  savedObjectsClient,
  logger,
  pkgName,
  pkgTitle,
  packageInstallContext,
  installedPkg,
  spaceId,
  assetTags,
  installAsAdditionalSpace
}) {
  if (installedPkg && !installAsAdditionalSpace) {
    // Install in every space => upgrades
    const refs = await installKibanaAssetsAndReferences({
      savedObjectsClient,
      logger,
      pkgName,
      pkgTitle,
      packageInstallContext,
      installedPkg,
      spaceId,
      assetTags,
      installAsAdditionalSpace
    });
    for (const additionnalSpaceId of Object.keys((_installedPkg$attribu = installedPkg.attributes.additional_spaces_installed_kibana) !== null && _installedPkg$attribu !== void 0 ? _installedPkg$attribu : {})) {
      var _installedPkg$attribu;
      await installKibanaAssetsAndReferences({
        savedObjectsClient,
        logger,
        pkgName,
        pkgTitle,
        packageInstallContext,
        installedPkg,
        spaceId: additionnalSpaceId,
        assetTags,
        installAsAdditionalSpace: true
      });
    }
    return refs;
  }
  return installKibanaAssetsAndReferences({
    savedObjectsClient,
    logger,
    pkgName,
    pkgTitle,
    packageInstallContext,
    installedPkg,
    spaceId,
    assetTags,
    installAsAdditionalSpace
  });
}
async function installKibanaAssetsAndReferences({
  savedObjectsClient,
  logger,
  pkgName,
  pkgTitle,
  packageInstallContext,
  installedPkg,
  spaceId,
  assetTags,
  installAsAdditionalSpace
}) {
  const {
    savedObjectsImporter,
    savedObjectTagAssignmentService,
    savedObjectTagClient
  } = (0, _saved_objects.getSpaceAwareSaveobjectsClients)(spaceId);
  // This is where the memory consumption is rising up in the first place
  const kibanaAssetsArchiveIterator = getKibanaAssetsArchiveIterator(packageInstallContext);
  if (installedPkg) {
    await (0, _remove.deleteKibanaSavedObjectsAssets)({
      savedObjectsClient,
      installedPkg,
      spaceId
    });
  }
  let installedKibanaAssetsRefs = [];
  const importedAssets = await installKibanaAssets({
    savedObjectsClient,
    logger,
    savedObjectsImporter,
    pkgName,
    kibanaAssetsArchiveIterator
  });
  const assets = importedAssets.map(({
    id,
    type,
    destinationId
  }) => ({
    id: destinationId !== null && destinationId !== void 0 ? destinationId : id,
    ...(destinationId ? {
      originId: id
    } : {}),
    type
  }));
  installedKibanaAssetsRefs = await (0, _install2.saveKibanaAssetsRefs)(savedObjectsClient, pkgName, assets, installedPkg && installedPkg.attributes.installed_kibana_space_id === spaceId ? false : installAsAdditionalSpace);
  await (0, _utils2.withPackageSpan)('Create and assign package tags', () => (0, _tag_assets.tagKibanaAssets)({
    savedObjectTagAssignmentService,
    savedObjectTagClient,
    pkgTitle,
    pkgName,
    spaceId,
    importedAssets,
    assetTags
  }));
  return installedKibanaAssetsRefs;
}
async function deleteKibanaAssetsAndReferencesForSpace({
  savedObjectsClient,
  logger,
  pkgName,
  installedPkg,
  spaceId
}) {
  if (!installedPkg) {
    return;
  }
  if (installedPkg.attributes.installed_kibana_space_id === spaceId) {
    throw new _errors.FleetError('Impossible to delete kibana assets from the space where the package was installed, you must uninstall the package.');
  }
  await (0, _remove.deleteKibanaSavedObjectsAssets)({
    savedObjectsClient,
    installedPkg,
    spaceId
  });
  await (0, _install2.saveKibanaAssetsRefs)(savedObjectsClient, pkgName, null, true);
}
const kibanaAssetTypes = Object.values(_types.KibanaAssetType);
const isKibanaAssetType = path => {
  const parts = (0, _archive.getPathParts)(path);
  return parts.service === 'kibana' && kibanaAssetTypes.includes(parts.type);
};
exports.isKibanaAssetType = isKibanaAssetType;
function getKibanaAssetsArchiveIterator(packageInstallContext) {
  return onEntry => {
    return packageInstallContext.archiveIterator.traverseEntries(async entry => {
      if (!entry.buffer) {
        return;
      }
      const asset = JSON.parse(entry.buffer.toString('utf8'));
      const assetType = (0, _archive.getPathParts)(entry.path).type;
      const soType = KibanaSavedObjectTypeMapping[assetType];
      if (!validKibanaAssetTypes.has(assetType)) {
        return;
      }
      if (asset.type === soType) {
        await onEntry({
          path: entry.path,
          assetType,
          asset
        });
      }
    }, isKibanaAssetType);
  };
}
const isImportConflictError = e => {
  var _e$error;
  return (e === null || e === void 0 ? void 0 : (_e$error = e.error) === null || _e$error === void 0 ? void 0 : _e$error.type) === 'conflict';
};
/**
 * retry saved object import if only conflict errors are encountered
 */
async function retryImportOnConflictError(importCall, {
  logger,
  maxAttempts = 50,
  _attempt = 0
} = {}) {
  var _result$errors;
  const result = await importCall();
  const errors = (_result$errors = result.errors) !== null && _result$errors !== void 0 ? _result$errors : [];
  if (_attempt < maxAttempts && errors.length && errors.every(isImportConflictError)) {
    const retryCount = _attempt + 1;
    const retryDelayMs = 1000 + Math.floor(Math.random() * 3000); // 1s + 0-3s of jitter

    logger === null || logger === void 0 ? void 0 : logger.debug(() => `Retrying import operation after [${retryDelayMs * 1000}s] due to conflict errors: ${JSON.stringify(errors)}`);
    await (0, _promises.setTimeout)(retryDelayMs);
    return retryImportOnConflictError(importCall, {
      logger,
      _attempt: retryCount
    });
  }
  return result;
}

// only exported for testing
async function installKibanaSavedObjects({
  savedObjectsImporter,
  kibanaAssets,
  assetsChunkSize,
  logger
}) {
  if (!assetsChunkSize || kibanaAssets.length <= assetsChunkSize || hasReferences(kibanaAssets)) {
    return await installKibanaSavedObjectsChunk({
      logger,
      savedObjectsImporter,
      kibanaAssets,
      refresh: 'wait_for'
    });
  }
  const installedAssets = [];

  // If the package size is too large, we need to install in chunks to avoid
  // memory issues as the SO import creates a lot of objects in memory

  // NOTE: if there are references, we can't chunk the install because
  // referenced objects might end up in different chunks leading to import
  // errors.
  const assetChunks = (0, _lodash.chunk)(kibanaAssets, assetsChunkSize);
  const allAssetChunksButLast = assetChunks.slice(0, -1);
  const lastAssetChunk = assetChunks.slice(-1)[0];
  for (const assetChunk of allAssetChunksButLast) {
    const result = await installKibanaSavedObjectsChunk({
      logger,
      savedObjectsImporter,
      kibanaAssets: assetChunk,
      refresh: false
    });
    installedAssets.push(...result);
  }
  const result = await installKibanaSavedObjectsChunk({
    logger,
    savedObjectsImporter,
    kibanaAssets: lastAssetChunk,
    refresh: 'wait_for'
  });
  installedAssets.push(...result);
  return installedAssets;
}

// only exported for testing
async function installKibanaSavedObjectsChunk({
  savedObjectsImporter,
  kibanaAssets,
  logger,
  refresh
}) {
  if (!kibanaAssets.length) {
    return [];
  }
  const toBeSavedObjects = kibanaAssets.map(asset => createSavedObjectKibanaAsset(asset));
  let allSuccessResults = [];
  const {
    successResults: importSuccessResults = [],
    errors: importErrors = [],
    success
  } = await retryImportOnConflictError(() => {
    const readStream = (0, _utils.createListStream)(toBeSavedObjects);
    return savedObjectsImporter.import({
      overwrite: true,
      readStream,
      createNewCopies: false,
      managed: true,
      refresh
    });
  });
  if (success) {
    allSuccessResults = importSuccessResults;
  }
  const [referenceErrors, otherErrors] = (0, _lodash.partition)(importErrors, e => {
    var _e$error2;
    return (e === null || e === void 0 ? void 0 : (_e$error2 = e.error) === null || _e$error2 === void 0 ? void 0 : _e$error2.type) === 'missing_references';
  });
  if (otherErrors !== null && otherErrors !== void 0 && otherErrors.length) {
    throw new _errors.KibanaSOReferenceError(`Encountered ${otherErrors.length} errors creating saved objects: ${formatImportErrorsForLog(otherErrors)}`);
  }

  /*
    A reference error here means that a saved object reference in the references
    array cannot be found. This is an error in the package its-self but not a fatal
    one. For example a dashboard may still refer to the legacy `metricbeat-*` index
    pattern. We ignore reference errors here so that legacy version of a package
    can still be installed, but if a warning is logged it should be reported to
    the integrations team. */
  if (referenceErrors.length) {
    logger.debug(() => `Resolving ${referenceErrors.length} reference errors creating saved objects: ${formatImportErrorsForLog(referenceErrors)}`);
    const retries = toBeSavedObjects.map(({
      id,
      type
    }) => {
      if (referenceErrors.find(({
        id: idToSearch
      }) => idToSearch === id)) {
        return {
          id,
          type,
          ignoreMissingReferences: true,
          replaceReferences: [],
          overwrite: true
        };
      }
      return {
        id,
        type,
        overwrite: true,
        replaceReferences: []
      };
    });
    const {
      successResults: resolveSuccessResults = [],
      errors: resolveErrors = []
    } = await savedObjectsImporter.resolveImportErrors({
      readStream: (0, _utils.createListStream)(toBeSavedObjects),
      createNewCopies: false,
      managed: true,
      retries
    });
    if (resolveErrors !== null && resolveErrors !== void 0 && resolveErrors.length) {
      throw new _errors.KibanaSOReferenceError(`Encountered ${resolveErrors.length} errors resolving reference errors: ${formatImportErrorsForLog(resolveErrors)}`);
    }
    allSuccessResults = allSuccessResults.concat(resolveSuccessResults);
  }
  return allSuccessResults;
}

// Filter out any reserved index patterns
function removeReservedIndexPatterns(kibanaAssets) {
  const reservedPatterns = _install.indexPatternTypes.map(pattern => `${pattern}-*`);
  return kibanaAssets.filter(asset => !reservedPatterns.includes(asset.id));
}
function toAssetReference({
  id,
  type
}) {
  const reference = {
    id,
    type: type
  };
  return reference;
}
function hasReferences(assetsToInstall) {
  return assetsToInstall.some(asset => {
    var _asset$references;
    return (_asset$references = asset.references) === null || _asset$references === void 0 ? void 0 : _asset$references.length;
  });
}