"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.onSaveSearch = onSaveSearch;
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _eui = require("@elastic/eui");
var _i18nReact = require("@kbn/i18n-react");
var _public = require("@kbn/saved-objects-plugin/public");
var _get_allowed_sample_size = require("../../../../utils/get_allowed_sample_size");
var _redux = require("../../state_management/redux");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function saveDataSource({
  savedSearch,
  saveOptions,
  services,
  state,
  navigateOrReloadSavedSearch
}) {
  const prevSavedSearchId = savedSearch.id;
  function onSuccess(id) {
    if (id) {
      services.toastNotifications.addSuccess({
        title: _i18n.i18n.translate('discover.notifications.savedSearchTitle', {
          defaultMessage: `Discover session ''{savedSearchTitle}'' was saved`,
          values: {
            savedSearchTitle: savedSearch.title
          }
        }),
        'data-test-subj': 'saveSearchSuccess'
      });
      if (navigateOrReloadSavedSearch) {
        if (id !== prevSavedSearchId) {
          services.locator.navigate({
            savedSearchId: id
          });
        } else {
          // Update defaults so that "reload saved query" functions correctly
          state.actions.undoSavedSearchChanges();
        }
      }
    }
  }
  function onError(error) {
    services.toastNotifications.addDanger({
      title: _i18n.i18n.translate('discover.notifications.notSavedSearchTitle', {
        defaultMessage: `Discover session ''{savedSearchTitle}'' was not saved.`,
        values: {
          savedSearchTitle: savedSearch.title
        }
      }),
      text: error.message
    });
  }
  try {
    const response = await state.savedSearchState.persist(savedSearch, saveOptions);
    if (response !== null && response !== void 0 && response.id) {
      onSuccess(response.id);
    }
    return response;
  } catch (error) {
    onError(error);
  }
}
async function onSaveSearch({
  savedSearch,
  services,
  state,
  initialCopyOnSave,
  onClose,
  onSaveCb
}) {
  var _dataView$isTimeBased, _savedSearch$title, _savedSearch$tags;
  const {
    uiSettings,
    savedObjectsTagging
  } = services;
  const dataView = savedSearch.searchSource.getField('index');
  const currentTab = state.getCurrentTab();
  const overriddenVisContextAfterInvalidation = currentTab.overriddenVisContextAfterInvalidation;
  const onSave = async ({
    newTitle,
    newCopyOnSave,
    newTimeRestore,
    newDescription,
    newTags,
    isTitleDuplicateConfirmed,
    onTitleDuplicate
  }) => {
    const appState = state.appState.getState();
    const currentTitle = savedSearch.title;
    const currentTimeRestore = savedSearch.timeRestore;
    const currentRowsPerPage = savedSearch.rowsPerPage;
    const currentSampleSize = savedSearch.sampleSize;
    const currentDescription = savedSearch.description;
    const currentTags = savedSearch.tags;
    const currentVisContext = savedSearch.visContext;
    savedSearch.title = newTitle;
    savedSearch.description = newDescription;
    savedSearch.timeRestore = newTimeRestore;
    savedSearch.rowsPerPage = appState.rowsPerPage;

    // save the custom value or reset it if it's invalid
    const appStateSampleSize = appState.sampleSize;
    const allowedSampleSize = (0, _get_allowed_sample_size.getAllowedSampleSize)(appStateSampleSize, uiSettings);
    savedSearch.sampleSize = appStateSampleSize && allowedSampleSize === appStateSampleSize ? appStateSampleSize : undefined;
    if (savedObjectsTagging) {
      savedSearch.tags = newTags;
    }
    if (overriddenVisContextAfterInvalidation) {
      savedSearch.visContext = overriddenVisContextAfterInvalidation;
    }
    const saveOptions = {
      onTitleDuplicate,
      copyOnSave: newCopyOnSave,
      isTitleDuplicateConfirmed
    };
    if (newCopyOnSave) {
      await state.actions.updateAdHocDataViewId();
    }
    const navigateOrReloadSavedSearch = !Boolean(onSaveCb);
    const response = await saveDataSource({
      saveOptions,
      services,
      savedSearch,
      state,
      navigateOrReloadSavedSearch
    });

    // If the save wasn't successful, put the original values back.
    if (!response) {
      savedSearch.title = currentTitle;
      savedSearch.timeRestore = currentTimeRestore;
      savedSearch.rowsPerPage = currentRowsPerPage;
      savedSearch.sampleSize = currentSampleSize;
      savedSearch.description = currentDescription;
      savedSearch.visContext = currentVisContext;
      if (savedObjectsTagging) {
        savedSearch.tags = currentTags;
      }
    } else {
      state.internalState.dispatch(state.injectCurrentTab(_redux.internalStateActions.resetOnSavedSearchChange)());
      state.appState.resetInitialState();
    }
    onSaveCb === null || onSaveCb === void 0 ? void 0 : onSaveCb();
    return response !== null && response !== void 0 ? response : {};
  };
  const saveModal = /*#__PURE__*/_react.default.createElement(SaveSearchObjectModal, {
    isTimeBased: (_dataView$isTimeBased = dataView === null || dataView === void 0 ? void 0 : dataView.isTimeBased()) !== null && _dataView$isTimeBased !== void 0 ? _dataView$isTimeBased : false,
    services: services,
    title: (_savedSearch$title = savedSearch.title) !== null && _savedSearch$title !== void 0 ? _savedSearch$title : '',
    showCopyOnSave: !!savedSearch.id,
    initialCopyOnSave: initialCopyOnSave,
    description: savedSearch.description,
    timeRestore: savedSearch.timeRestore,
    tags: (_savedSearch$tags = savedSearch.tags) !== null && _savedSearch$tags !== void 0 ? _savedSearch$tags : [],
    managed: savedSearch.managed,
    onSave: onSave,
    onClose: onClose !== null && onClose !== void 0 ? onClose : () => {}
  });
  (0, _public.showSaveModal)(saveModal);
}
const SaveSearchObjectModal = ({
  isTimeBased,
  services,
  title,
  description,
  tags,
  showCopyOnSave,
  initialCopyOnSave,
  timeRestore: savedTimeRestore,
  onSave,
  onClose,
  managed
}) => {
  const {
    savedObjectsTagging
  } = services;
  const [timeRestore, setTimeRestore] = (0, _react.useState)(isTimeBased && savedTimeRestore || false);
  const [currentTags, setCurrentTags] = (0, _react.useState)(tags);
  const onModalSave = async params => {
    return onSave({
      ...params,
      newTimeRestore: timeRestore,
      newTags: currentTags
    });
  };
  const tagSelector = savedObjectsTagging ? /*#__PURE__*/_react.default.createElement(savedObjectsTagging.ui.components.SavedObjectSaveModalTagSelector, {
    initialSelection: currentTags,
    onTagsSelected: newTags => {
      setCurrentTags(newTags);
    }
  }) : undefined;
  const timeSwitch = isTimeBased ? /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
    helpText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "discover.topNav.saveModal.storeTimeWithSearchToggleDescription",
      defaultMessage: "Update the time filter and refresh interval to the current selection when using this session."
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": "storeTimeWithSearch",
    checked: timeRestore,
    onChange: event => setTimeRestore(event.target.checked),
    label: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "discover.topNav.saveModal.storeTimeWithSearchToggleLabel",
      defaultMessage: "Store time with Discover session"
    })
  })) : null;
  const options = tagSelector ? /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, tagSelector, timeSwitch) : timeSwitch;
  return /*#__PURE__*/_react.default.createElement(_public.SavedObjectSaveModalWithSaveResult, {
    title: title,
    showCopyOnSave: showCopyOnSave,
    initialCopyOnSave: initialCopyOnSave,
    description: description,
    objectType: _i18n.i18n.translate('discover.localMenu.saveSaveSearchObjectType', {
      defaultMessage: 'Discover session'
    }),
    showDescription: true,
    options: options,
    onSave: onModalSave,
    onClose: onClose,
    mustCopyOnSaveMessage: managed ? _i18n.i18n.translate('discover.localMenu.mustCopyOnSave', {
      defaultMessage: 'Elastic manages this Discover session. Save any changes to a new Discover session.'
    }) : undefined
  });
};