"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _mcp_gateway = require("./mcp_gateway");
var _prompts = require("./prompts");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAgentGraph = async ({
  agent,
  chatModel,
  session,
  logger
}) => {
  const StateAnnotation = _langgraph.Annotation.Root({
    // inputs
    initialMessages: (0, _langgraph.Annotation)({
      reducer: _langgraph.messagesStateReducer,
      default: () => []
    }),
    // outputs
    addedMessages: (0, _langgraph.Annotation)({
      reducer: _langgraph.messagesStateReducer,
      default: () => []
    }),
    citations: (0, _langgraph.Annotation)({
      reducer: (a = [], b = []) => [...a, ...b],
      default: () => []
    })
  });
  const toolsProvider = new _mcp_gateway.ToolsProvider({
    session,
    logger
  });
  const allTools = await toolsProvider.getAllTools();
  const tools = [...allTools];
  const toolNode = new _prebuilt.ToolNode(tools);
  const model = chatModel.bindTools(tools).withConfig({
    tags: ['workflow', `agent:${agent.id}`]
  });
  const callModel = async state => {
    const response = await model.invoke(await (0, _prompts.withSystemPrompt)({
      agentPrompt: agent.configuration.systemPrompt,
      messages: [...state.initialMessages, ...state.addedMessages]
    }));
    return {
      addedMessages: [response]
    };
  };
  const shouldContinue = async state => {
    var _lastMessage$tool_cal;
    const messages = state.addedMessages;
    const lastMessage = messages[messages.length - 1];
    if (lastMessage && (_lastMessage$tool_cal = lastMessage.tool_calls) !== null && _lastMessage$tool_cal !== void 0 && _lastMessage$tool_cal.length) {
      return 'tools';
    }
    return '__end__';
  };
  const toolHandler = async state => {
    const toolNodeResult = await toolNode.invoke(state.addedMessages);
    const citations = (0, _utils.extractCitations)({
      messages: toolNodeResult
    });
    return {
      addedMessages: [...toolNodeResult],
      citations
    };
  };
  const graph = new _langgraph.StateGraph(StateAnnotation).addNode('agent', callModel).addNode('tools', toolHandler).addEdge('__start__', 'agent').addEdge('tools', 'agent').addConditionalEdges('agent', shouldContinue, {
    tools: 'tools',
    __end__: '__end__'
  }).compile();
  return graph;
};
exports.createAgentGraph = createAgentGraph;