"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ChatService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _chat_events = require("../../../common/chat_events");
var _errors = require("../../../common/errors");
var _conversation_events = require("../../../common/conversation_events");
var _generate_conversation_title = require("./generate_conversation_title");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ChatService {
  constructor({
    inference,
    logger,
    conversationService,
    agentFactory
  }) {
    (0, _defineProperty2.default)(this, "inference", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "conversationService", void 0);
    (0, _defineProperty2.default)(this, "agentFactory", void 0);
    this.inference = inference;
    this.logger = logger;
    this.conversationService = conversationService;
    this.agentFactory = agentFactory;
  }
  converse({
    agentId,
    conversationId,
    connectorId,
    request,
    nextUserMessage
  }) {
    const logError = (source, err) => {
      var _err$stack;
      this.logger.error(`Error during converse from ${source}:\n${(_err$stack = err.stack) !== null && _err$stack !== void 0 ? _err$stack : err.message}`);
    };
    const isNewConversation = !conversationId;
    const nextUserEvent = (0, _conversation_events.createUserMessage)({
      content: nextUserMessage
    });
    return (0, _rxjs.forkJoin)({
      agentRunner: (0, _rxjs.defer)(() => this.agentFactory.getAgentRunner({
        request,
        connectorId,
        agentId
      })),
      conversationClient: (0, _rxjs.defer)(() => this.conversationService.getScopedClient({
        request
      })),
      chatModel: (0, _rxjs.defer)(() => this.inference.getChatModel({
        request,
        connectorId,
        chatModelOptions: {}
      }))
    }).pipe((0, _rxjs.switchMap)(({
      conversationClient,
      chatModel,
      agentRunner
    }) => {
      const conversation$ = getConversation$({
        agentId,
        conversationId,
        conversationClient
      });
      const conversationEvents$ = conversation$.pipe((0, _rxjs.map)(conversation => {
        return [...conversation.events, nextUserEvent];
      }), (0, _rxjs.shareReplay)());
      const agentEvents$ = conversationEvents$.pipe((0, _rxjs.switchMap)(conversationEvents => {
        return (0, _rxjs.defer)(() => agentRunner.run({
          previousEvents: conversationEvents
        }));
      }), (0, _rxjs.switchMap)(agentRunResult => {
        return agentRunResult.events$;
      }), (0, _rxjs.shareReplay)());

      // generate a title for the new conversations
      const title$ = isNewConversation ? generatedTitle$({
        chatModel,
        conversationEvents$
      }) : conversation$.pipe((0, _rxjs.switchMap)(conversation => {
        return (0, _rxjs.of)(conversation.title);
      }));

      // extract new conversation events from the agent output
      const newConversationEvents$ = extractNewConversationEvents$({
        agentEvents$
      }).pipe((0, _rxjs.map)(events => {
        return [nextUserEvent, ...events];
      }));

      // save or update the conversation and emit the corresponding chat event
      const saveOrUpdateAndEmit$ = isNewConversation ? createConversation$({
        agentId,
        title$,
        newConversationEvents$,
        conversationClient
      }) : updateConversation$({
        title$,
        conversation$,
        conversationClient,
        newConversationEvents$
      });
      return (0, _rxjs.merge)(agentEvents$, saveOrUpdateAndEmit$).pipe((0, _rxjs.catchError)(err => {
        logError('main observable', err);
        return (0, _rxjs.throwError)(() => {
          if ((0, _errors.isChatError)(err)) {
            return err;
          }
          return (0, _errors.createChatError)('internalError', err.message, {});
        });
      }), (0, _rxjs.shareReplay)());
    }));
  }
}
exports.ChatService = ChatService;
const generatedTitle$ = ({
  chatModel,
  conversationEvents$
}) => {
  return conversationEvents$.pipe((0, _rxjs.switchMap)(conversationEvents => {
    return (0, _rxjs.defer)(async () => (0, _generate_conversation_title.generateConversationTitle)({
      conversationEvents,
      chatModel
    })).pipe((0, _rxjs.shareReplay)());
  }));
};
const getConversation$ = ({
  agentId,
  conversationId,
  conversationClient
}) => {
  return (0, _rxjs.defer)(() => {
    if (conversationId) {
      return conversationClient.get({
        conversationId
      });
    } else {
      return (0, _rxjs.of)(placeholderConversation({
        agentId
      }));
    }
  }).pipe((0, _rxjs.shareReplay)());
};
const placeholderConversation = ({
  agentId
}) => {
  return {
    id: (0, _uuid.v4)(),
    title: 'New conversation',
    // TODO: translate default
    agentId,
    events: [],
    lastUpdated: new Date().toISOString(),
    user: {
      id: 'unknown',
      name: 'unknown'
    }
  };
};

/**
 * Extract the new conversation events from the output of the agent.
 *
 * Emits only once with an array of event when the agent events observable completes.
 */
const extractNewConversationEvents$ = ({
  agentEvents$
}) => {
  return agentEvents$.pipe((0, _rxjs.filter)(event => {
    return (0, _chat_events.isMessageEvent)(event) || (0, _chat_events.isToolResultEvent)(event);
  }), (0, _rxjs.toArray)(), (0, _rxjs.mergeMap)(newMessages => {
    return (0, _rxjs.of)(newMessages.map(message => {
      if ((0, _chat_events.isMessageEvent)(message)) {
        return message.message;
      } else {
        return (0, _conversation_events.createToolResult)({
          toolCallId: message.toolResult.callId,
          toolResult: message.toolResult.result
        });
      }
    }));
  }), (0, _rxjs.shareReplay)());
};

/**
 * Persist a new conversation and emit the corresponding event
 */
const createConversation$ = ({
  agentId,
  conversationClient,
  title$,
  newConversationEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    title: title$,
    newConversationEvents: newConversationEvents$
  }).pipe((0, _rxjs.switchMap)(({
    title,
    newConversationEvents
  }) => {
    return conversationClient.create({
      title,
      agentId,
      events: [...newConversationEvents]
    });
  }), (0, _rxjs.switchMap)(updatedConversation => {
    return (0, _rxjs.of)((0, _chat_events.conversationCreatedEvent)({
      title: updatedConversation.title,
      id: updatedConversation.id
    }));
  }));
};

/**
 * Update an existing conversation and emit the corresponding event
 */
const updateConversation$ = ({
  conversationClient,
  conversation$,
  title$,
  newConversationEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    conversation: conversation$,
    title: title$,
    newConversationEvents: newConversationEvents$
  }).pipe((0, _rxjs.switchMap)(({
    conversation,
    title,
    newConversationEvents
  }) => {
    return conversationClient.update(conversation.id, {
      title,
      events: [...conversation.events, ...newConversationEvents]
    });
  }), (0, _rxjs.switchMap)(updatedConversation => {
    return (0, _rxjs.of)((0, _chat_events.conversationUpdatedEvent)({
      title: updatedConversation.title,
      id: updatedConversation.id
    }));
  }));
};