"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerIntegrationsRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _wciCommon = require("@kbn/wci-common");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerIntegrationsRoutes = ({
  getServices,
  router,
  logger
}) => {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // Get a single integration by ID
  router.get({
    path: '/internal/workchat/tools/{integrationId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        integrationId: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      integrationsService
    } = getServices();
    const client = await integrationsService.getScopedClient({
      request
    });
    const {
      integrationId
    } = request.params;
    const integration = await client.get({
      integrationId
    });
    return res.ok({
      body: integration
    });
  }));

  // List all integrations
  router.get({
    path: '/internal/workchat/tools',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: {}
  }, wrapHandler(async (ctx, request, res) => {
    const {
      integrationsService
    } = getServices();
    const client = await integrationsService.getScopedClient({
      request
    });
    const integrations = await client.list();
    return res.ok({
      body: {
        integrations
      }
    });
  }));

  // Create a new integration
  router.post({
    path: '/internal/workchat/tools',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        type: _configSchema.schema.oneOf(
        // @ts-expect-error complains that IntegrationType may have less than 1 element...
        Object.values(_wciCommon.IntegrationType).map(val => _configSchema.schema.literal(val))),
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        configuration: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      integrationsService
    } = getServices();
    const client = await integrationsService.getScopedClient({
      request
    });
    const {
      type,
      name,
      description,
      configuration
    } = request.body;
    const integration = await client.create({
      type,
      name,
      description,
      configuration
    });
    return res.ok({
      body: integration
    });
  }));

  // Update an existing integration
  router.put({
    path: '/internal/workchat/tools/{integrationId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        integrationId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.maybe(_configSchema.schema.string()),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        configuration: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      integrationsService
    } = getServices();
    const client = await integrationsService.getScopedClient({
      request
    });
    const {
      integrationId
    } = request.params;
    const {
      name,
      description,
      configuration
    } = request.body;
    const integration = await client.update(integrationId, {
      name,
      description,
      configuration
    });
    return res.ok({
      body: integration
    });
  }));

  // Delete an integration
  router.delete({
    path: '/internal/workchat/tools/{integrationId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        integrationId: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      integrationsService
    } = getServices();
    const client = await integrationsService.getScopedClient({
      request
    });
    const {
      integrationId
    } = request.params;
    await client.delete(integrationId);
    return res.ok({
      body: {
        success: true
      }
    });
  }));
};
exports.registerIntegrationsRoutes = registerIntegrationsRoutes;