"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConversationRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerConversationRoutes = ({
  getServices,
  router,
  logger
}) => {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // get conversation by id
  router.get({
    path: '/internal/workchat/conversations/{conversationId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        conversationId: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      conversationService
    } = getServices();
    const client = await conversationService.getScopedClient({
      request
    });
    const {
      conversationId
    } = request.params;
    const conversation = await client.get({
      conversationId
    });
    return res.ok({
      body: conversation
    });
  }));

  // list all conversations for a given agent
  router.post({
    path: '/internal/workchat/conversations',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        agentId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      conversationService
    } = getServices();
    const client = await conversationService.getScopedClient({
      request
    });
    const params = request.body;
    const conversations = await client.list({
      agentId: params.agentId
    });
    const summaries = conversations.map(conv => {
      return {
        id: conv.id,
        agentId: conv.agentId,
        title: conv.title,
        lastUpdated: conv.lastUpdated
      };
    });
    return res.ok({
      body: {
        conversations: summaries
      }
    });
  }));
};
exports.registerConversationRoutes = registerConversationRoutes;