"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useChat = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _conversation_events = require("../../../common/conversation_events");
var _use_workchat_service = require("./use_workchat_service");
var _use_kibana = require("./use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useChat = ({
  conversationId,
  agentId,
  connectorId,
  onConversationUpdate,
  onError
}) => {
  const {
    chatService
  } = (0, _use_workchat_service.useWorkChatServices)();
  const {
    services: {
      notifications
    }
  } = (0, _use_kibana.useKibana)();
  const [conversationEvents, setConversationEvents] = (0, _react.useState)([]);
  const [pendingMessages, setPendingMessages] = (0, _react.useState)([]);
  const [progressionEvents, setProgressionEvents] = (0, _react.useState)([]);
  const [status, setStatus] = (0, _react.useState)('ready');
  const sendMessage = (0, _react.useCallback)(nextMessage => {
    if (status === 'loading') {
      return;
    }
    setStatus('loading');
    setConversationEvents(prevEvents => [...prevEvents, (0, _conversation_events.createUserMessage)({
      content: nextMessage
    })]);
    const events$ = chatService.converse({
      nextMessage,
      conversationId,
      agentId,
      connectorId
    });
    const streamMessages = [];
    let concatenatedChunks = '';
    const getAllStreamMessages = () => {
      return streamMessages.concat(concatenatedChunks.length ? [(0, _conversation_events.createAssistantMessage)({
        content: concatenatedChunks
      })] : []);
    };
    events$.subscribe({
      next: event => {
        // chunk received, we append it to the chunk buffer
        if (event.type === 'message_chunk') {
          concatenatedChunks += event.content_chunk;
          setPendingMessages(getAllStreamMessages());
          setProgressionEvents([]);
        }

        // full message received - we purge the chunk buffer
        // and insert the received message into the temporary list
        if (event.type === 'message') {
          concatenatedChunks = '';
          streamMessages.push(event.message);
          setPendingMessages(getAllStreamMessages());
        }
        if (event.type === 'tool_result') {
          concatenatedChunks = '';
          streamMessages.push((0, _conversation_events.createToolResult)({
            toolCallId: event.toolResult.callId,
            toolResult: event.toolResult.result
          }));
          setPendingMessages(getAllStreamMessages());
        }
        if (event.type === 'conversation_created') {
          onConversationUpdate(event.conversation);
        }
        if (event.type === 'progression') {
          setProgressionEvents(previous => [...previous, event]);
        }
      },
      complete: () => {
        setPendingMessages([]);
        setProgressionEvents([]);
        setConversationEvents(prevEvents => [...prevEvents, ...streamMessages]);
        setStatus('ready');
      },
      error: err => {
        setPendingMessages([]);
        setProgressionEvents([]);
        setConversationEvents(prevEvents => [...prevEvents, ...streamMessages]);
        setStatus('error');
        onError === null || onError === void 0 ? void 0 : onError(err);
        notifications.toasts.addError(err, {
          title: _i18n.i18n.translate('xpack.workchatApp.chat.chatError.title', {
            defaultMessage: 'Error loading chat response'
          }),
          toastMessage: `${err.code} - ${err.message}`
        });
      }
    });
  }, [chatService, notifications, status, agentId, conversationId, connectorId, onConversationUpdate, onError]);
  const setConversationEventsExternal = (0, _react.useCallback)(newEvents => {
    // TODO: unsub from observable + set status ready
    setConversationEvents(newEvents);
    setProgressionEvents([]);
    setPendingMessages([]);
  }, []);
  const allEvents = (0, _react.useMemo)(() => {
    return [...conversationEvents, ...pendingMessages];
  }, [conversationEvents, pendingMessages]);
  return {
    status,
    sendMessage,
    conversationEvents: allEvents,
    progressionEvents,
    setConversationEvents: setConversationEventsExternal
  };
};
exports.useChat = useChat;