"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConfigurationRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _features = require("@kbn/workchat-app/common/features");
var _wcIndexSchemaBuilder = require("@kbn/wc-index-schema-builder");
var _wcGenaiUtils = require("@kbn/wc-genai-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerConfigurationRoutes = ({
  router,
  core,
  logger
}) => {
  // generate the index integration schema for a given index
  router.post({
    path: '/internal/wci-index-source/configuration/generate',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, async (ctx, request, res) => {
    try {
      const [, {
        actions,
        inference
      }] = await core.getStartServices();
      const {
        elasticsearch
      } = await ctx.core;
      const connectors = await (0, _wcGenaiUtils.getConnectorList)({
        actions,
        request
      });
      const connector = (0, _wcGenaiUtils.getDefaultConnector)({
        connectors
      });
      const chatModel = await inference.getChatModel({
        request,
        connectorId: connector.connectorId,
        chatModelOptions: {}
      });
      const {
        indexName
      } = request.body;
      const definition = await (0, _wcIndexSchemaBuilder.buildSchema)({
        indexName,
        chatModel,
        esClient: elasticsearch.client.asCurrentUser,
        logger
      });
      return res.ok({
        body: {
          definition
        }
      });
    } catch (e) {
      logger.error(e);
      throw e;
    }
  });
  router.get({
    path: '/internal/wci-index-source/indices-autocomplete',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      query: _configSchema.schema.object({
        index: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (ctx, request, res) => {
    const {
      elasticsearch
    } = await ctx.core;
    let pattern = request.query.index || '';
    if (pattern.length >= 3) {
      pattern = `${pattern}*`;
    }
    const esClient = elasticsearch.client.asCurrentUser;
    try {
      const response = await esClient.cat.indices({
        index: [pattern],
        h: 'index',
        expand_wildcards: 'open',
        format: 'json'
      });
      return res.ok({
        body: {
          indexNames: response.map(indexRecord => indexRecord.index).filter(index => !!index)
        }
      });
    } catch (e) {
      var _e$meta, _e$meta$body, _e$meta$body$error;
      // TODO: sigh, is there a better way?
      if ((e === null || e === void 0 ? void 0 : (_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : (_e$meta$body = _e$meta.body) === null || _e$meta$body === void 0 ? void 0 : (_e$meta$body$error = _e$meta$body.error) === null || _e$meta$body$error === void 0 ? void 0 : _e$meta$body$error.type) === 'index_not_found_exception') {
        return res.ok({
          body: {
            indexNames: []
          }
        });
      }
      throw e;
    }
  });
};
exports.registerConfigurationRoutes = registerConfigurationRoutes;