"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AbstractSearchStrategy = void 0;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _fields_utils = require("../../../../common/fields_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getRequestAbortedSignal(aborted$) {
  const controller = new AbortController();
  aborted$.subscribe(() => controller.abort());
  return controller.signal;
}
class AbstractSearchStrategy {
  async search(requestContext, req, esRequests, trackedEsSearches, indexType) {
    const requests = [];
    const searchContext = await requestContext.search;
    esRequests.forEach(({
      body = {},
      index,
      trackingEsSearchMeta,
      ...rest
    }) => {
      // User may abort the request without waiting for the results
      // we need to handle this scenario by aborting underlying server requests
      const abortSignal = getRequestAbortedSignal(req.events.aborted$);
      const startTime = Date.now();
      const searchBody = {
        ...rest,
        ...(typeof body === 'string' ? {
          body
        } : body)
      };
      requests.push(searchContext.search({
        indexType,
        params: {
          ...searchBody,
          index
        }
      }, {
        ...req.body.searchSession,
        abortSignal
      }).pipe((0, _rxjs.tap)(data => {
        if (trackingEsSearchMeta !== null && trackingEsSearchMeta !== void 0 && trackingEsSearchMeta.requestId && trackedEsSearches) {
          trackedEsSearches[trackingEsSearchMeta.requestId] = {
            body: searchBody,
            time: Date.now() - startTime,
            label: trackingEsSearchMeta.requestLabel,
            response: (0, _lodash.omit)(data.rawResponse, 'aggregations')
          };
        }
      })).toPromise());
    });
    return Promise.all(requests);
  }
  checkForViability(requestContext, req, fetchedIndexPattern) {
    throw new TypeError('Must override method');
  }
  async getFieldsForWildcard(fetchedIndexPattern, indexPatternsService, capabilities, options) {
    var _fetchedIndexPattern$;
    return (0, _fields_utils.toSanitizedFieldType)(fetchedIndexPattern.indexPattern ? fetchedIndexPattern.indexPattern.getNonScriptedFields() : await indexPatternsService.getFieldsForWildcard({
      pattern: (_fetchedIndexPattern$ = fetchedIndexPattern.indexPatternString) !== null && _fetchedIndexPattern$ !== void 0 ? _fetchedIndexPattern$ : '',
      metaFields: [],
      ...options
    }));
  }
}
exports.AbstractSearchStrategy = AbstractSearchStrategy;