"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchUsageCounters = searchUsageCounters;
var _lodash = require("lodash");
var _saved_objects = require("../saved_objects");
var _kuery_utils = require("../common/kuery_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function searchUsageCounters(repository, params) {
  const {
    filters,
    options = {}
  } = params;
  const {
    namespace: filterNamespace
  } = filters;
  const baseFindParams = {
    ...(filterNamespace && {
      namespaces: [filterNamespace]
    }),
    type: _saved_objects.USAGE_COUNTERS_SAVED_OBJECT_TYPE,
    filter: (0, _kuery_utils.usageCountersSearchParamsToKueryFilter)(filters),
    perPage: options.perPage || 100
  };

  // create a PIT to perform consecutive searches
  const pit = await repository.openPointInTimeForType(_saved_objects.USAGE_COUNTERS_SAVED_OBJECT_TYPE);
  // create a data structure to store/aggregate all counters
  const countersMap = new Map();
  // the current offset for the iterative search
  let searchAfter;
  do {
    var _res$saved_objects$po;
    const findParams = {
      ...baseFindParams,
      pit,
      ...(searchAfter && {
        searchAfter
      })
    };

    // this is where the actual search call is performed
    const res = await repository.find(findParams);
    res.saved_objects.forEach(result => processResult(countersMap, result));
    searchAfter = (_res$saved_objects$po = res.saved_objects.pop()) === null || _res$saved_objects$po === void 0 ? void 0 : _res$saved_objects$po.sort;
  } while (searchAfter);
  await repository.closePointInTime(pit.id);
  const counters = Array.from(countersMap.values());

  // sort daily counters descending
  counters.forEach(snapshot => snapshot.records = (0, _lodash.orderBy)(snapshot.records, 'updatedAt', 'desc'));
  return {
    counters
  };
}
function processResult(countersMap, result) {
  const {
    attributes,
    updated_at: updatedAt,
    namespaces
  } = result;
  const namespace = namespaces === null || namespaces === void 0 ? void 0 : namespaces[0];
  const key = (0, _saved_objects.serializeCounterKey)({
    ...attributes,
    namespace
  });
  let counterSnapshot = countersMap.get(key);
  if (!counterSnapshot) {
    counterSnapshot = {
      domainId: attributes.domainId,
      counterName: attributes.counterName,
      counterType: attributes.counterType,
      source: attributes.source,
      ...(namespace && (namespaces === null || namespaces === void 0 ? void 0 : namespaces[0]) && {
        namespace: namespaces[0]
      }),
      records: [{
        updatedAt: updatedAt,
        count: attributes.count
      }],
      count: attributes.count
    };
    countersMap.set(key, counterSnapshot);
  } else {
    counterSnapshot.records.push({
      updatedAt: updatedAt,
      count: attributes.count
    });
    counterSnapshot.count += attributes.count;
  }
}