"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readStream = readStream;
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _stream_crud = require("../../../lib/streams/stream_crud");
var _logs_layer = require("../../../lib/streams/component_templates/logs_layer");
var _fields = require("../../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function readStream({
  name,
  assetClient,
  streamsClient,
  scopedClusterClient
}) {
  const [streamDefinition, dashboardsAndQueries] = await Promise.all([streamsClient.getStream(name), await assetClient.getAssetLinks(name, ['dashboard', 'query'])]);
  const [dashboardLinks, queryLinks] = (0, _lodash.partition)(dashboardsAndQueries, asset => asset[_fields.ASSET_TYPE] === 'dashboard');
  const dashboards = dashboardLinks.map(dashboard => dashboard['asset.id']);
  const queries = queryLinks.map(query => {
    return query.query;
  });
  if (_streamsSchema.Streams.GroupStream.Definition.is(streamDefinition)) {
    return {
      stream: streamDefinition,
      dashboards,
      queries
    };
  }

  // These queries are only relavant for IngestStreams
  const [ancestors, dataStream, privileges] = await Promise.all([streamsClient.getAncestors(name), streamsClient.getDataStream(name).catch(e => {
    if (e.statusCode === 404) {
      return null;
    }
    throw e;
  }), streamsClient.getPrivileges(name)]);
  if (_streamsSchema.Streams.UnwiredStream.Definition.is(streamDefinition)) {
    return {
      stream: streamDefinition,
      privileges,
      elasticsearch_assets: dataStream && privileges.manage ? await (0, _stream_crud.getUnmanagedElasticsearchAssets)({
        dataStream,
        scopedClusterClient
      }) : undefined,
      data_stream_exists: !!dataStream,
      effective_lifecycle: (0, _stream_crud.getDataStreamLifecycle)(dataStream),
      dashboards,
      queries
    };
  }
  const inheritedFields = (0, _logs_layer.addAliasesForNamespacedFields)(streamDefinition, (0, _streamsSchema.getInheritedFieldsFromAncestors)(ancestors));
  const body = {
    stream: streamDefinition,
    dashboards,
    privileges,
    queries,
    effective_lifecycle: (0, _streamsSchema.findInheritedLifecycle)(streamDefinition, ancestors),
    inherited_fields: inheritedFields
  };
  return body;
}