"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConnectorTypeId = void 0;
exports.executor = executor;
exports.getConnectorType = getConnectorType;
var _i18n = require("@kbn/i18n");
var _axios = _interopRequireDefault(require("axios"));
var _pipeable = require("fp-ts/pipeable");
var _Option = require("fp-ts/Option");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _common = require("@kbn/actions-plugin/common");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _lib = require("@kbn/actions-plugin/server/lib");
var _constants = require("../../../common/auth/constants");
var _http_response_retry_header = require("../lib/http_response_retry_header");
var _result_type = require("../lib/result_type");
var _schema = require("./schema");
var _utils = require("../../../common/auth/utils");
var _schema2 = require("../../../common/auth/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ConnectorTypeId = exports.ConnectorTypeId = '.webhook';

// connector type definition
function getConnectorType() {
  return {
    id: ConnectorTypeId,
    minimumLicenseRequired: 'gold',
    name: _i18n.i18n.translate('xpack.stackConnectors.webhook.title', {
      defaultMessage: 'Webhook'
    }),
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId],
    validate: {
      config: {
        schema: _schema.ConfigSchema,
        customValidator: validateConnectorTypeConfig
      },
      secrets: {
        schema: _schema2.SecretConfigurationSchema
      },
      params: {
        schema: _schema.ParamsSchema
      }
    },
    renderParameterTemplates,
    executor
  };
}
function renderParameterTemplates(logger, params, variables) {
  if (!params.body) return params;
  return {
    body: (0, _mustache_renderer.renderMustacheString)(logger, params.body, variables, 'json')
  };
}
function validateConnectorTypeConfig(configObject, validatorServices) {
  const {
    configurationUtilities
  } = validatorServices;
  const configuredUrl = configObject.url;
  try {
    new URL(configuredUrl);
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationErrorNoHostname', {
      defaultMessage: 'error validation webhook action config: unable to parse url: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
  try {
    configurationUtilities.ensureUriAllowed(configuredUrl);
  } catch (allowListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationError', {
      defaultMessage: 'error validation webhook action config: {message}',
      values: {
        message: allowListError.message
      }
    }));
  }
  if (Boolean(configObject.authType) && !configObject.hasAuth) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.authConfigurationError', {
      defaultMessage: 'error validation webhook action config: authType must be null or undefined if hasAuth is false'
    }));
  }
  if (configObject.certType === _constants.SSLCertType.PFX) {
    const webhookSettings = configurationUtilities.getWebhookSettings();
    if (!webhookSettings.ssl.pfx.enabled) {
      throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.pfxConfigurationError', {
        defaultMessage: 'error validation webhook action config: certType "{certType}" is disabled',
        values: {
          certType: _constants.SSLCertType.PFX
        }
      }));
    }
  }
}

// action executor
async function executor(execOptions) {
  var _basicAuth$auth, _basicAuth$auth2;
  const {
    actionId,
    config,
    params,
    configurationUtilities,
    logger,
    connectorUsageCollector
  } = execOptions;
  const {
    method,
    url,
    headers = {},
    hasAuth,
    authType,
    ca,
    verificationMode
  } = config;
  const {
    body: data
  } = params;
  const secrets = execOptions.secrets;
  const {
    basicAuth,
    sslOverrides
  } = (0, _utils.buildConnectorAuth)({
    hasAuth,
    authType,
    secrets,
    verificationMode,
    ca
  });
  const axiosInstance = _axios.default.create();
  const headersWithBasicAuth = (0, _lib.combineHeadersWithBasicAuthHeader)({
    username: (_basicAuth$auth = basicAuth.auth) === null || _basicAuth$auth === void 0 ? void 0 : _basicAuth$auth.username,
    password: (_basicAuth$auth2 = basicAuth.auth) === null || _basicAuth$auth2 === void 0 ? void 0 : _basicAuth$auth2.password,
    headers
  });
  const result = await (0, _result_type.promiseResult)((0, _axios_utils.request)({
    axios: axiosInstance,
    method,
    url,
    logger,
    headers: headersWithBasicAuth,
    data,
    configurationUtilities,
    sslOverrides,
    connectorUsageCollector
  }));
  if (result == null) {
    return errorResultUnexpectedNullResponse(actionId);
  }
  if ((0, _result_type.isOk)(result)) {
    const {
      value: {
        status,
        statusText
      }
    } = result;
    logger.debug(`response from webhook action "${actionId}": [HTTP ${status}] ${statusText}`);
    return successResult(actionId, data);
  } else {
    const {
      error
    } = result;
    if (error.response) {
      const {
        status,
        statusText,
        headers: responseHeaders,
        data: {
          message: responseMessage
        }
      } = error.response;
      const responseMessageAsSuffix = responseMessage ? `: ${responseMessage}` : '';
      const message = `[${status}] ${statusText}${responseMessageAsSuffix}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      // The request was made and the server responded with a status code
      // that falls out of the range of 2xx
      // special handling for 5xx
      if (status >= 500) {
        return retryResult(actionId, message);
      }

      // special handling for rate limiting
      if (status === 429) {
        return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(responseHeaders), (0, _Option.map)(retry => retryResultSeconds(actionId, message, retry)), (0, _Option.getOrElse)(() => retryResult(actionId, message)));
      }
      return errorResultInvalid(actionId, message);
    } else if (error.code) {
      const message = `[${error.code}] ${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return errorResultRequestFailed(actionId, message);
    } else if (error.isAxiosError) {
      const message = `${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return errorResultRequestFailed(actionId, message);
    }
    logger.error(`error on ${actionId} webhook action: unexpected error`);
    return errorResultUnexpectedError(actionId);
  }
}

// Action Executor Result w/ internationalisation
function successResult(actionId, data) {
  return {
    status: 'ok',
    data,
    actionId
  };
}
function errorResultInvalid(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseErrorMessage', {
    defaultMessage: 'error calling webhook, invalid response'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultRequestFailed(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.requestFailedErrorMessage', {
    defaultMessage: 'error calling webhook, request failed'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultUnexpectedError(actionId) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.unreachableErrorMessage', {
    defaultMessage: 'error calling webhook, unexpected error'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId
  };
}
function errorResultUnexpectedNullResponse(actionId) {
  const message = _i18n.i18n.translate('xpack.stackConnectors.webhook.unexpectedNullResponseErrorMessage', {
    defaultMessage: 'unexpected null response from webhook'
  });
  return {
    status: 'error',
    actionId,
    message
  };
}
function retryResult(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseRetryLaterErrorMessage', {
    defaultMessage: 'error calling webhook, retry later'
  });
  return {
    status: 'error',
    message: errMessage,
    retry: true,
    actionId,
    serviceMessage
  };
}
function retryResultSeconds(actionId, serviceMessage, retryAfter) {
  const retryEpoch = Date.now() + retryAfter * 1000;
  const retry = new Date(retryEpoch);
  const retryString = retry.toISOString();
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseRetryDateErrorMessage', {
    defaultMessage: 'error calling webhook, retry at {retryString}',
    values: {
      retryString
    }
  });
  return {
    status: 'error',
    message: errMessage,
    retry,
    actionId,
    serviceMessage
  };
}