"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PrivilegeMonitoringDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _papaparse = _interopRequireDefault(require("papaparse"));
var _stream = require("stream");
var _constants = require("../../../../common/constants");
var _utils = require("../../../../common/entity_analytics/privilege_monitoring/utils");
var _common = require("../../../../common/api/entity_analytics/privilege_monitoring/common.gen");
var _privilege_monitoring_task = require("./tasks/privilege_monitoring_task");
var _create_or_update_index = require("../utils/create_or_update_index");
var _indices = require("./elasticsearch/indices");
var _constants2 = require("./constants");
var _audit = require("../audit");
var _actions = require("./auditing/actions");
var _events = require("../../telemetry/event_based/events");
var _batching = require("../shared/streams/batching");
var _query_existing_users = require("./users/query_existing_users");
var _upsert_batch = require("./users/bulk/upsert_batch");
var _soft_delete_omitted_users = require("./users/soft_delete_omitted_users");
var _privileged_user_parse_transform = require("./users/privileged_user_parse_transform");
var _utils2 = require("./users/bulk/utils");
var _saved_objects = require("./saved_objects");
var _event_ingested = require("./elasticsearch/pipelines/event_ingested");
var _helpers = require("./saved_objects/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PrivilegeMonitoringDataClient {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "apiKeyGenerator", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "internalUserClient", void 0);
    (0, _defineProperty2.default)(this, "engineClient", void 0);
    (0, _defineProperty2.default)(this, "monitoringIndexSourceClient", void 0);
    (0, _defineProperty2.default)(this, "createOrUpdateDefaultDataSource", async () => {
      // const sourceName = `default-monitoring-index-${this.opts.namespace}`;
      const sourceName = this.getIndex(); // `entity_analytics.monitoring.users-${this.opts.namespace}`;
      const defaultIndexSource = {
        type: 'index',
        managed: true,
        indexPattern: (0, _constants.defaultMonitoringUsersIndex)(this.opts.namespace),
        name: sourceName
      };
      const existingSources = await this.monitoringIndexSourceClient.find({
        name: sourceName
      });
      if (existingSources.saved_objects.length > 0) {
        this.log('info', 'Default index source already exists, updating it.');
        const existingSource = existingSources.saved_objects[0];
        try {
          await this.monitoringIndexSourceClient.update({
            id: existingSource.id,
            ...defaultIndexSource
          });
        } catch (e) {
          this.log('error', `Failed to update default index source for privilege monitoring: ${e.message}`);
          this.audit(_actions.PrivilegeMonitoringEngineActions.INIT, _common.EngineComponentResourceEnum.privmon_engine, 'Failed to update default index source for privilege monitoring', e);
        }
      } else {
        this.log('info', 'Creating default index source for privilege monitoring.');
        try {
          const indexSourceDescriptor = this.monitoringIndexSourceClient.create(defaultIndexSource);
          this.log('debug', `Created index source for privilege monitoring: ${JSON.stringify(indexSourceDescriptor)}`);
        } catch (e) {
          this.log('error', `Failed to create default index source for privilege monitoring: ${e.message}`);
          this.audit(_actions.PrivilegeMonitoringEngineActions.INIT, _common.EngineComponentResourceEnum.privmon_engine, 'Failed to create default index source for privilege monitoring', e);
        }
      }
    });
    this.opts = opts;
    this.esClient = opts.clusterClient.asCurrentUser;
    this.internalUserClient = opts.clusterClient.asInternalUser;
    this.apiKeyGenerator = opts.apiKeyManager;
    this.engineClient = new _saved_objects.PrivilegeMonitoringEngineDescriptorClient({
      soClient: opts.soClient,
      namespace: opts.namespace
    });
    this.monitoringIndexSourceClient = new _saved_objects.MonitoringEntitySourceDescriptorClient({
      soClient: opts.soClient,
      namespace: opts.namespace
    });
  }
  async init() {
    if (!this.opts.taskManager) {
      throw new Error('Task Manager is not available');
    }
    const setupStartTime = (0, _moment.default)().utc().toISOString();
    this.audit(_actions.PrivilegeMonitoringEngineActions.INIT, _common.EngineComponentResourceEnum.privmon_engine, 'Initializing privilege monitoring engine');
    const descriptor = await this.engineClient.init();
    this.log('debug', `Initialized privileged monitoring engine saved object`);
    await this.createOrUpdateDefaultDataSource();
    try {
      var _this$opts$telemetry;
      this.log('debug', 'Creating privilege user monitoring event.ingested pipeline');
      await this.createIngestPipelineIfDoesNotExist();
      await this.createOrUpdateIndex().catch(e => {
        if (e.meta.body.error.type === 'resource_already_exists_exception') {
          this.opts.logger.info('Privilege monitoring index already exists');
        } else {
          throw e;
        }
      });
      if (this.apiKeyGenerator) {
        await this.apiKeyGenerator.generate();
      }
      await (0, _privilege_monitoring_task.startPrivilegeMonitoringTask)({
        logger: this.opts.logger,
        namespace: this.opts.namespace,
        taskManager: this.opts.taskManager
      });
      const setupEndTime = (0, _moment.default)().utc().toISOString();
      const duration = (0, _moment.default)(setupEndTime).diff((0, _moment.default)(setupStartTime), 'seconds');
      (_this$opts$telemetry = this.opts.telemetry) === null || _this$opts$telemetry === void 0 ? void 0 : _this$opts$telemetry.reportEvent(_events.PRIVMON_ENGINE_INITIALIZATION_EVENT.eventType, {
        duration
      });
    } catch (e) {
      var _this$opts$telemetry2;
      this.log('error', `Error initializing privilege monitoring engine: ${e}`);
      this.audit(_actions.PrivilegeMonitoringEngineActions.INIT, _common.EngineComponentResourceEnum.privmon_engine, 'Failed to initialize privilege monitoring engine', e);
      (_this$opts$telemetry2 = this.opts.telemetry) === null || _this$opts$telemetry2 === void 0 ? void 0 : _this$opts$telemetry2.reportEvent(_events.PRIVMON_ENGINE_RESOURCE_INIT_FAILURE_EVENT.eventType, {
        error: e.message
      });
      await this.engineClient.update({
        status: _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.ERROR,
        error: {
          message: e.message,
          stack: e.stack,
          action: 'init'
        }
      });
    }
    return descriptor;
  }
  async delete(deleteData = false) {
    this.log('info', 'Deleting privilege monitoring engine');
    await this.engineClient.delete().catch(_helpers.ignoreSONotFoundError);
    if (deleteData) {
      await this.esClient.indices.delete({
        index: this.getIndex()
      }, {
        ignore: [404]
      });
    }
    if (!this.opts.taskManager) {
      throw new Error('Task Manager is not available');
    }
    await (0, _privilege_monitoring_task.removePrivilegeMonitoringTask)({
      logger: this.opts.logger,
      namespace: this.opts.namespace,
      taskManager: this.opts.taskManager
    });
    const allDataSources = await this.monitoringIndexSourceClient.findAll({});
    const deleteSourcePromises = allDataSources.map(so => this.monitoringIndexSourceClient.delete(so.id));
    await Promise.all(deleteSourcePromises);
    return {
      deleted: true
    };
  }
  async getEngineStatus() {
    const findResponse = await this.engineClient.find();
    const engineDescriptor = findResponse.total > 0 ? findResponse.saved_objects[0].attributes : undefined;
    if (!engineDescriptor) {
      return {
        status: _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.NOT_INSTALLED,
        error: undefined
      };
    }
    return {
      status: engineDescriptor.status,
      error: engineDescriptor.error
    };
  }
  async createOrUpdateIndex() {
    this.log('info', `Creating or updating index: ${this.getIndex()}`);
    await (0, _create_or_update_index.createOrUpdateIndex)({
      esClient: this.internalUserClient,
      logger: this.opts.logger,
      options: {
        index: this.getIndex(),
        mappings: (0, _indices.generateUserIndexMappings)(),
        settings: {
          hidden: true,
          mode: 'lookup',
          default_pipeline: _event_ingested.PRIVMON_EVENT_INGEST_PIPELINE_ID
        }
      }
    });
  }
  async doesIndexExist() {
    try {
      return await this.internalUserClient.indices.exists({
        index: this.getIndex()
      });
    } catch (e) {
      return false;
    }
  }
  async createIngestPipelineIfDoesNotExist() {
    const pipelinesResponse = await this.internalUserClient.ingest.getPipeline({
      id: _event_ingested.PRIVMON_EVENT_INGEST_PIPELINE_ID
    }, {
      ignore: [404]
    });
    if (!pipelinesResponse[_event_ingested.PRIVMON_EVENT_INGEST_PIPELINE_ID]) {
      this.log('info', 'Privileged user monitoring ingest pipeline does not exist, creating.');
      await this.internalUserClient.ingest.putPipeline(_event_ingested.eventIngestPipeline);
    } else {
      this.log('info', 'Privileged user monitoring ingest pipeline already exists.');
    }
  }

  /**
   * This creates an index for the user to populate privileged users.
   * It already defines the mappings and settings for the index.
   */
  createPrivilegesImportIndex(indexName, mode) {
    this.log('info', `Creating privileges import index: ${indexName} with mode: ${mode}`);
    // Use the current user client to create the index, the internal user does not have permissions to any index
    return this.esClient.indices.create({
      index: indexName,
      mappings: {
        properties: _indices.PRIVILEGED_MONITOR_IMPORT_USERS_INDEX_MAPPING
      },
      settings: {
        mode
      }
    });
  }
  async searchPrivilegesIndices(query) {
    var _fields$userName$key, _fields$userName, _fields$userName$keyw;
    const {
      indices,
      fields
    } = await this.esClient.fieldCaps({
      index: [query ? `*${query}*` : '*', ..._constants2.PRE_EXCLUDE_INDICES],
      types: ['keyword'],
      fields: ['user.name'],
      include_unmapped: true,
      ignore_unavailable: true,
      allow_no_indices: true,
      expand_wildcards: 'open',
      include_empty_fields: true,
      filters: '-parent'
    });
    const indicesWithUserName = (_fields$userName$key = (_fields$userName = fields['user.name']) === null || _fields$userName === void 0 ? void 0 : (_fields$userName$keyw = _fields$userName.keyword) === null || _fields$userName$keyw === void 0 ? void 0 : _fields$userName$keyw.indices) !== null && _fields$userName$key !== void 0 ? _fields$userName$key : indices;
    if (!Array.isArray(indicesWithUserName) || indicesWithUserName.length === 0) {
      return [];
    }
    return indicesWithUserName.filter(name => !_constants2.POST_EXCLUDE_INDICES.some(pattern => name.startsWith(pattern)));
  }
  getIndex() {
    return (0, _utils.getPrivilegedMonitorUsersIndex)(this.opts.namespace);
  }
  async createUser(user, source) {
    const doc = (0, _lodash.merge)(user, {
      user: {
        is_privileged: true
      },
      labels: {
        sources: [source]
      }
    });
    const res = await this.esClient.index({
      index: this.getIndex(),
      refresh: 'wait_for',
      document: doc
    });
    const newUser = await this.getUser(res._id);
    if (!newUser) {
      throw new Error(`Failed to create user: ${res._id}`);
    }
    return newUser;
  }
  async getUser(id) {
    const response = await this.esClient.get({
      index: this.getIndex(),
      id
    });
    return response.found ? {
      ...response._source,
      id: response._id
    } : undefined;
  }
  async updateUser(id, user) {
    await this.esClient.update({
      index: this.getIndex(),
      refresh: 'wait_for',
      id,
      doc: user
    });
    return this.getUser(id);
  }
  async deleteUser(id) {
    await this.esClient.delete({
      index: this.getIndex(),
      id
    });
  }
  async listUsers(kuery) {
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    const response = await this.esClient.search({
      size: 10000,
      index: this.getIndex(),
      query
    });
    return response.hits.hits.map(hit => ({
      id: hit._id,
      ...hit._source
    }));
  }
  async uploadUsersCSV(stream, {
    retries,
    flushBytes
  }) {
    const csvStream = _papaparse.default.parse(_papaparse.default.NODE_STREAM_INPUT, {
      header: false,
      dynamicTyping: true,
      skipEmptyLines: true
    });
    const batches = _stream.Readable.from(stream.pipe(csvStream)).pipe((0, _privileged_user_parse_transform.privilegedUserParserTransform)()).pipe((0, _batching.batchPartitions)(100));
    let results = {
      users: [],
      errors: [],
      failed: 0,
      successful: 0
    };
    for await (const batch of batches) {
      const usrs = await (0, _query_existing_users.queryExistingUsers)(this.esClient, this.getIndex())(batch);
      const upserted = await (0, _upsert_batch.bulkUpsertBatch)(this.esClient, this.getIndex(), {
        flushBytes,
        retries
      })(usrs);
      results = (0, _utils2.accumulateUpsertResults)({
        users: [],
        errors: [],
        failed: 0,
        successful: 0
      }, upserted);
    }
    const softDeletedResults = await (0, _soft_delete_omitted_users.softDeleteOmittedUsers)(this.esClient, this.getIndex(), {
      flushBytes,
      retries
    })(results);
    return {
      errors: softDeletedResults.updated.errors.concat(softDeletedResults.deleted.errors),
      stats: {
        failed: softDeletedResults.updated.failed + softDeletedResults.deleted.failed,
        successful: softDeletedResults.updated.successful + softDeletedResults.deleted.successful,
        total: softDeletedResults.updated.failed + softDeletedResults.updated.successful + softDeletedResults.deleted.failed + softDeletedResults.deleted.successful
      }
    };
  }
  log(level, msg) {
    this.opts.logger[level](`[Privileged Monitoring Engine][namespace: ${this.opts.namespace}] ${msg}`);
  }
  audit(action, resource, msg, error) {
    var _this$opts$auditLogge;
    // NOTE: Excluding errors, all auditing events are currently WRITE events, meaning the outcome is always UNKNOWN.
    // This may change in the future, depending on the audit action.
    const outcome = error ? _audit.AUDIT_OUTCOME.FAILURE : _audit.AUDIT_OUTCOME.UNKNOWN;
    const type = action === _actions.PrivilegeMonitoringEngineActions.CREATE ? _audit.AUDIT_TYPE.CREATION : _actions.PrivilegeMonitoringEngineActions.DELETE ? _audit.AUDIT_TYPE.DELETION : _audit.AUDIT_TYPE.CHANGE;
    const category = _audit.AUDIT_CATEGORY.DATABASE;
    const message = error ? `${msg}: ${error.message}` : msg;
    const event = {
      message: `[Privilege Monitoring] ${message}`,
      event: {
        action: `${action}_${resource}`,
        category,
        outcome,
        type
      }
    };
    return (_this$opts$auditLogge = this.opts.auditLogger) === null || _this$opts$auditLogge === void 0 ? void 0 : _this$opts$auditLogge.log(event);
  }

  /**
   * Synchronizes users from monitoring index sources and soft-deletes (mark as not privileged) stale entries.
   *
   * This method:
   * - Retrieves all saved objects of type 'index' that define monitoring sources.
   * - For each valid source with an index pattern, fetches usernames from the monitoring index.
   * - Identifies users no longer present in the source index (stale users).
   * - Performs a bulk soft-delete (marks as not privileged) for all stale users found.
   * - Handles missing indices gracefully by logging a warning and skipping them.
   *
   * Additionally, all users from index sources are synced with the internal privileged user index,
   * ensuring each user is either created or updated with the latest data.
   *
   * @returns {Promise<void>} Resolves when synchronization and soft-deletion are complete.
   */
  async plainIndexSync() {
    // get all monitoring index source saved objects of type 'index'
    const indexSources = await this.monitoringIndexSourceClient.findByIndex();
    if (indexSources.length === 0) {
      this.log('debug', 'No monitoring index sources found. Skipping sync.');
      return;
    }
    const allStaleUsers = [];
    for (const source of indexSources) {
      // eslint-disable-next-line no-continue
      if (!source.indexPattern) continue; // if no index pattern, skip this source
      const index = source.indexPattern;
      try {
        var _source$filter;
        const batchUserNames = await this.syncUsernamesFromIndex({
          indexName: index,
          kuery: (_source$filter = source.filter) === null || _source$filter === void 0 ? void 0 : _source$filter.kuery
        });
        // collect stale users
        const staleUsers = await this.findStaleUsersForIndex(index, batchUserNames);
        allStaleUsers.push(...staleUsers);
      } catch (error) {
        var _error$meta, _error$meta$body, _error$meta$body$erro, _error$message;
        if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) === 'index_not_found_exception' || error !== null && error !== void 0 && (_error$message = error.message) !== null && _error$message !== void 0 && _error$message.includes('index_not_found_exception')) {
          this.log('warn', `Index "${index}" not found — skipping.`);
          // eslint-disable-next-line no-continue
          continue;
        }
        this.log('error', `Unexpected error during sync for index "${index}": ${error.message}`);
      }
    }
    // Soft delete stale users
    this.log('debug', `Found ${allStaleUsers.length} stale users across all index sources.`);
    if (allStaleUsers.length > 0) {
      const ops = this.bulkOperationsForSoftDeleteUsers(allStaleUsers, this.getIndex());
      await this.esClient.bulk({
        body: ops,
        refresh: true
      });
    }
  }

  /**
   * Synchronizes usernames from a specified index by collecting them in batches
   * and performing create or update operations in the privileged user index.
   *
   * This method:
   * - Executes a paginated search on the provided index (with optional KQL filter).
   * - Extracts `user.name` values from each document.
   * - Checks for existing monitored users to determine if each username should be created or updated.
   * - Performs bulk operations to insert or update users in the internal privileged user index.
   *
   * Designed to support large indices through pagination (`search_after`) and batching.
   * Logs each step and handles errors during bulk writes.
   *
   * @param indexName - Name of the Elasticsearch index to pull usernames from.
   * @param kuery - Optional KQL filter to narrow down results.
   * @returns A list of all usernames processed from the source index.
   */
  async syncUsernamesFromIndex({
    indexName,
    kuery
  }) {
    // let batchUniqueUsernames: string[] = [];
    const allUsernames = []; // Collect all usernames across batches
    let searchAfter;
    const batchSize = 100;
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    while (true) {
      const response = await this.searchUsernamesInIndex({
        indexName,
        batchSize,
        searchAfter,
        query
      });
      const hits = response.hits.hits;
      if (hits.length === 0) break;
      const batchUsernames = hits.map(hit => {
        var _hit$_source, _hit$_source$user;
        return (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : (_hit$_source$user = _hit$_source.user) === null || _hit$_source$user === void 0 ? void 0 : _hit$_source$user.name;
      }).filter(username => !!username);
      allUsernames.push(...batchUsernames); // Collect usernames from this batch
      const batchUniqueUsernames = (0, _lodash.uniq)(batchUsernames); // Ensure uniqueness within the batch

      this.log('debug', `Found ${batchUniqueUsernames.length} unique usernames in ${batchUsernames.length} hits.`);
      const existingUserRes = await this.getMonitoredUsers(batchUniqueUsernames);
      const existingUserMap = new Map();
      for (const hit of existingUserRes.hits.hits) {
        var _hit$_source2, _hit$_source2$user;
        const username = (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : (_hit$_source2$user = _hit$_source2.user) === null || _hit$_source2$user === void 0 ? void 0 : _hit$_source2$user.name;
        this.log('debug', `Found existing user: ${username} with ID: ${hit._id}`);
        if (username) existingUserMap.set(username, hit._id);
      }
      const usersToWrite = batchUniqueUsernames.map(username => ({
        username,
        indexName,
        existingUserId: existingUserMap.get(username)
      }));
      if (usersToWrite.length === 0) return batchUniqueUsernames;
      const ops = this.buildBulkOperationsForUsers(usersToWrite, this.getIndex());
      this.log('debug', `Executing bulk operations for ${usersToWrite.length} users`);
      try {
        this.log('debug', `Bulk ops preview:\n${JSON.stringify(ops, null, 2)}`);
        await this.esClient.bulk({
          body: ops,
          refresh: true
        });
      } catch (error) {
        this.log('error', `Error executing bulk operations: ${error}`);
      }
      searchAfter = hits[hits.length - 1].sort;
    }
    return (0, _lodash.uniq)(allUsernames); // Return all unique usernames collected across batches
  }
  async findStaleUsersForIndex(indexName, userNames) {
    const response = await this.esClient.search({
      index: this.getIndex(),
      size: 10,
      // check this
      _source: ['user.name', 'labels.source_indices'],
      query: {
        bool: {
          must: [{
            term: {
              'user.is_privileged': true
            }
          }, {
            term: {
              'labels.source_indices.keyword': indexName
            }
          }],
          must_not: {
            terms: {
              'user.name': userNames
            }
          }
        }
      }
    });
    return response.hits.hits.map(hit => {
      var _hit$_source$user$nam, _hit$_source3, _hit$_source3$user;
      return {
        username: (_hit$_source$user$nam = (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : (_hit$_source3$user = _hit$_source3.user) === null || _hit$_source3$user === void 0 ? void 0 : _hit$_source3$user.name) !== null && _hit$_source$user$nam !== void 0 ? _hit$_source$user$nam : 'unknown',
        existingUserId: hit._id,
        indexName
      };
    });
  }
  async getMonitoredUsers(batchUsernames) {
    return this.esClient.search({
      index: this.getIndex(),
      size: batchUsernames.length,
      query: {
        bool: {
          must: [{
            terms: {
              'user.name': batchUsernames
            }
          }]
        }
      }
    });
  }

  /**
   * Builds a list of Elasticsearch bulk operations to upsert privileged users.
   *
   * For each user:
   * - If the user already exists (has an ID), generates an `update` operation using a Painless script
   *   to append the index name to `labels.source_indices` and ensure `'index'` is listed in `labels.sources`.
   * - If the user is new, generates an `index` operation to create a new document with default labels.
   *
   * Logs key steps during operation generation and returns the bulk operations array, ready for submission to the ES Bulk API.
   *
   * @param users - List of users to create or update.
   * @param userIndexName - Name of the Elasticsearch index where user documents are stored.
   * @returns An array of bulk operations suitable for the Elasticsearch Bulk API.
   */
  buildBulkOperationsForUsers(users, userIndexName) {
    const ops = [];
    this.log('info', `Building bulk operations for ${users.length} users`);
    for (const user of users) {
      if (user.existingUserId) {
        // Update user with painless script
        this.log('info', `Updating existing user: ${user.username} with ID: ${user.existingUserId}`);
        ops.push({
          update: {
            _index: userIndexName,
            _id: user.existingUserId
          }
        }, {
          script: {
            source: `
              if (!ctx._source.labels.source_indices.contains(params.index)) {
                ctx._source.labels.source_indices.add(params.index);
              }
              if (!ctx._source.labels.sources.contains("index")) {
                ctx._source.labels.sources.add("index");
              }
            `,
            params: {
              index: user.indexName
            }
          }
        });
      } else {
        // New user — create
        this.log('info', `Creating new user: ${user.username} with index: ${user.indexName}`);
        ops.push({
          index: {
            _index: userIndexName
          }
        }, {
          user: {
            name: user.username,
            is_privileged: true
          },
          labels: {
            sources: ['index'],
            source_indices: [user.indexName]
          }
        });
      }
    }
    this.log('info', `Built ${ops.length} bulk operations for users`);
    return ops;
  }

  /**
   * Builds bulk operations to soft-delete users by updating their privilege status.
   *
   * For each user:
   * - Removes the specified `index` from `labels.source_indices`.
   * - If no source indices remain, removes `'index'` from `labels.sources`.
   * - If no sources remain, sets `user.is_privileged` to `false`, effectively marking the user as no longer privileged.
   *
   * These operations are used to clean up users that are no longer found in the associated index sources
   * without deleting their documents entirely.
   *
   * @param users - Users to be soft-deleted based on missing index source association.
   * @param userIndexName - The Elasticsearch index where user documents are stored.
   * @returns An array of bulk update operations compatible with the Elasticsearch Bulk API.
   */
  bulkOperationsForSoftDeleteUsers(users, userIndexName) {
    const ops = [];
    this.log('info', `Building bulk operations for soft delete users`);
    for (const user of users) {
      ops.push({
        update: {
          _index: userIndexName,
          _id: user.existingUserId
        }
      }, {
        script: {
          source: `
            if (ctx._source.labels?.source_indices != null) {
              ctx._source.labels.source_indices.removeIf(idx -> idx == params.index);
            }

            if (ctx._source.labels?.source_indices == null || ctx._source.labels.source_indices.isEmpty()) {
              if (ctx._source.labels?.sources != null) {
                ctx._source.labels.sources.removeIf(src -> src == 'index');
              }
            }

            if (ctx._source.labels?.sources == null || ctx._source.labels.sources.isEmpty()) {
              ctx._source.user.is_privileged = false;
            }
          `,
          params: {
            index: user.indexName
          }
        }
      });
    }
    return ops;
  }
  async searchUsernamesInIndex({
    indexName,
    batchSize,
    searchAfter,
    query
  }) {
    return this.esClient.search({
      index: indexName,
      size: batchSize,
      _source: ['user.name'],
      sort: [{
        'user.name': 'asc'
      }],
      search_after: searchAfter,
      query
    });
  }
  async disable() {
    this.log('info', 'Disabling Privileged Monitoring Engine');
    // Check the current status of the engine
    const currentEngineStatus = await this.getEngineStatus();
    if (currentEngineStatus.status !== _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED) {
      this.log('info', 'Privilege Monitoring Engine is not in STARTED state, skipping disable operation');
      return {
        status: currentEngineStatus.status,
        error: null
      };
    }
    try {
      // 1. Remove the privileged user monitoring task
      if (!this.opts.taskManager) {
        throw new Error('Task Manager is not available');
      }
      this.log('debug', 'Disabling Privileged Monitoring Engine: removing task');
      await (0, _privilege_monitoring_task.removePrivilegeMonitoringTask)({
        logger: this.opts.logger,
        namespace: this.opts.namespace,
        taskManager: this.opts.taskManager
      });

      // 2. Update status in Saved Objects
      this.log('debug', 'Disabling Privileged Monitoring Engine: Updating status to DISABLED in Saved Objects');
      await this.engineClient.updateStatus(_constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.DISABLED);
      this.audit(_actions.PrivilegeMonitoringEngineActions.DISABLE, _common.EngineComponentResourceEnum.privmon_engine, 'Privilege Monitoring Engine disabled');
      this.log('info', 'Privileged Monitoring Engine disabled successfully');
      return {
        status: _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.DISABLED,
        error: null
      };
    } catch (e) {
      const msg = `Failed to disable Privileged Monitoring Engine: ${e.message}`;
      this.log('error', msg);
      this.audit(_actions.PrivilegeMonitoringEngineActions.DISABLE, _common.EngineComponentResourceEnum.privmon_engine, 'Failed to disable Privileged Monitoring Engine', e);
      throw new Error(msg);
    }
  }
  async scheduleNow() {
    if (!this.opts.taskManager) {
      throw new Error('Task Manager is not available');
    }
    const engineStatus = await this.getEngineStatus();
    if (engineStatus.status !== _constants2.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED) {
      throw new Error(`The Privileged Monitoring Engine must be enabled to schedule a run. Current status: ${engineStatus.status}.`);
    }
    this.audit(_actions.PrivilegeMonitoringEngineActions.SCHEDULE_NOW, _common.EngineComponentResourceEnum.privmon_engine, 'Privilege Monitoring Engine scheduled for immediate run');
    return (0, _privilege_monitoring_task.scheduleNow)({
      taskManager: this.opts.taskManager,
      namespace: this.opts.namespace,
      logger: this.opts.logger
    });
  }
}
exports.PrivilegeMonitoringDataClient = PrivilegeMonitoringDataClient;