"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stringifyAfterKey = exports.racFieldMappings = exports.parseInterval = exports.mergeReturns = exports.makeFloatString = exports.isWrappedSignalHit = exports.isWrappedEventHit = exports.isWrappedDetectionAlert = exports.isThresholdParams = exports.isThreatParams = exports.isQueryParams = exports.isMachineLearningParams = exports.isEsqlParams = exports.isEqlParams = exports.isDetectionAlert = exports.hasTimestampFields = exports.hasReadIndexPrivileges = exports.getValidDateFromDoc = exports.getUnprocessedExceptionsWarnings = exports.getTotalHitsValue = exports.getSuppressionMaxSignalsWarning = exports.getSafeSortIds = exports.getRuleRangeTuples = exports.getNumCatchupIntervals = exports.getMaxSignalsWarning = exports.getGapBetweenRuns = exports.getField = exports.getExceptions = exports.getDisabledActionsWarningText = exports.getCatchupTuples = exports.generateId = exports.createSearchAfterReturnTypeFromResponse = exports.createSearchAfterReturnType = exports.createErrorsFromShard = exports.checkPrivilegesFromEsClient = exports.checkForFrozenIndices = exports.calculateTotal = exports.calculateFromValue = exports.buildShellAlertSuppressionTermsAndFields = exports.addToSearchAfterReturn = exports.MAX_RULE_GAP_RATIO = void 0;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _crypto = require("crypto");
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _objectHash = _interopRequireDefault(require("object-hash"));
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _esQuery = require("@kbn/es-query");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _server = require("@kbn/alerting-plugin/server");
var _rule_monitoring = require("../../../../../common/api/detection_engine/rule_monitoring");
var _with_security_span = require("../../../../utils/with_security_span");
var _constants = require("../../../../../common/constants");
var _suppression_utils = require("./suppression_utils");
var _robust_field_access = require("./source_fields_merging/utils/robust_field_access");
var _apm_field_names = require("./apm_field_names");
var _build_events_query = require("./build_events_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RULE_GAP_RATIO = exports.MAX_RULE_GAP_RATIO = 4;
const hasReadIndexPrivileges = async args => {
  const {
    privileges,
    ruleExecutionLogger,
    uiSettingsClient,
    docLinks
  } = args;
  const apiKeyDocs = docLinks.links.alerting.authorization;
  const isCcsPermissionWarningEnabled = await uiSettingsClient.get(_constants.ENABLE_CCS_READ_WARNING_SETTING);
  const indexNames = Object.keys(privileges.index);
  const filteredIndexNames = isCcsPermissionWarningEnabled ? indexNames : indexNames.filter(indexName => {
    return !(0, _esQuery.isCCSRemoteIndexName)(indexName);
  });
  const [, indexesWithNoReadPrivileges] = (0, _lodash.partition)(filteredIndexNames, indexName => privileges.index[indexName].read);
  let warningStatusMessage;

  // Some indices have read privileges others do not.
  if (indexesWithNoReadPrivileges.length > 0) {
    const indexesString = JSON.stringify(indexesWithNoReadPrivileges);
    warningStatusMessage = `This rule's API key is unable to access all indices that match the ${indexesString} pattern. To learn how to update and manage API keys, refer to ${apiKeyDocs}.`;
    await ruleExecutionLogger.logStatusChange({
      newStatus: _rule_monitoring.RuleExecutionStatusEnum['partial failure'],
      message: warningStatusMessage
    });
  }
  return warningStatusMessage;
};
exports.hasReadIndexPrivileges = hasReadIndexPrivileges;
const hasTimestampFields = async args => {
  var _timestampFieldCapsRe, _timestampFieldCapsRe2, _timestampFieldCapsRe3;
  const {
    timestampField,
    timestampFieldCapsResponse,
    inputIndices,
    ruleExecutionLogger
  } = args;
  const {
    ruleName
  } = ruleExecutionLogger.context;
  _elasticApmNode.default.setCustomContext({
    [_apm_field_names.SECURITY_NUM_INDICES_MATCHING_PATTERN]: (_timestampFieldCapsRe = timestampFieldCapsResponse.body.indices) === null || _timestampFieldCapsRe === void 0 ? void 0 : _timestampFieldCapsRe.length
  });
  if ((0, _lodash.isEmpty)(timestampFieldCapsResponse.body.indices)) {
    const errorString = `This rule is attempting to query data from Elasticsearch indices listed in the "Index patterns" section of the rule definition, however no index matching: ${JSON.stringify(inputIndices)} was found. This warning will continue to appear until a matching index is created or this rule is disabled. ${ruleName === 'Endpoint Security' ? 'If you have recently enrolled agents enabled with Endpoint Security through Fleet, this warning should stop once an alert is sent from an agent.' : ''}`;
    await ruleExecutionLogger.logStatusChange({
      newStatus: _rule_monitoring.RuleExecutionStatusEnum['partial failure'],
      message: errorString.trimEnd()
    });
    return {
      foundNoIndices: true,
      warningMessage: errorString.trimEnd()
    };
  } else if ((0, _lodash.isEmpty)(timestampFieldCapsResponse.body.fields) || timestampFieldCapsResponse.body.fields[timestampField] == null || ((_timestampFieldCapsRe2 = timestampFieldCapsResponse.body.fields[timestampField]) === null || _timestampFieldCapsRe2 === void 0 ? void 0 : (_timestampFieldCapsRe3 = _timestampFieldCapsRe2.unmapped) === null || _timestampFieldCapsRe3 === void 0 ? void 0 : _timestampFieldCapsRe3.indices) != null) {
    var _timestampFieldCapsRe4, _timestampFieldCapsRe5;
    // if there is a timestamp override and the unmapped array for the timestamp override key is not empty,
    // warning
    const errorString = `The following indices are missing the ${timestampField === '@timestamp' ? 'timestamp field "@timestamp"' : `timestamp override field "${timestampField}"`}: ${JSON.stringify((0, _lodash.isEmpty)(timestampFieldCapsResponse.body.fields) || (0, _lodash.isEmpty)(timestampFieldCapsResponse.body.fields[timestampField]) ? timestampFieldCapsResponse.body.indices : (_timestampFieldCapsRe4 = timestampFieldCapsResponse.body.fields[timestampField]) === null || _timestampFieldCapsRe4 === void 0 ? void 0 : (_timestampFieldCapsRe5 = _timestampFieldCapsRe4.unmapped) === null || _timestampFieldCapsRe5 === void 0 ? void 0 : _timestampFieldCapsRe5.indices)}`;
    await ruleExecutionLogger.logStatusChange({
      newStatus: _rule_monitoring.RuleExecutionStatusEnum['partial failure'],
      message: errorString
    });
    return {
      foundNoIndices: false,
      warningMessage: errorString
    };
  }
  return {
    foundNoIndices: false,
    warningMessage: undefined
  };
};

/**
 * Identifies frozen indices from the provided input indices.
 * If any of the input indices resolve to frozen indices within the specified time range, they are returned by this function.
 * @param {string[]} params.inputIndices - The list of input index patterns or indices to check.
 * @param {ElasticsearchClient} params.internalEsClient - A client to be used to query the elasticsearch cluster on behalf of the internal Kibana user.
 * @param {ElasticsearchClient} params.currentUserEsClient - A client to be used to query the elasticsearch cluster on behalf of the user that initiated the request to the Kibana server.
 * @param {string} params.to - The end of the time range for the query (e.g., "now").
 * @param {string} params.from - The start of the time range for the query (e.g., "now-1d").
 * @param {string} params.primaryTimestamp - The primary timestamp field used for filtering.
 * @param {string | undefined} params.secondaryTimestamp - The secondary timestamp field used for filtering, if applicable.
 * @returns {Promise<string[]>} A promise that resolves to a list of frozen indices.
 */
exports.hasTimestampFields = hasTimestampFields;
const checkForFrozenIndices = async ({
  inputIndices,
  internalEsClient,
  currentUserEsClient,
  to,
  from,
  primaryTimestamp,
  secondaryTimestamp
}) => {
  const fieldCapsResponse = await currentUserEsClient.fieldCaps({
    index: inputIndices,
    fields: ['_id'],
    ignore_unavailable: true,
    index_filter: (0, _build_events_query.buildTimeRangeFilter)({
      to,
      from,
      primaryTimestamp,
      secondaryTimestamp
    })
  });
  const resolvedQueryIndices = (0, _lodash.isArray)(fieldCapsResponse.indices) ? fieldCapsResponse.indices : [fieldCapsResponse.indices];

  // Frozen indices start with `partial-`, but it's possible
  // for some regular hot/warm index to start with that prefix as well by coincidence. If we find indices with that naming pattern,
  // we fetch information about the index using ilm explain to verify that they are actually frozen indices.
  const partialIndices = resolvedQueryIndices.filter(index => index.startsWith('partial-'));
  if (partialIndices.length <= 0) {
    return [];
  }
  const explainResponse = await internalEsClient.ilm.explainLifecycle({
    // Use the original index patterns again instead of just the concrete names of the indices:
    // the list of concrete indices could be huge and make the request URL too large, but we know the list of index patterns works
    index: inputIndices.join(','),
    filter_path: 'indices.*.phase,indices.*.managed'
  });
  return partialIndices.filter(index => {
    const indexResponse = explainResponse.indices[index];
    return indexResponse !== undefined && indexResponse.managed && indexResponse.phase === 'frozen';
  });
};
exports.checkForFrozenIndices = checkForFrozenIndices;
const checkPrivilegesFromEsClient = async (esClient, indices) => (0, _with_security_span.withSecuritySpan)('checkPrivilegesFromEsClient', async () => await esClient.transport.request({
  path: '/_security/user/_has_privileges',
  method: 'POST',
  body: {
    index: [{
      names: indices !== null && indices !== void 0 ? indices : [],
      allow_restricted_indices: true,
      privileges: ['read']
    }]
  }
}));
exports.checkPrivilegesFromEsClient = checkPrivilegesFromEsClient;
const getNumCatchupIntervals = ({
  gap,
  intervalDuration
}) => {
  if (gap.asMilliseconds() <= 0 || intervalDuration.asMilliseconds() <= 0) {
    return 0;
  }
  const ratio = Math.ceil(gap.asMilliseconds() / intervalDuration.asMilliseconds());
  // maxCatchup is to ensure we are not trying to catch up too far back.
  // This allows for a maximum of 4 consecutive rule execution misses
  // to be included in the number of signals generated.
  return ratio < MAX_RULE_GAP_RATIO ? ratio : MAX_RULE_GAP_RATIO;
};
exports.getNumCatchupIntervals = getNumCatchupIntervals;
const getExceptions = async ({
  client,
  lists
}) => {
  return (0, _with_security_span.withSecuritySpan)('getExceptions', async () => {
    if (lists.length > 0) {
      try {
        const listIds = lists.map(({
          list_id: listId
        }) => listId);
        const namespaceTypes = lists.map(({
          namespace_type: namespaceType
        }) => namespaceType);

        // Stream the results from the Point In Time (PIT) finder into this array
        let items = [];
        const executeFunctionOnStream = response => {
          items = [...items, ...response.data];
        };
        await client.findExceptionListsItemPointInTimeFinder({
          executeFunctionOnStream,
          listId: listIds,
          namespaceType: namespaceTypes,
          perPage: 1_000,
          // See https://github.com/elastic/kibana/issues/93770 for choice of 1k
          filter: [],
          maxSize: undefined,
          // NOTE: This is unbounded when it is "undefined"
          sortOrder: undefined,
          sortField: undefined
        });
        _elasticApmNode.default.setCustomContext({
          [_apm_field_names.SECURITY_NUM_EXCEPTION_ITEMS]: items.length
        });
        return items;
      } catch (e) {
        throw new Error(`unable to fetch exception list items, message: "${e.message}" full error: "${e}"`);
      }
    } else {
      return [];
    }
  });
};
exports.getExceptions = getExceptions;
const generateId = (docIndex, docId, version, ruleId) => (0, _crypto.createHash)('sha256').update(docIndex.concat(docId, version, ruleId)).digest('hex');
exports.generateId = generateId;
const parseInterval = intervalString => {
  try {
    return _moment.default.duration((0, _server.parseDuration)(intervalString));
  } catch (err) {
    return null;
  }
};
exports.parseInterval = parseInterval;
const getGapBetweenRuns = ({
  previousStartedAt,
  originalFrom,
  originalTo,
  startedAt
}) => {
  if (previousStartedAt == null) {
    return _moment.default.duration(0);
  }
  const driftTolerance = _moment.default.duration(originalTo.diff(originalFrom));
  _elasticApmNode.default.addLabels({
    [_apm_field_names.SECURITY_QUERY_SPAN_S]: driftTolerance.asSeconds()
  }, false);
  const currentDuration = _moment.default.duration((0, _moment.default)(startedAt).diff(previousStartedAt));
  return currentDuration.subtract(driftTolerance);
};
exports.getGapBetweenRuns = getGapBetweenRuns;
const makeFloatString = num => Number(num).toFixed(2);
exports.makeFloatString = makeFloatString;
const getRuleRangeTuples = async ({
  startedAt,
  previousStartedAt,
  from,
  to,
  interval,
  maxSignals,
  ruleExecutionLogger,
  alerting
}) => {
  const originalFrom = _datemath.default.parse(from, {
    forceNow: startedAt
  });
  const originalTo = _datemath.default.parse(to, {
    forceNow: startedAt
  });
  let warningStatusMessage;
  if (originalFrom == null || originalTo == null) {
    throw new Error('Failed to parse date math of rule.from or rule.to');
  }
  const maxAlertsAllowed = alerting.getConfig().run.alerts.max;
  let maxSignalsToUse = maxSignals;
  if (maxSignals > maxAlertsAllowed) {
    maxSignalsToUse = maxAlertsAllowed;
    warningStatusMessage = `The rule's max alerts per run setting (${maxSignals}) is greater than the Kibana alerting limit (${maxAlertsAllowed}). The rule will only write a maximum of ${maxAlertsAllowed} alerts per rule run.`;
    await ruleExecutionLogger.logStatusChange({
      newStatus: _rule_monitoring.RuleExecutionStatusEnum['partial failure'],
      message: warningStatusMessage
    });
  }
  const tuples = [{
    to: originalTo,
    from: originalFrom,
    maxSignals: maxSignalsToUse
  }];
  const intervalDuration = parseInterval(interval);
  if (intervalDuration == null) {
    ruleExecutionLogger.error(`Failed to compute gap between rule runs: could not parse rule interval "${JSON.stringify(interval)}"`);
    return {
      tuples,
      remainingGap: _moment.default.duration(0),
      warningStatusMessage,
      originalFrom,
      originalTo
    };
  }
  const gap = getGapBetweenRuns({
    previousStartedAt,
    originalTo,
    originalFrom,
    startedAt
  });
  const catchup = getNumCatchupIntervals({
    gap,
    intervalDuration
  });
  const catchupTuples = getCatchupTuples({
    originalTo,
    originalFrom,
    ruleParamsMaxSignals: maxSignalsToUse,
    catchup,
    intervalDuration
  });
  tuples.push(...catchupTuples);

  // Each extra tuple adds one extra intervalDuration to the time range this rule will cover.
  const remainingGapMilliseconds = Math.max(gap.asMilliseconds() - catchup * intervalDuration.asMilliseconds(), 0);
  let gapRange;
  if (remainingGapMilliseconds > 0 && previousStartedAt != null) {
    gapRange = {
      gte: previousStartedAt.toISOString(),
      lte: (0, _moment.default)(previousStartedAt).add(remainingGapMilliseconds).toDate().toISOString()
    };
  }
  return {
    tuples: tuples.reverse(),
    remainingGap: _moment.default.duration(remainingGapMilliseconds),
    warningStatusMessage,
    gap: gapRange,
    originalFrom,
    originalTo
  };
};

/**
 * Creates rule range tuples needed to cover gaps since the last rule run.
 * @param to moment.Moment representing the rules 'to' property
 * @param from moment.Moment representing the rules 'from' property
 * @param ruleParamsMaxSignals int representing the maxSignals property on the rule (usually unmodified at 100)
 * @param catchup number the number of additional rule run intervals to add
 * @param intervalDuration moment.Duration the interval which the rule runs
 */
exports.getRuleRangeTuples = getRuleRangeTuples;
const getCatchupTuples = ({
  originalTo,
  originalFrom,
  ruleParamsMaxSignals,
  catchup,
  intervalDuration
}) => {
  const catchupTuples = [];
  const intervalInMilliseconds = intervalDuration.asMilliseconds();
  let currentTo = originalTo;
  let currentFrom = originalFrom;
  // This loop will create tuples with overlapping time ranges, the same way rule runs have overlapping time
  // ranges due to the additional lookback. We could choose to create tuples that don't overlap here by using the
  // "from" value from one tuple as "to" in the next one, however, the overlap matters for rule types like EQL and
  // threshold rules that look for sets of documents within the query. Thus we keep the overlap so that these
  // extra tuples behave as similarly to the regular rule runs as possible.
  while (catchupTuples.length < catchup) {
    const nextTo = currentTo.clone().subtract(intervalInMilliseconds);
    const nextFrom = currentFrom.clone().subtract(intervalInMilliseconds);
    catchupTuples.push({
      to: nextTo,
      from: nextFrom,
      maxSignals: ruleParamsMaxSignals
    });
    currentTo = nextTo;
    currentFrom = nextFrom;
  }
  return catchupTuples;
};

/**
 * Takes the rule schedule fields `interval` and `lookback` and uses them to calculate the `from` value for a rule
 *
 * @param interval string representing the interval on which the rule runs
 * @param lookback string representing the rule's additional lookback
 * @returns string representing the rule's 'from' property
 */
exports.getCatchupTuples = getCatchupTuples;
const calculateFromValue = (interval, lookback) => {
  var _parseInterval, _parseInterval2;
  const parsedInterval = (_parseInterval = parseInterval(interval)) !== null && _parseInterval !== void 0 ? _parseInterval : _moment.default.duration(0);
  const parsedFrom = (_parseInterval2 = parseInterval(lookback)) !== null && _parseInterval2 !== void 0 ? _parseInterval2 : _moment.default.duration(0);
  const duration = parsedFrom.asSeconds() + parsedInterval.asSeconds();
  return `now-${duration}s`;
};

/**
 * Given errors from a search query this will return an array of strings derived from the errors.
 * @param errors The errors to derive the strings from
 */
exports.calculateFromValue = calculateFromValue;
const createErrorsFromShard = ({
  errors
}) => {
  return errors.map(error => {
    const {
      index,
      reason: {
        reason,
        type,
        caused_by: {
          reason: causedByReason,
          type: causedByType
        } = {
          reason: undefined,
          type: undefined
        }
      } = {}
    } = error;
    return [...(index != null ? [`index: "${index}"`] : []), ...(reason != null ? [`reason: "${reason}"`] : []), ...(type != null ? [`type: "${type}"`] : []), ...(causedByReason != null ? [`caused by reason: "${causedByReason}"`] : []), ...(causedByType != null ? [`caused by type: "${causedByType}"`] : [])].join(' ');
  });
};

/**
 * Given a search hit this will return a valid last date if it can find one, otherwise it
 * will return undefined. This tries the "fields" first to get a formatted date time if it can, but if
 * it cannot it will resort to using the "_source" fields second which can be problematic if the date time
 * is not correctly ISO8601 or epoch milliseconds formatted.
 * @param searchResult The result to try and parse out the timestamp.
 * @param primaryTimestamp The primary timestamp to use.
 */
exports.createErrorsFromShard = createErrorsFromShard;
const getValidDateFromDoc = ({
  doc,
  primaryTimestamp
}) => {
  const timestampValue = doc.fields != null && doc.fields[primaryTimestamp] != null ? doc.fields[primaryTimestamp][0] : doc._source != null ? doc._source[primaryTimestamp] : undefined;
  const lastTimestamp = typeof timestampValue === 'string' || typeof timestampValue === 'number' ? timestampValue : undefined;
  if (lastTimestamp != null) {
    const tempMoment = (0, _moment.default)(lastTimestamp);
    if (tempMoment.isValid()) {
      return tempMoment.toDate();
    } else if (typeof timestampValue === 'string') {
      // worst case we have a string from fields API or other areas of Elasticsearch that have given us a number as a string,
      // so we try one last time to parse this best we can by converting from string to a number
      const maybeDate = (0, _moment.default)(+lastTimestamp);
      if (maybeDate.isValid()) {
        return maybeDate.toDate();
      } else {
        return undefined;
      }
    } else {
      return undefined;
    }
  }
};
exports.getValidDateFromDoc = getValidDateFromDoc;
const createSearchAfterReturnTypeFromResponse = ({
  searchResult,
  primaryTimestamp
}) => {
  var _searchResult$_shards;
  return createSearchAfterReturnType({
    success: searchResult._shards.failed === 0 || ((_searchResult$_shards = searchResult._shards.failures) === null || _searchResult$_shards === void 0 ? void 0 : _searchResult$_shards.every(failure => {
      var _failure$reason, _failure$reason$reaso, _failure$reason2, _failure$reason2$reas;
      return ((_failure$reason = failure.reason) === null || _failure$reason === void 0 ? void 0 : (_failure$reason$reaso = _failure$reason.reason) === null || _failure$reason$reaso === void 0 ? void 0 : _failure$reason$reaso.includes('No mapping found for [@timestamp] in order to sort on')) || ((_failure$reason2 = failure.reason) === null || _failure$reason2 === void 0 ? void 0 : (_failure$reason2$reas = _failure$reason2.reason) === null || _failure$reason2$reas === void 0 ? void 0 : _failure$reason2$reas.includes(`No mapping found for [${primaryTimestamp}] in order to sort on`));
    }))
  });
};
exports.createSearchAfterReturnTypeFromResponse = createSearchAfterReturnTypeFromResponse;
const createSearchAfterReturnType = ({
  success,
  warning,
  searchAfterTimes,
  enrichmentTimes,
  bulkCreateTimes,
  createdSignalsCount,
  createdSignals,
  errors,
  warningMessages,
  suppressedAlertsCount
} = {}) => {
  return {
    success: success !== null && success !== void 0 ? success : true,
    warning: warning !== null && warning !== void 0 ? warning : false,
    searchAfterTimes: searchAfterTimes !== null && searchAfterTimes !== void 0 ? searchAfterTimes : [],
    enrichmentTimes: enrichmentTimes !== null && enrichmentTimes !== void 0 ? enrichmentTimes : [],
    bulkCreateTimes: bulkCreateTimes !== null && bulkCreateTimes !== void 0 ? bulkCreateTimes : [],
    createdSignalsCount: createdSignalsCount !== null && createdSignalsCount !== void 0 ? createdSignalsCount : 0,
    createdSignals: createdSignals !== null && createdSignals !== void 0 ? createdSignals : [],
    errors: errors !== null && errors !== void 0 ? errors : [],
    warningMessages: warningMessages !== null && warningMessages !== void 0 ? warningMessages : [],
    suppressedAlertsCount: suppressedAlertsCount !== null && suppressedAlertsCount !== void 0 ? suppressedAlertsCount : 0
  };
};

/**
 * Merges the return values from bulk creating alerts into the appropriate fields in the combined return object.
 */
exports.createSearchAfterReturnType = createSearchAfterReturnType;
const addToSearchAfterReturn = ({
  current,
  next
}) => {
  current.success = current.success && next.success;
  current.createdSignalsCount += next.createdItemsCount;
  current.createdSignals.push(...next.createdItems);
  current.bulkCreateTimes.push(next.bulkCreateDuration);
  current.enrichmentTimes.push(next.enrichmentDuration);
  current.errors = [...new Set([...current.errors, ...next.errors])];
  if (next.suppressedItemsCount != null) {
    var _current$suppressedAl;
    current.suppressedAlertsCount = ((_current$suppressedAl = current.suppressedAlertsCount) !== null && _current$suppressedAl !== void 0 ? _current$suppressedAl : 0) + next.suppressedItemsCount;
  }
};
exports.addToSearchAfterReturn = addToSearchAfterReturn;
const mergeReturns = searchAfters => {
  return searchAfters.reduce((prev, next) => {
    const {
      success: existingSuccess,
      warning: existingWarning,
      searchAfterTimes: existingSearchAfterTimes,
      bulkCreateTimes: existingBulkCreateTimes,
      enrichmentTimes: existingEnrichmentTimes,
      createdSignalsCount: existingCreatedSignalsCount,
      createdSignals: existingCreatedSignals,
      errors: existingErrors,
      warningMessages: existingWarningMessages,
      suppressedAlertsCount: existingSuppressedAlertsCount
    } = prev;
    const {
      success: newSuccess,
      warning: newWarning,
      searchAfterTimes: newSearchAfterTimes,
      enrichmentTimes: newEnrichmentTimes,
      bulkCreateTimes: newBulkCreateTimes,
      createdSignalsCount: newCreatedSignalsCount,
      createdSignals: newCreatedSignals,
      errors: newErrors,
      warningMessages: newWarningMessages,
      suppressedAlertsCount: newSuppressedAlertsCount
    } = next;
    return {
      success: existingSuccess && newSuccess,
      warning: existingWarning || newWarning,
      searchAfterTimes: [...existingSearchAfterTimes, ...newSearchAfterTimes],
      enrichmentTimes: [...existingEnrichmentTimes, ...newEnrichmentTimes],
      bulkCreateTimes: [...existingBulkCreateTimes, ...newBulkCreateTimes],
      createdSignalsCount: existingCreatedSignalsCount + newCreatedSignalsCount,
      createdSignals: [...existingCreatedSignals, ...newCreatedSignals],
      errors: [...new Set([...existingErrors, ...newErrors])],
      warningMessages: [...existingWarningMessages, ...newWarningMessages],
      suppressedAlertsCount: (existingSuppressedAlertsCount !== null && existingSuppressedAlertsCount !== void 0 ? existingSuppressedAlertsCount : 0) + (newSuppressedAlertsCount !== null && newSuppressedAlertsCount !== void 0 ? newSuppressedAlertsCount : 0)
    };
  });
};
exports.mergeReturns = mergeReturns;
const getTotalHitsValue = totalHits => typeof totalHits === 'undefined' ? -1 : typeof totalHits === 'number' ? totalHits : totalHits.value;
exports.getTotalHitsValue = getTotalHitsValue;
const calculateTotal = (prevTotal, nextTotal) => {
  const prevTotalHits = getTotalHitsValue(prevTotal);
  const nextTotalHits = getTotalHitsValue(nextTotal);
  if (prevTotalHits === -1 || nextTotalHits === -1) {
    return -1;
  }
  return prevTotalHits + nextTotalHits;
};
exports.calculateTotal = calculateTotal;
const isEqlParams = params => params.type === 'eql';
exports.isEqlParams = isEqlParams;
const isEsqlParams = params => params.type === 'esql';
exports.isEsqlParams = isEsqlParams;
const isThresholdParams = params => params.type === 'threshold';
exports.isThresholdParams = isThresholdParams;
const isQueryParams = params => params.type === 'query';
exports.isQueryParams = isQueryParams;
const isThreatParams = params => params.type === 'threat_match';
exports.isThreatParams = isThreatParams;
const isMachineLearningParams = params => params.type === 'machine_learning';

/**
 * Prevent javascript from returning Number.MAX_SAFE_INTEGER when Elasticsearch expects
 * Java's Long.MAX_VALUE. This happens when sorting fields by date which are
 * unmapped in the provided index
 *
 * Ref: https://github.com/elastic/elasticsearch/issues/28806#issuecomment-369303620
 *
 * return stringified Long.MAX_VALUE if we receive Number.MAX_SAFE_INTEGER
 * @param sortIds estypes.SortResults | undefined
 * @returns SortResults
 */
exports.isMachineLearningParams = isMachineLearningParams;
const getSafeSortIds = sortIds => {
  return sortIds === null || sortIds === void 0 ? void 0 : sortIds.map(sortId => {
    // haven't determined when we would receive a null value for a sort id
    // but in case we do, default to sending the stringified Java max_int
    if (sortId == null || sortId === '' || Number(sortId) >= Number.MAX_SAFE_INTEGER) {
      return '9223372036854775807';
    }
    return sortId;
  });
};
exports.getSafeSortIds = getSafeSortIds;
const isWrappedEventHit = event => {
  return !isWrappedSignalHit(event) && !isWrappedDetectionAlert(event);
};
exports.isWrappedEventHit = isWrappedEventHit;
const isWrappedSignalHit = event => {
  var _source;
  return (event === null || event === void 0 ? void 0 : (_source = event._source) === null || _source === void 0 ? void 0 : _source.signal) != null;
};
exports.isWrappedSignalHit = isWrappedSignalHit;
const isWrappedDetectionAlert = event => {
  var _source2;
  return (event === null || event === void 0 ? void 0 : (_source2 = event._source) === null || _source2 === void 0 ? void 0 : _source2[_ruleDataUtils.ALERT_UUID]) != null;
};
exports.isWrappedDetectionAlert = isWrappedDetectionAlert;
const isDetectionAlert = event => {
  return (0, _lodash.get)(event, _ruleDataUtils.ALERT_UUID) != null;
};
exports.isDetectionAlert = isDetectionAlert;
const racFieldMappings = exports.racFieldMappings = {
  'signal.rule.id': _ruleDataUtils.ALERT_RULE_UUID,
  'signal.rule.description': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.description`,
  'signal.rule.filters': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.filters`,
  'signal.rule.language': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.language`,
  'signal.rule.query': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.query`,
  'signal.rule.risk_score': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.riskScore`,
  'signal.rule.severity': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.severity`,
  'signal.rule.building_block_type': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.buildingBlockType`,
  'signal.rule.namespace': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.namespace`,
  'signal.rule.note': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.note`,
  'signal.rule.license': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.license`,
  'signal.rule.output_index': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.outputIndex`,
  'signal.rule.timeline_id': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.timelineId`,
  'signal.rule.timeline_title': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.timelineTitle`,
  'signal.rule.meta': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.meta`,
  'signal.rule.rule_name_override': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.ruleNameOverride`,
  'signal.rule.timestamp_override': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.timestampOverride`,
  'signal.rule.author': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.author`,
  'signal.rule.false_positives': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.falsePositives`,
  'signal.rule.from': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.from`,
  'signal.rule.rule_id': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.ruleId`,
  'signal.rule.max_signals': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.maxSignals`,
  'signal.rule.risk_score_mapping': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.riskScoreMapping`,
  'signal.rule.severity_mapping': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.severityMapping`,
  'signal.rule.threat': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.threat`,
  'signal.rule.to': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.to`,
  'signal.rule.references': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.references`,
  'signal.rule.version': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.version`,
  'signal.rule.exceptions_list': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.exceptionsList`,
  'signal.rule.immutable': `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.immutable`
};
const getField = (event, field) => {
  if (isWrappedDetectionAlert(event)) {
    var _racFieldMappings$fie;
    const mappedField = (_racFieldMappings$fie = racFieldMappings[field]) !== null && _racFieldMappings$fie !== void 0 ? _racFieldMappings$fie : field.replace('signal', 'kibana.alert');
    const parts = mappedField.split('.');
    if (mappedField.includes(_ruleDataUtils.ALERT_RULE_PARAMETERS) && parts[parts.length - 1] !== 'parameters') {
      const params = (0, _lodash.get)(event._source, _ruleDataUtils.ALERT_RULE_PARAMETERS);
      return (0, _lodash.get)(params, parts[parts.length - 1]);
    }
    return (0, _lodash.get)(event._source, mappedField);
  } else if (isWrappedSignalHit(event)) {
    var _invert$field;
    const mappedField = (_invert$field = (0, _lodash.invert)(racFieldMappings)[field]) !== null && _invert$field !== void 0 ? _invert$field : field.replace('kibana.alert', 'signal');
    return (0, _lodash.get)(event._source, mappedField);
  } else if (isWrappedEventHit(event)) {
    return (0, _lodash.get)(event._source, field);
  }
};
exports.getField = getField;
const getUnprocessedExceptionsWarnings = unprocessedExceptions => {
  if (unprocessedExceptions.length > 0) {
    const exceptionNames = unprocessedExceptions.map(exception => exception.name);
    return `The following exceptions won't be applied to rule execution: ${exceptionNames.join(', ')}`;
  }
};
exports.getUnprocessedExceptionsWarnings = getUnprocessedExceptionsWarnings;
const getMaxSignalsWarning = () => {
  return `This rule reached the maximum alert limit for the rule execution. Some alerts were not created.`;
};
exports.getMaxSignalsWarning = getMaxSignalsWarning;
const getSuppressionMaxSignalsWarning = () => {
  return `This rule reached the maximum alert limit for the rule execution. Some alerts were not created or suppressed.`;
};
exports.getSuppressionMaxSignalsWarning = getSuppressionMaxSignalsWarning;
const getDisabledActionsWarningText = ({
  alertsCreated,
  disabledActions
}) => {
  const uniqueActionTypes = new Set(disabledActions.map(action => action.actionTypeId));
  const actionTypesJoined = [...uniqueActionTypes].join(', ');

  // This rule generated alerts but did not send external notifications because rule action connectors ${actionTypes} aren't enabled. To send notifications, you need a higher Security Analytics tier.
  const alertsGeneratedText = alertsCreated ? 'This rule generated alerts but did not send external notifications because rule action' : 'Rule action';
  if (uniqueActionTypes.size > 1) {
    return `${alertsGeneratedText} connectors ${actionTypesJoined} are not enabled. To send notifications, you need a higher Security Analytics license / tier`;
  } else {
    return `${alertsGeneratedText} connector ${actionTypesJoined} is not enabled. To send notifications, you need a higher Security Analytics license / tier`;
  }
};
exports.getDisabledActionsWarningText = getDisabledActionsWarningText;
/**
 * converts ES after_key object into string
 * for example: { "agent.name": "test" } would become `agent.name: test`
 */
const stringifyAfterKey = afterKey => {
  if (!afterKey) {
    return;
  }
  return Object.entries(afterKey).map(entry => entry.join(': ')).join(', ');
};
exports.stringifyAfterKey = stringifyAfterKey;
const buildShellAlertSuppressionTermsAndFields = ({
  sharedParams,
  shellAlert,
  buildingBlockAlerts
}) => {
  var _completeRule$rulePar, _ref, _ref2;
  const {
    alertTimestampOverride,
    primaryTimestamp,
    secondaryTimestamp,
    completeRule,
    spaceId
  } = sharedParams;
  const suppressionTerms = (0, _suppression_utils.getSuppressionTerms)({
    alertSuppression: completeRule === null || completeRule === void 0 ? void 0 : (_completeRule$rulePar = completeRule.ruleParams) === null || _completeRule$rulePar === void 0 ? void 0 : _completeRule$rulePar.alertSuppression,
    input: shellAlert._source
  });
  const instanceId = (0, _objectHash.default)([suppressionTerms, completeRule.alertId, spaceId]);
  const primarySuppressionTime = (0, _robust_field_access.robustGet)({
    key: primaryTimestamp,
    document: shellAlert._source
  });
  const secondarySuppressionTime = secondaryTimestamp && (0, _robust_field_access.robustGet)({
    key: secondaryTimestamp,
    document: shellAlert._source
  });
  const suppressionTime = new Date((_ref = (_ref2 = primarySuppressionTime !== null && primarySuppressionTime !== void 0 ? primarySuppressionTime : secondarySuppressionTime) !== null && _ref2 !== void 0 ? _ref2 : alertTimestampOverride) !== null && _ref !== void 0 ? _ref : shellAlert._source[_ruleDataUtils.TIMESTAMP]);
  const suppressionFields = {
    [_ruleDataUtils.ALERT_INSTANCE_ID]: instanceId,
    [_ruleDataUtils.ALERT_SUPPRESSION_TERMS]: suppressionTerms,
    [_ruleDataUtils.ALERT_SUPPRESSION_START]: suppressionTime,
    [_ruleDataUtils.ALERT_SUPPRESSION_END]: suppressionTime,
    [_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT]: 0
  };
  (0, _lodash.merge)(shellAlert._source, suppressionFields);
  return {
    _id: shellAlert._id,
    _index: shellAlert._index,
    _source: shellAlert._source,
    subAlerts: buildingBlockAlerts
  };
};
exports.buildShellAlertSuppressionTermsAndFields = buildShellAlertSuppressionTermsAndFields;