"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUrlQuery = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _cloudSecurityPosture = require("@kbn/cloud-security-posture");
var _query_utils = require("@kbn/cloud-security-posture/src/utils/query_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const URL_PARAM_KEYS = [_query_utils.QUERY_PARAM_KEY, _query_utils.FLYOUT_PARAM_KEY];

/**
 * @description uses 'rison' to encode/decode a url query
 * @todo replace getDefaultQuery with schema. validate after decoded from URL, use defaultValues
 * @note shallow-merges default, current and next query
 */
const useUrlQuery = getDefaultQuery => {
  const {
    push,
    replace
  } = (0, _reactRouterDom.useHistory)();
  const {
    search,
    key
  } = (0, _reactRouterDom.useLocation)();
  const urlQuery = (0, _react.useMemo)(() => {
    const decodedParams = (0, _query_utils.decodeMultipleRisonParams)(search, URL_PARAM_KEYS);

    // Extract query parameters (the main query parameters)
    const queryParams = decodedParams[_query_utils.QUERY_PARAM_KEY] || {};

    // Extract flyout parameters
    const flyoutParams = decodedParams[_query_utils.FLYOUT_PARAM_KEY] || {};

    // Keep parameters separate to avoid conflicts
    return {
      ...getDefaultQuery(),
      ...queryParams,
      // Keep flyout parameters in a separate namespace
      flyout: flyoutParams
    };
  }, [getDefaultQuery, search]);
  const setUrlQuery = (0, _react.useCallback)(query => {
    const mergedQuery = {
      ...getDefaultQuery(),
      ...urlQuery,
      ...query
    };
    const {
      flyout,
      ...queryParams
    } = mergedQuery;

    // Build search string components
    const queryParamsSearch = (0, _cloudSecurityPosture.encodeQuery)(queryParams);
    const flyoutSearch = buildFlyoutSearchString(flyout);

    // Combine and set final search string
    const finalSearch = combineSearchParts([queryParamsSearch, flyoutSearch]);
    push({
      search: finalSearch
    });
  }, [getDefaultQuery, urlQuery, push]);

  // Set initial query
  (0, _react.useEffect)(() => {
    if (search) return;
    replace({
      search: (0, _cloudSecurityPosture.encodeQuery)(getDefaultQuery())
    });
  }, [getDefaultQuery, search, replace]);
  return {
    key,
    urlQuery,
    setUrlQuery
  };
};

// Helper function to build flyout search string
exports.useUrlQuery = useUrlQuery;
function buildFlyoutSearchString(flyoutParams) {
  if (Object.keys(flyoutParams).length === 0) {
    return '';
  }
  return (0, _query_utils.encodeRisonParam)(_query_utils.FLYOUT_PARAM_KEY, flyoutParams) || '';
}

// Helper function to combine search parts
function combineSearchParts(searchParts) {
  return searchParts.filter(Boolean).join('&');
}