"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usageTracker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _std = require("@kbn/std");
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable max-classes-per-file,@typescript-eslint/no-explicit-any */

/**
 * Keep track of usage of stuff. Example: can track how many time a given utility/function was called.
 *
 * ** Should not be used for production code **
 */
class UsageTracker {
  constructor({
    logger = (0, _utils.createToolingLogger)(),
    dumpOnProcessExit = false,
    maxRecordsPerType = 25
  } = {}) {
    (0, _defineProperty2.default)(this, "records", {});
    (0, _defineProperty2.default)(this, "options", void 0);
    (0, _defineProperty2.default)(this, "wrappedCallbacks", new WeakSet());
    this.options = {
      logger,
      dumpOnProcessExit,
      maxRecordsPerType
    };
    try {
      if (dumpOnProcessExit && process && process.once) {
        const nodeEvents = ['SIGINT', 'exit', 'uncaughtException', 'unhandledRejection'];
        const logStats = (0, _lodash.once)(() => {
          logger.verbose(`Tooling usage tracking:
${this.toText()}`);
        });
        logger.verbose(`${this.constructor.name}: Setting up event listeners for: ${nodeEvents.join(' | ')}`);
        nodeEvents.forEach(event => {
          process.once(event, logStats);
        });
      }
    } catch (err) {
      logger.debug(`Unable to setup 'dumpOnProcessExit': ${err.message}`);
    }
  }
  formatDuration(durationMs) {
    const durationObj = _moment.default.duration(durationMs);
    const pad = (num, max = 2) => {
      return String(num).padStart(max, '0');
    };
    const hours = pad(durationObj.hours());
    const minutes = pad(durationObj.minutes());
    const seconds = pad(durationObj.seconds());
    const milliseconds = pad(durationObj.milliseconds(), 3);
    return `${hours}:${minutes}:${seconds}.${milliseconds}`;
  }
  create(id) {
    var _this$records$id;
    this.records[id] = (_this$records$id = this.records[id]) !== null && _this$records$id !== void 0 ? _this$records$id : {
      count: 0,
      records: []
    };
    const maxRecords = this.options.maxRecordsPerType;
    const usageRecord = new UsageRecord(id);
    const usageForId = this.records[id];
    usageForId.count++;
    usageForId.records.push(usageRecord);
    if (usageForId.records.length > maxRecords) {
      usageForId.records.splice(0, usageForId.records.length - maxRecords);
    }
    return usageRecord;
  }
  toJSON() {
    return Object.values(this.records).map(({
      records
    }) => records).flat().map(record => record.toJSON());
  }

  /**
   * Returns a `JSON.parse()` compatible string of all of the entries captured
   */
  toString() {
    return JSON.stringify(this.toJSON());
  }
  getSummary() {
    return Object.entries(this.records).map(([funcName, record]) => {
      const funcSummary = {
        name: funcName,
        count: record.count,
        shortestMs: 0,
        longestMs: 0
      };
      for (const instanceRecord of record.records) {
        const instanceDuration = instanceRecord.toJSON().durationMs;
        funcSummary.shortestMs = funcSummary.shortestMs > 0 ? Math.min(funcSummary.shortestMs, instanceDuration) : instanceDuration;
        funcSummary.longestMs = funcSummary.longestMs > 0 ? Math.max(funcSummary.longestMs, instanceDuration) : instanceDuration;
      }
      return funcSummary;
    });
  }
  toSummaryTable() {
    const separator = ' | ';
    const width = {
      name: 60,
      count: 5,
      shortest: 12,
      longest: 12
    };
    const maxLineLength = Object.values(width).reduce((acc, n) => acc + n, 0) + (Object.keys(width).length - 1) * separator.length;
    const summaryText = this.getSummary().map(({
      name,
      count,
      shortestMs,
      longestMs
    }) => {
      const fmtName = name.padEnd(width.name);
      const fmtCount = String(count).padEnd(width.count);
      const fmtShortest = this.formatDuration(shortestMs);
      const fmtLongest = this.formatDuration(longestMs);
      return `${fmtName}${separator}${fmtCount}${separator}${fmtShortest}${separator}${fmtLongest}`;
    });
    return `${'-'.repeat(maxLineLength)}
${'Name'.padEnd(width.name)}${separator}${'Count'.padEnd(width.count)}${separator}${'Shortest'.padEnd(width.shortest)}${separator}${'longest'.padEnd(width.longest)}
${'-'.repeat(maxLineLength)}
${summaryText.join('\n')}
${'-'.repeat(maxLineLength)}
`;
  }

  /**
   * Returns a string with information about the entries captured
   */
  toText() {
    return this.toSummaryTable() + Object.entries(this.records).map(([key, {
      count,
      records: usageRecords
    }]) => {
      return `
[${key}] Invoked ${count} times. Records${count > this.options.maxRecordsPerType ? ` (last ${this.options.maxRecordsPerType})` : ''}:
${'-'.repeat(98)}
${usageRecords.map(record => {
        return record.toText();
      }).join('\n')}
`;
    }).join('');
  }
  dump(logger) {
    (logger !== null && logger !== void 0 ? logger : this.options.logger).info(`${this.constructor.name}: usage tracking:
${this.toText()}`);
  }

  /**
   * Will wrap the provided callback and provide usage tracking on it.
   * @param callback
   * @param name
   */

  track(callbackOrName, maybeCallback) {
    var _Error$stack;
    const isArg1Callback = typeof callbackOrName === 'function';
    if (!isArg1Callback && !maybeCallback) {
      throw new Error(`Second argument to 'track()' can not be undefined when first argument defined a name`);
    }
    const callback = isArg1Callback ? callbackOrName : maybeCallback;
    const name = isArg1Callback ? undefined : callbackOrName;
    if (this.wrappedCallbacks.has(callback)) {
      return callback;
    }
    const functionName = name || callback.name ||
    // Get the file/line number where function was defined
    (((_Error$stack = new Error('-').stack) !== null && _Error$stack !== void 0 ? _Error$stack : '').split('\n')[2] || '').trim() ||
    // Last resort: get 50 first char. of function code
    callback.toString().trim().substring(0, 50);
    const wrappedFunction = (...args) => {
      const usageRecord = this.create(functionName);
      try {
        const response = callback(...args);
        if ((0, _std.isPromise)(response)) {
          response.then(() => {
            usageRecord.set('success');
          }).catch(e => {
            usageRecord.set('failure', e.message);
          });
        } else {
          usageRecord.set('success');
        }
        return response;
      } catch (e) {
        usageRecord.set('failure', e.message);
        throw e;
      }
    };
    this.wrappedCallbacks.add(wrappedFunction);
    return wrappedFunction;
  }
}
class UsageRecord {
  constructor(id) {
    (0, _defineProperty2.default)(this, "start", new Date().toISOString());
    (0, _defineProperty2.default)(this, "finish", '');
    (0, _defineProperty2.default)(this, "durationMs", 0);
    (0, _defineProperty2.default)(this, "duration", '');
    (0, _defineProperty2.default)(this, "status", 'pending');
    (0, _defineProperty2.default)(this, "error", void 0);
    (0, _defineProperty2.default)(this, "stack", '');
    this.id = id;
    Error.captureStackTrace(this);
    this.stack = `\n${this.stack.split('\n').slice(2).join('\n')}`;
  }
  set(status, error) {
    this.finish = new Date().toISOString();
    this.error = error;
    this.status = status;
    const durationDiff = _moment.default.duration((0, _moment.default)(this.finish).diff(this.start));
    this.durationMs = durationDiff.asMilliseconds();
    this.duration = `h[${durationDiff.hours()}]  m[${durationDiff.minutes()}]  s[${durationDiff.seconds()}]  ms[${durationDiff.milliseconds()}]`;
  }
  toJSON() {
    const {
      id,
      start,
      finish,
      status,
      error,
      duration,
      durationMs,
      stack
    } = this;
    return {
      id,
      start,
      finish,
      durationMs,
      duration,
      status,
      stack,
      ...(error ? {
        error
      } : {})
    };
  }
  toString() {
    return JSON.stringify(this.toJSON());
  }
  toText() {
    const data = this.toJSON();
    const keys = Object.keys(data).sort();
    return keys.reduce((acc, key) => {
      return acc.concat(`\n${key}: ${data[key]}`);
    }, '');
  }
}
const usageTracker = exports.usageTracker = new UsageTracker({
  dumpOnProcessExit: true
});