"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFormWithWarnings = useFormWithWarnings;
var _react = require("react");
var _lodash = require("lodash");
var _shared_imports = require("../../../shared_imports");
var _extract_validation_results = require("./extract_validation_results");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Form lib implements warning functionality via non blocking validators. `validations` allows to
 * specify validation configuration with validator functions and extra parameters including
 * `isBlocking`. Validators marked as `isBlocking` will produce non blocking validation errors
 * a.k.a. warnings.
 *
 * The problem with the supported approach is lack of flexibility and necessary API like one for getting
 * only blocking or non blocking errors. Flexibility requirement comes from complex async validators
 * producing blocking and non blocking validation errors. There is no way to use `isBlocking` configuration
 * option to separate errors. Separating such validating functions in two would lead to sending two
 * HTTP requests and performing another async operations twice.
 *
 * On top of just having an ability to mark validation errors as non blocking via `isBlocking: false`
 * configuration we require a way to return blocking and non blocking errors from a single validation
 * function. It'd be possible by returning an error with `isBlocking` (or `isWarning`) flag along with
 * `message` and `code` fields from a validator function. Attempts to reuse `__isBlocking__` internal
 * field lead to inconsistent behavior.
 *
 * `useFormWithWarnings` implements warnings (non blocking errors) on top of `FormHook` using validation
 * error codes as a flexible way to determine whether an error is a blocking error or it's a warning.
 * It provides little interface extension to simplify errors and warnings consumption
 *
 * In some cases business logic requires implementing functionality to allow users perform an action
 * despite non-critical validation errors a.k.a. warnings. Usually it's also required to inform users
 * about warnings they got before proceeding for example via a modal.
 *
 * Since `FormHook` returned by `useForm` lacks of such functionality `useFormWithWarnings` is here to
 * provide warnings functionality. It could be used and passed as `FormHook` when warnings functionality
 * isn't required making absolutely transparent.
 *
 * **Important:** Validators use short circuiting by default. It means that any failed validation in
 * `validations` configuration array will prevent the rest validators from running. When used with warnings
 * it may lead to bugs when validator checks first for warnings. You have to make sure a value is validated
 * for errors first and then for warnings.
 *
 * There is a ticket to move this functionality to Form lib https://github.com/elastic/kibana/issues/203097.
 */
function useFormWithWarnings(formConfig) {
  const {
    onSubmit,
    options: {
      warningValidationCodes
    }
  } = formConfig;
  const {
    form
  } = (0, _shared_imports.useForm)(formConfig);
  const {
    validate: originalValidate,
    getFormData,
    getFields
  } = form;
  const validationResultsRef = (0, _react.useRef)({
    errors: [],
    warnings: []
  });
  const [isSubmitted, setIsSubmitted] = (0, _react.useState)(false);
  const [isSubmitting, setSubmitting] = (0, _react.useState)(false);
  const [isValid, setIsValid] = (0, _react.useState)();
  const isMounted = (0, _react.useRef)(false);
  const validate = (0, _react.useCallback)(async () => {
    await originalValidate();
    validationResultsRef.current = (0, _extract_validation_results.extractValidationResults)(Object.values(getFields()), warningValidationCodes);
    const isFormValid = (0, _lodash.isEmpty)(validationResultsRef.current.errors);
    setIsValid(isFormValid);
    return isFormValid;
  }, [originalValidate, getFields, warningValidationCodes, validationResultsRef]);
  const submit = (0, _react.useCallback)(async e => {
    if (e) {
      e.preventDefault();
    }
    setIsSubmitted(true);
    setSubmitting(true);
    const isFormValid = await validate();
    const formData = isFormValid ? getFormData() : {};
    if (onSubmit) {
      await onSubmit(formData, isFormValid, validationResultsRef.current);
    }
    if (isMounted.current) {
      setSubmitting(false);
    }
    return {
      data: formData,
      isValid: isFormValid
    };
  }, [validate, getFormData, onSubmit, validationResultsRef]);

  // Track form's mounted state
  (0, _react.useEffect)(() => {
    isMounted.current = true;
    return () => {
      isMounted.current = false;
    };
  }, []);
  return (0, _react.useMemo)(() => ({
    form: {
      ...form,
      isValid,
      isSubmitted,
      isSubmitting,
      validate,
      submit,
      getErrors: () => validationResultsRef.current.errors.map(x => x.message),
      getValidationWarnings: () => validationResultsRef.current.warnings
    }
  }), [form, validate, submit, isSubmitted, isSubmitting, isValid, validationResultsRef]);
}