"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPlannerAgentGraph = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _graph = require("../chat/graph");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const StateAnnotation = _langgraph.Annotation.Root({
  // inputs
  initialMessages: (0, _langgraph.Annotation)({
    reducer: _langgraph.messagesStateReducer,
    default: () => []
  }),
  remainingCycles: (0, _langgraph.Annotation)(),
  // internal state
  plan: (0, _langgraph.Annotation)(),
  backlog: (0, _langgraph.Annotation)({
    reducer: (current, next) => {
      return [...current, ...next];
    },
    default: () => []
  }),
  // outputs
  generatedAnswer: (0, _langgraph.Annotation)()
});
const createPlannerAgentGraph = async ({
  chatModel,
  tools,
  logger: log
}) => {
  const stringify = obj => JSON.stringify(obj, null, 2);

  /**
   * Create a plan based on the current discussion.
   */
  const createPlan = async state => {
    const plannerModel = chatModel.withStructuredOutput(_zod.z.object({
      reasoning: _zod.z.string().describe(`Internal reasoning of how you come to such plan`),
      plan: _zod.z.array(_zod.z.string()).describe('Steps identified for the action plan')
    })).withConfig({
      tags: ['planner:create_plan']
    });
    const response = await plannerModel.invoke((0, _prompts.getPlanningPrompt)({
      discussion: state.initialMessages
    }));
    log.trace(() => `createPlan - response: ${stringify(response)}`);
    const plan = {
      reasoning: response.reasoning,
      steps: response.plan
    };
    return {
      plan: plan.steps,
      backlog: [plan]
    };
  };

  /**
   * Delegates execution of the next step in the plan to an executor agent.
   */
  const executeStep = async state => {
    const nextTask = state.plan[0];
    const executorAgent = (0, _graph.createAgentGraph)({
      chatModel,
      tools,
      logger: log,
      systemPrompt: ''
    });
    const {
      addedMessages
    } = await executorAgent.invoke({
      initialMessages: (0, _prompts.getExecutionPrompt)({
        task: nextTask,
        backlog: state.backlog
      })
    }, {
      tags: ['executor_agent'],
      metadata: {
        graphName: 'executor_agent'
      }
    });
    const messageContent = (0, _langchain.extractTextContent)(addedMessages[addedMessages.length - 1]);
    log.trace(() => `executeStep - step: ${nextTask} - response: ${messageContent}`);
    return {
      plan: state.plan.slice(1),
      backlog: [{
        step: nextTask,
        output: messageContent
      }]
    };
  };

  /**
   * Eventually revise the plan according to the result of the last action.
   */
  const revisePlan = async state => {
    const revisePlanModel = chatModel.withStructuredOutput(_zod.z.object({
      reasoning: _zod.z.string().describe(`Internal reasoning of how you come to update the plan`),
      plan: _zod.z.array(_zod.z.string()).describe('Steps identified for the revised action plan')
    })).withConfig({
      tags: ['planner:revise-plan']
    });
    const response = await revisePlanModel.invoke((0, _prompts.getReplanningPrompt)({
      discussion: state.initialMessages,
      plan: state.plan,
      backlog: state.backlog
    }));
    const plan = {
      reasoning: response.reasoning,
      steps: response.plan
    };
    log.trace(() => `revisePlan - ${stringify(plan)}`);
    return {
      plan: plan.steps,
      backlog: [plan]
    };
  };
  const revisePlanTransition = async state => {
    const remainingCycles = state.remainingCycles;
    if (state.plan.length <= 0 || remainingCycles <= 0) {
      return 'answer';
    }
    return 'execute_step';
  };
  const answer = async state => {
    const answerModel = chatModel.withConfig({
      tags: ['planner:answer']
    });
    const response = await answerModel.invoke((0, _prompts.getAnswerPrompt)({
      discussion: state.initialMessages,
      backlog: state.backlog
    }));
    const generatedAnswer = (0, _langchain.extractTextContent)(response);
    log.trace(() => `answer - response ${stringify(generatedAnswer)}`);
    return {
      generatedAnswer
    };
  };

  // note: the node names are used in the event convertion logic, they should *not* be changed
  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('create_plan', createPlan).addNode('execute_step', executeStep).addNode('revise_plan', revisePlan).addNode('answer', answer)
  // edges
  .addEdge('__start__', 'create_plan').addEdge('create_plan', 'execute_step').addEdge('execute_step', 'revise_plan').addConditionalEdges('revise_plan', revisePlanTransition, {
    execute_step: 'execute_step',
    answer: 'answer'
  }).addEdge('answer', '__end__').compile();
  return graph;
};
exports.createPlannerAgentGraph = createPlannerAgentGraph;