"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flowRouteRepository = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var t = _interopRequireWildcard(require("io-ts"));
var _server = require("@kbn/fleet-plugin/server");
var _jsYaml = require("js-yaml");
var _output_client = require("@kbn/fleet-plugin/server/services/output_client");
var _telemetry_events = require("../../../common/telemetry_events");
var _state = require("../../lib/state");
var _create_observability_onboarding_server_route = require("../create_observability_onboarding_server_route");
var _get_has_logs = require("./get_has_logs");
var _get_fallback_urls = require("../../lib/get_fallback_urls");
var _get_agent_version = require("../../lib/get_agent_version");
var _types = require("../types");
var _create_shipper_api_key = require("../../lib/api_key/create_shipper_api_key");
var _create_install_api_key = require("../../lib/api_key/create_install_api_key");
var _has_log_monitoring_privileges = require("../../lib/api_key/has_log_monitoring_privileges");
var _make_tar = require("./make_tar");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const stepProgressUpdateRoute = (0, _create_observability_onboarding_server_route.createObservabilityOnboardingServerRoute)({
  endpoint: 'POST /internal/observability_onboarding/flow/{id}/step/{name}',
  security: {
    authz: {
      enabled: false,
      reason: "This endpoint is meant to be called from user's terminal and authenticated using API key with a limited privileges. For this reason there is no authorization and saved object is accessed using an internal Kibana user (the API key used by the user should not have those privileges)"
    }
  },
  params: t.type({
    path: t.type({
      id: t.string,
      name: t.string
    }),
    body: t.intersection([t.type({
      status: t.string
    }), t.partial({
      message: t.string
    }), t.partial({
      payload: _types.StepProgressPayloadRT
    })])
  }),
  async handler(resources) {
    const {
      params: {
        path: {
          id,
          name
        },
        body: {
          status,
          message,
          payload
        }
      },
      core
    } = resources;

    /**
     * Message is base64 encoded as it might include arbitrary error messages
     * from user's terminal containing special characters that would otherwise
     * break the request.
     */
    const decodedMessage = Buffer.from(message !== null && message !== void 0 ? message : '', 'base64').toString('utf-8');
    const coreStart = await core.start();
    const savedObjectsClient = coreStart.savedObjects.createInternalRepository();
    const savedObservabilityOnboardingState = await (0, _state.getObservabilityOnboardingFlow)({
      savedObjectsClient,
      savedObjectId: id
    });
    if (!savedObservabilityOnboardingState) {
      throw _boom.default.notFound('Unable to report setup progress - onboarding session not found.');
    }
    const {
      id: savedObjectId,
      updatedAt,
      ...observabilityOnboardingState
    } = savedObservabilityOnboardingState;
    await (0, _state.saveObservabilityOnboardingFlow)({
      savedObjectsClient,
      savedObjectId,
      observabilityOnboardingState: {
        ...observabilityOnboardingState,
        progress: {
          ...observabilityOnboardingState.progress,
          [name]: {
            status,
            message: decodedMessage,
            payload
          }
        }
      }
    });
    coreStart.analytics.reportEvent(_telemetry_events.OBSERVABILITY_ONBOARDING_TELEMETRY_EVENT.eventType, {
      flow_type: observabilityOnboardingState.type,
      flow_id: id,
      step: name,
      step_status: status,
      step_message: decodedMessage,
      payload
    });
    return {
      name,
      status,
      message,
      payload
    };
  }
});
const getProgressRoute = (0, _create_observability_onboarding_server_route.createObservabilityOnboardingServerRoute)({
  endpoint: 'GET /internal/observability_onboarding/flow/{onboardingId}/progress',
  security: {
    authz: {
      enabled: false,
      reason: 'Authorization is checked by the Saved Object client'
    }
  },
  params: t.type({
    path: t.type({
      onboardingId: t.string
    })
  }),
  async handler(resources) {
    var _progress$eaStatus;
    const {
      params: {
        path: {
          onboardingId
        }
      },
      core,
      request
    } = resources;
    const coreStart = await core.start();
    const savedObjectsClient = coreStart.savedObjects.getScopedClient(request);
    const savedObservabilityOnboardingState = await (0, _state.getObservabilityOnboardingFlow)({
      savedObjectsClient,
      savedObjectId: onboardingId
    });
    if (!savedObservabilityOnboardingState) {
      throw _boom.default.notFound('Unable to report setup progress - onboarding session not found.');
    }
    const progress = {
      ...(savedObservabilityOnboardingState === null || savedObservabilityOnboardingState === void 0 ? void 0 : savedObservabilityOnboardingState.progress)
    };
    const esClient = coreStart.elasticsearch.client.asScoped(request).asCurrentUser;
    if (((_progress$eaStatus = progress['ea-status']) === null || _progress$eaStatus === void 0 ? void 0 : _progress$eaStatus.status) === 'complete') {
      var _progress$eaStatus2;
      const {
        agentId
      } = (_progress$eaStatus2 = progress['ea-status']) === null || _progress$eaStatus2 === void 0 ? void 0 : _progress$eaStatus2.payload;
      try {
        const hasLogs = await (0, _get_has_logs.getHasLogs)(esClient, agentId);
        progress['logs-ingest'] = {
          status: hasLogs ? 'complete' : 'loading'
        };
      } catch (error) {
        progress['logs-ingest'] = {
          status: 'warning',
          message: error.message
        };
      }
    } else {
      progress['logs-ingest'] = {
        status: 'incomplete'
      };
    }
    return {
      progress
    };
  }
});

/**
 * This endpoint starts a new onboarding flow and creates two API keys:
 * 1. A short-lived API key with privileges to install integrations.
 * 2. An API key with privileges to ingest log and metric data used to configure Elastic Agent.
 *
 * It also returns all required information to download the onboarding script and install the
 * Elastic agent.
 *
 * If the user does not have all necessary privileges a 403 Forbidden response is returned.
 */
const createFlowRoute = (0, _create_observability_onboarding_server_route.createObservabilityOnboardingServerRoute)({
  endpoint: 'POST /internal/observability_onboarding/flow',
  security: {
    authz: {
      enabled: false,
      reason: 'Authorization is checked by the Saved Object client'
    }
  },
  async handler(resources) {
    var _plugins$cloud;
    const {
      context,
      core,
      request,
      plugins,
      kibanaVersion
    } = resources;
    const coreStart = await core.start();
    const {
      elasticsearch: {
        client
      }
    } = await context.core;
    const savedObjectsClient = coreStart.savedObjects.getScopedClient(request);
    const hasPrivileges = await (0, _has_log_monitoring_privileges.hasLogMonitoringPrivileges)(client.asCurrentUser);
    if (!hasPrivileges) {
      throw _boom.default.forbidden('Unauthorized to create log indices');
    }
    const fleetPluginStart = await plugins.fleet.start();
    const [onboardingFlow, ingestApiKey, installApiKey, elasticAgentVersionInfo] = await Promise.all([(0, _state.saveObservabilityOnboardingFlow)({
      savedObjectsClient,
      observabilityOnboardingState: {
        type: 'autoDetect',
        state: undefined,
        progress: {}
      }
    }), (0, _create_shipper_api_key.createShipperApiKey)(client.asCurrentUser, 'standalone-elastic-agent'), (await context.resolve(['core'])).core.security.authc.apiKeys.create((0, _create_install_api_key.createInstallApiKey)('onboarding-install')), (0, _get_agent_version.getAgentVersionInfo)(fleetPluginStart, kibanaVersion)]);
    if (!installApiKey) {
      throw _boom.default.notFound('License does not allow API key creation.');
    }
    const kibanaUrl = (0, _get_fallback_urls.getKibanaUrl)(core.setup, (_plugins$cloud = plugins.cloud) === null || _plugins$cloud === void 0 ? void 0 : _plugins$cloud.setup);
    const scriptDownloadUrl = new URL(core.setup.http.staticAssets.getPluginAssetHref('auto_detect.sh'), kibanaUrl).toString();
    return {
      onboardingFlow,
      ingestApiKey: ingestApiKey.encoded,
      installApiKey: installApiKey.encoded,
      elasticAgentVersionInfo,
      kibanaUrl,
      scriptDownloadUrl
    };
  }
});

/**
 * This endpoints installs the requested integrations and returns the corresponding config file for
 * Elastic Agent.
 *
 * The request format is TSV (tab-separated values) to simplify parsing in bash.
 *
 * The response format is a tar archive containing the Elastic Agent configuration, depending on the
 * `Accept` header.
 *
 * Errors during installation are ignore unless all integrations fail to install. When that happens
 * a 500 Internal Server Error is returned with the first error message.
 *
 * Example request:
 *
 * ```text
 * POST /internal/observability_onboarding/flow/${ONBOARDING_ID}/integrations/install
 *
 * system registry
 * product_service custom /path/to/access.log
 * product_service custom /path/to/error.log
 * checkout_service custom /path/to/access.log
 * checkout_service custom /path/to/error.log
 * ```
 *
 * Example response (tarball):
 *
 * ```
 * -rw-r--r--  113 elastic-agent.yml
 * drwxr-xr-x    0 inputs.d/
 * -rw-r--r-- 4890 inputs.d/system.yml
 * -rw-r--r--  240 inputs.d/product_service.yml
 * -rw-r--r--  243 inputs.d/checkout_service.yml
 * ```
 *
 * Example curl:
 *
 * ```bash
 * curl --request POST \
 *  --url "http://localhost:5601/internal/observability_onboarding/flow/${ONBOARDING_ID}/integrations/install" \
 *  --header "Authorization: ApiKey ${ENCODED_API_KEY}" \
 *  --header "Accept: application/x-tar" \
 *  --header "Content-Type: text/tab-separated-values" \
 *  --header "kbn-xsrf: true" \
 *  --header "x-elastic-internal-origin: Kibana" \
 *  --data $'system\tregistry\twebserver01\nproduct_service\tcustom\t/path/to/access.log\ncheckout_service\tcustom\t/path/to/access.log' \
 *  --output - | tar -tvf -
 * ```
 */
const integrationsInstallRoute = (0, _create_observability_onboarding_server_route.createObservabilityOnboardingServerRoute)({
  endpoint: 'POST /internal/observability_onboarding/flow/{onboardingId}/integrations/install',
  security: {
    authz: {
      enabled: false,
      reason: "This endpoint is meant to be called from user's terminal. Authorization is partially checked by the Package Service client, and saved object is accessed using internal Kibana user because the API key used for installing integrations should not have those privileges."
    }
  },
  params: t.type({
    path: t.type({
      onboardingId: t.string
    }),
    body: t.string
  }),
  async handler({
    context,
    request,
    response,
    params,
    core,
    plugins,
    services
  }) {
    const coreStart = await core.start();
    const fleetStart = await plugins.fleet.start();
    const savedObjectsClient = coreStart.savedObjects.createInternalRepository();
    const packageClient = fleetStart.packageService.asScoped(request);
    const savedObservabilityOnboardingState = await (0, _state.getObservabilityOnboardingFlow)({
      savedObjectsClient,
      savedObjectId: params.path.onboardingId
    });
    if (!savedObservabilityOnboardingState) {
      throw _boom.default.notFound(`Onboarding session '${params.path.onboardingId}' not found.`);
    }
    const outputClient = await fleetStart.createOutputClient(request);
    const defaultOutputId = await outputClient.getDefaultDataOutputId();
    if (!defaultOutputId) {
      throw _boom.default.notFound('Default data output not found');
    }
    const output = await outputClient.get(defaultOutputId);
    const integrationsToInstall = parseIntegrationsTSV(params.body);
    if (!integrationsToInstall.length) {
      return response.badRequest({
        body: {
          message: 'Please specify a list of integrations to install'
        }
      });
    }
    let installedIntegrations = [];
    try {
      const settledResults = await ensureInstalledIntegrations(integrationsToInstall, packageClient);
      installedIntegrations = settledResults.reduce((acc, result) => {
        if (result.status === 'fulfilled') {
          acc.push(result.value);
        }
        return acc;
      }, []);
      // Errors during installation are ignored unless all integrations fail to install. When that happens
      // a 500 Internal Server Error is returned with the first error message.
      if (!installedIntegrations.length) {
        throw settledResults[0].reason;
      }
    } catch (error) {
      if (error instanceof _server.FleetUnauthorizedError) {
        return response.forbidden({
          body: {
            message: error.message
          }
        });
      }
      throw error;
    }
    await (0, _state.saveObservabilityOnboardingFlow)({
      savedObjectsClient,
      savedObjectId: params.path.onboardingId,
      observabilityOnboardingState: {
        ...savedObservabilityOnboardingState,
        progress: {
          ...savedObservabilityOnboardingState.progress,
          'install-integrations': {
            status: 'complete',
            payload: installedIntegrations
          }
        }
      }
    });
    coreStart.analytics.reportEvent(_telemetry_events.OBSERVABILITY_ONBOARDING_TELEMETRY_EVENT.eventType, {
      flow_type: savedObservabilityOnboardingState.type,
      flow_id: params.path.onboardingId,
      step: 'install-integrations',
      step_status: 'complete',
      payload: {
        integrations: installedIntegrations.map(({
          title,
          pkgName,
          pkgVersion,
          installSource
        }) => ({
          title,
          pkgName,
          pkgVersion,
          installSource
        }))
      }
    });
    return response.ok({
      headers: {
        'content-type': 'application/x-tar'
      },
      body: generateAgentConfigTar(output, installedIntegrations)
    });
  }
});
async function ensureInstalledIntegrations(integrationsToInstall, packageClient) {
  return Promise.allSettled(integrationsToInstall.map(async integration => {
    const {
      pkgName,
      installSource
    } = integration;
    if (installSource === 'registry') {
      var _packageInfo$data_str, _packageInfo$data_str2;
      const installation = await packageClient.ensureInstalledPackage({
        pkgName
      });
      const pkg = installation.package;
      const config = filterUnsupportedInputs(await packageClient.getAgentPolicyConfigYAML(pkg.name, pkg.version));
      const {
        packageInfo
      } = await packageClient.getPackage(pkg.name, pkg.version);
      return {
        installSource,
        pkgName: pkg.name,
        pkgVersion: pkg.version,
        title: packageInfo.title,
        config,
        dataStreams: (_packageInfo$data_str = (_packageInfo$data_str2 = packageInfo.data_streams) === null || _packageInfo$data_str2 === void 0 ? void 0 : _packageInfo$data_str2.map(({
          type,
          dataset
        }) => ({
          type,
          dataset
        }))) !== null && _packageInfo$data_str !== void 0 ? _packageInfo$data_str : [],
        kibanaAssets: pkg.installed_kibana,
        metadata: integration.metadata
      };
    }
    const dataStream = {
      type: 'logs',
      dataset: pkgName
    };
    const installed = {
      installSource,
      pkgName,
      pkgVersion: '1.0.0',
      // Custom integrations are always installed as version `1.0.0`
      title: pkgName,
      config: (0, _jsYaml.dump)({
        inputs: [{
          id: `filestream-${pkgName}`,
          type: 'filestream',
          streams: [{
            id: `filestream-${pkgName}`,
            data_stream: dataStream,
            paths: integration.logFilePaths,
            processors: [{
              add_fields: {
                target: 'service',
                fields: {
                  name: pkgName
                }
              }
            }]
          }]
        }]
      }),
      dataStreams: [dataStream],
      kibanaAssets: []
    };
    try {
      await packageClient.installCustomIntegration({
        pkgName,
        datasets: [{
          name: dataStream.dataset,
          type: dataStream.type
        }]
      });
      return installed;
    } catch (error) {
      // If the error is a naming collision, we can assume the integration is already installed and treat this step as successful
      if (error instanceof _server.NamingCollisionError) {
        return installed;
      } else {
        throw error;
      }
    }
  }));
}
function filterUnsupportedInputs(policyYML) {
  const policy = (0, _jsYaml.load)(policyYML);
  if (!policy) {
    return policyYML;
  }
  return (0, _jsYaml.dump)({
    ...policy,
    inputs: (policy.inputs || []).filter(input => {
      return input.type !== 'httpjson';
    })
  });
}

/**
 * Parses and validates a TSV (tab-separated values) string of integrations with params.
 *
 * Returns an object of integrations to install.
 *
 * Example input:
 *
 * ```text
 * system registry hostname
 * nginx registry
 * product_service custom /path/to/access.log
 * product_service custom /path/to/error.log
 * checkout_service custom /path/to/access.log
 * checkout_service custom /path/to/error.log
 * ```
 */
function parseIntegrationsTSV(tsv) {
  if (tsv.trim() === '') {
    return [];
  }
  return Object.values(tsv.trim().split('\n').map(line => line.split('\t', 3)).reduce((acc, [pkgName, installSource, parameter]) => {
    const key = `${pkgName}-${installSource}`;
    if (installSource === 'registry') {
      const metadata = parseRegistryIntegrationMetadata(pkgName, parameter);
      acc[key] = {
        pkgName,
        installSource,
        metadata
      };
      return acc;
    } else if (installSource === 'custom') {
      if (!parameter) {
        throw new Error(`Missing file path for integration: ${pkgName}`);
      }
      // Append file path if integration is already in the list
      const existing = acc[key];
      if (existing && existing.installSource === 'custom') {
        existing.logFilePaths.push(parameter);
        return acc;
      }
      acc[key] = {
        pkgName,
        installSource,
        logFilePaths: [parameter]
      };
      return acc;
    }
    throw new Error(`Invalid install source: ${installSource}`);
  }, {}));
}
function parseRegistryIntegrationMetadata(pkgName, parameter) {
  switch (pkgName) {
    case 'system':
      if (!parameter) {
        throw new Error('Missing hostname for System integration');
      }
      return {
        hostname: parameter
      };
    default:
      return undefined;
  }
}
function generateAgentConfigTar(output, installedIntegrations) {
  const now = new Date();
  return (0, _make_tar.makeTar)([{
    type: 'File',
    path: 'elastic-agent.yml',
    mode: 0o644,
    mtime: now,
    data: (0, _jsYaml.dump)({
      outputs: {
        default: (0, _output_client.transformOutputToFullPolicyOutput)(output, undefined, true)
      }
    })
  }, {
    type: 'Directory',
    path: 'inputs.d/',
    mode: 0o755,
    mtime: now
  }, ...installedIntegrations.map(integration => ({
    type: 'File',
    path: `inputs.d/${integration.pkgName}.yml`,
    mode: 0o644,
    mtime: now,
    data: integration.config
  }))]);
}
const flowRouteRepository = exports.flowRouteRepository = {
  ...createFlowRoute,
  ...stepProgressUpdateRoute,
  ...getProgressRoute,
  ...integrationsInstallRoute
};