"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggest = suggest;
var _helpers = require("../../../shared/helpers");
var _types = require("../../../definitions/types");
var _util = require("./util");
var _complete_items = require("../../complete_items");
var _factories = require("../../factories");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function suggest({
  innerText,
  command,
  getPolicies,
  getPolicyMetadata,
  getAllColumnNames,
  getSuggestedUserDefinedColumnName
}) {
  var _command$args$find;
  const pos = (0, _util.getPosition)(innerText, command);
  const policyName = (_command$args$find = command.args.find(arg => (0, _helpers.isSingleItem)(arg) && arg.type === 'source')) === null || _command$args$find === void 0 ? void 0 : _command$args$find.name;
  const getFieldSuggestionsForWithClause = async () => {
    if (!policyName) {
      return [];
    }
    const policyMetadata = await getPolicyMetadata(policyName);
    if (!policyMetadata) {
      return [];
    }
    const fieldSuggestions = (0, _factories.buildFieldsDefinitions)(policyMetadata.enrichFields, false);
    const lastWord = (0, _helpers.findFinalWord)(innerText);
    if (lastWord) {
      // ENRICH ... WITH a <suggest>
      const rangeToReplace = {
        start: innerText.length - lastWord.length + 1,
        end: innerText.length + 1
      };
      fieldSuggestions.forEach(s => {
        s.rangeToReplace = rangeToReplace;
      });
    }
    return fieldSuggestions;
  };
  switch (pos) {
    case _util.Position.MODE:
      return _util.modeSuggestions;
    case _util.Position.POLICY:
      {
        const policies = await getPolicies();
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (lastWord !== '') {
          policies.forEach(policySuggestion => {
            policySuggestion.rangeToReplace = {
              start: innerText.length - lastWord.length + 1,
              end: innerText.length + 1
            };
          });
        }
        return policies.length ? policies : [_util.noPoliciesAvailableSuggestion];
      }
    case _util.Position.AFTER_POLICY:
      return [_util.onSuggestion, _util.withSuggestion, _complete_items.pipeCompleteItem];
    case _util.Position.MATCH_FIELD:
      {
        if (!policyName) {
          return [];
        }
        const policyMetadata = await getPolicyMetadata(policyName);
        if (!policyMetadata) {
          return [];
        }
        return (0, _util.buildMatchingFieldsDefinition)(policyMetadata.matchField, getAllColumnNames());
      }
    case _util.Position.AFTER_ON_CLAUSE:
      return [_util.withSuggestion, _complete_items.pipeCompleteItem];
    case _util.Position.WITH_NEW_CLAUSE:
      {
        if (!policyName) {
          return [];
        }
        const policyMetadata = await getPolicyMetadata(policyName);
        if (!policyMetadata) {
          return [];
        }
        const suggestions = [];
        suggestions.push((0, _factories.getNewUserDefinedColumnSuggestion)(getSuggestedUserDefinedColumnName(policyMetadata.enrichFields)));
        suggestions.push(...(await getFieldSuggestionsForWithClause()));
        return suggestions;
      }
    case _util.Position.WITH_AFTER_FIRST_WORD:
      {
        if (!policyName) {
          return [];
        }
        const policyMetadata = await getPolicyMetadata(policyName);
        if (!policyMetadata) {
          return [];
        }
        const word = (0, _helpers.findPreviousWord)(innerText);
        if (policyMetadata.enrichFields.includes((0, _helpers.unescapeColumnName)(word))) {
          // complete field name
          return [_complete_items.pipeCompleteItem, {
            ..._complete_items.commaCompleteItem,
            command: _factories.TRIGGER_SUGGESTION_COMMAND
          }];
        } else {
          // not recognized as a field name, assume new user-defined column name
          return (0, _factories.getOperatorSuggestions)({
            location: _types.Location.ENRICH
          });
        }
      }
    case _util.Position.WITH_AFTER_ASSIGNMENT:
      {
        const suggestions = [];
        suggestions.push(...(await getFieldSuggestionsForWithClause()));
        return suggestions;
      }
    case _util.Position.WITH_AFTER_COMPLETE_CLAUSE:
      {
        return [_complete_items.pipeCompleteItem, {
          ..._complete_items.commaCompleteItem,
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }];
      }
    default:
      return [];
  }
}