/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.datastreams.autosharding;

import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalDouble;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.LongSupplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.PriorityQueue;
import org.elasticsearch.action.admin.indices.stats.IndexStats;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingResult;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingType;
import org.elasticsearch.cluster.ProjectState;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexWriteLoad;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.IndexingStats;

public class DataStreamAutoShardingService {
    private static final Logger logger = LogManager.getLogger(DataStreamAutoShardingService.class);
    public static final String DATA_STREAMS_AUTO_SHARDING_ENABLED = "data_streams.auto_sharding.enabled";
    public static final Setting<List<String>> DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING = Setting.listSetting("data_streams.auto_sharding.excludes", List.of(), Function.identity(), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN = Setting.timeSetting("data_streams.auto_sharding.increase_shards.cooldown", TimeValue.timeValueSeconds(270L), TimeValue.timeValueSeconds(0L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN = Setting.timeSetting("data_streams.auto_sharding.decrease_shards.cooldown", TimeValue.timeValueDays(3L), TimeValue.timeValueSeconds(0L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS = Setting.intSetting("cluster.auto_sharding.min_write_threads", 2, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS = Setting.intSetting("cluster.auto_sharding.max_write_threads", 32, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<WriteLoadMetric> DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_LOAD_METRIC = Setting.enumSetting(WriteLoadMetric.class, "data_streams.auto_sharding.increase_shards.load_metric", WriteLoadMetric.PEAK, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<WriteLoadMetric> DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_LOAD_METRIC = Setting.enumSetting(WriteLoadMetric.class, "data_streams.auto_sharding.decrease_shards.load_metric", WriteLoadMetric.PEAK, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final ClusterService clusterService;
    private final boolean isAutoShardingEnabled;
    private final LongSupplier nowSupplier;
    private final Consumer<Decision> decisionLogger;
    private volatile TimeValue increaseShardsCooldown;
    private volatile TimeValue decreaseShardsCooldown;
    private volatile int minWriteThreads;
    private volatile int maxWriteThreads;
    private volatile List<String> dataStreamExcludePatterns;
    private volatile WriteLoadMetric increaseShardsMetric;
    private volatile WriteLoadMetric decreaseShardsMetric;

    public DataStreamAutoShardingService(Settings settings, ClusterService clusterService, LongSupplier nowSupplier) {
        this(settings, clusterService, nowSupplier, DataStreamAutoShardingService.createPeriodicLoggingDecisionConsumer(nowSupplier));
    }

    private static Consumer<Decision> createPeriodicLoggingDecisionConsumer(LongSupplier nowSupplier) {
        PeriodicDecisionLogger periodicDecisionLogger = new PeriodicDecisionLogger(nowSupplier);
        return periodicDecisionLogger::maybeLogDecision;
    }

    DataStreamAutoShardingService(Settings settings, ClusterService clusterService, LongSupplier nowSupplier, Consumer<Decision> decisionLogger) {
        this.clusterService = clusterService;
        this.isAutoShardingEnabled = settings.getAsBoolean(DATA_STREAMS_AUTO_SHARDING_ENABLED, false);
        this.increaseShardsCooldown = DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN.get(settings);
        this.decreaseShardsCooldown = DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN.get(settings);
        this.minWriteThreads = CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS.get(settings);
        this.maxWriteThreads = CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS.get(settings);
        this.dataStreamExcludePatterns = DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING.get(settings);
        this.increaseShardsMetric = DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_LOAD_METRIC.get(settings);
        this.decreaseShardsMetric = DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_LOAD_METRIC.get(settings);
        this.nowSupplier = nowSupplier;
        this.decisionLogger = decisionLogger;
    }

    public void init() {
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN, this::updateIncreaseShardsCooldown);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN, this::updateReduceShardsCooldown);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS, this::updateMinWriteThreads);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS, this::updateMaxWriteThreads);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING, this::updateDataStreamExcludePatterns);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_LOAD_METRIC, this::updateIncreaseShardsMetric);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_LOAD_METRIC, this::updateDecreaseShardsMetric);
    }

    public AutoShardingResult calculate(ProjectState state, DataStream dataStream, @Nullable IndexStats writeIndexStats) {
        if (!this.isAutoShardingEnabled) {
            logger.debug("Data stream auto-sharding service is not enabled.");
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        if (this.dataStreamExcludePatterns.stream().anyMatch(pattern -> Regex.simpleMatch(pattern, dataStream.getName()))) {
            logger.debug("Data stream [{}] is excluded from auto-sharding via the [{}] setting", (Object)dataStream.getName(), (Object)DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING.getKey());
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        if (writeIndexStats == null) {
            logger.debug("Data stream auto-sharding service cannot compute the optimal number of shards for data stream [{}] as the write index stats are not available", (Object)dataStream.getName());
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        double writeIndexAllTimeLoad = DataStreamAutoShardingService.sumLoadMetrics(writeIndexStats, IndexingStats.Stats::getWriteLoad);
        double writeIndexRecentLoad = DataStreamAutoShardingService.sumLoadMetrics(writeIndexStats, IndexingStats.Stats::getRecentWriteLoad);
        double writeIndexPeakLoad = DataStreamAutoShardingService.sumLoadMetrics(writeIndexStats, IndexingStats.Stats::getPeakWriteLoad);
        IndexMetadata writeIndex = state.metadata().index(dataStream.getWriteIndex());
        assert (writeIndex != null) : "the data stream write index must exist in the provided cluster metadata";
        Decision.Inputs inputs = new Decision.Inputs(this.increaseShardsCooldown, this.decreaseShardsCooldown, this.minWriteThreads, this.maxWriteThreads, this.increaseShardsMetric, this.decreaseShardsMetric, dataStream.getName(), writeIndex.getIndex().getName(), writeIndexAllTimeLoad, writeIndexRecentLoad, writeIndexPeakLoad, writeIndex.getNumberOfShards());
        Decision decision = this.innerCalculate(state.metadata(), dataStream, inputs);
        this.decisionLogger.accept(decision);
        return decision.result();
    }

    private static double sumLoadMetrics(IndexStats stats, Function<IndexingStats.Stats, Double> loadMetric) {
        return Arrays.stream(stats.getShards()).filter(shardStats -> shardStats.getStats().indexing != null).filter(shardStats -> shardStats.getShardRouting().primary()).map(shardStats -> shardStats.getStats().indexing.getTotal()).map(loadMetric).reduce(0.0, Double::sum);
    }

    private Decision innerCalculate(ProjectMetadata project, DataStream dataStream, Decision.Inputs inputs) {
        Decision.IncreaseCalculation increaseCalculation = this.calculateIncreaseShardsDecision(dataStream, inputs);
        if (increaseCalculation.increaseResult() != null) {
            return new Decision(inputs, increaseCalculation, null, increaseCalculation.increaseResult());
        }
        Decision.DecreaseCalculation decreaseCalculation = this.calculateDecreaseShardsDecision(project, dataStream, inputs);
        if (decreaseCalculation.decreaseResult() != null) {
            return new Decision(inputs, increaseCalculation, decreaseCalculation, decreaseCalculation.decreaseResult());
        }
        return new Decision(inputs, increaseCalculation, decreaseCalculation, new AutoShardingResult(AutoShardingType.NO_CHANGE_REQUIRED, inputs.currentNumberOfWriteIndexShards(), inputs.currentNumberOfWriteIndexShards(), TimeValue.ZERO));
    }

    private Decision.IncreaseCalculation calculateIncreaseShardsDecision(DataStream dataStream, Decision.Inputs inputs) {
        double writeIndexLoadForIncrease = DataStreamAutoShardingService.pickMetric(inputs.increaseShardsMetric(), inputs.writeIndexAllTimeLoad(), inputs.writeIndexRecentLoad(), inputs.writeIndexPeakLoad());
        int optimalShardCountForIncrease = DataStreamAutoShardingService.computeOptimalNumberOfShards(inputs.minWriteThreads(), inputs.maxWriteThreads(), writeIndexLoadForIncrease);
        if (optimalShardCountForIncrease > inputs.currentNumberOfWriteIndexShards()) {
            TimeValue timeSinceLastAutoShardingEvent = dataStream.getAutoShardingEvent() != null ? dataStream.getAutoShardingEvent().getTimeSinceLastAutoShardingEvent(this.nowSupplier) : TimeValue.MAX_VALUE;
            TimeValue coolDownRemaining = TimeValue.timeValueMillis(Math.max(0L, inputs.increaseShardsCooldown().millis() - timeSinceLastAutoShardingEvent.millis()));
            return new Decision.IncreaseCalculation(writeIndexLoadForIncrease, optimalShardCountForIncrease, new AutoShardingResult(coolDownRemaining.equals(TimeValue.ZERO) ? AutoShardingType.INCREASE_SHARDS : AutoShardingType.COOLDOWN_PREVENTED_INCREASE, inputs.currentNumberOfWriteIndexShards(), optimalShardCountForIncrease, coolDownRemaining));
        }
        return new Decision.IncreaseCalculation(writeIndexLoadForIncrease, optimalShardCountForIncrease, null);
    }

    private TimeValue getRemainingDecreaseShardsCooldown(ProjectMetadata project, DataStream dataStream, TimeValue decreaseShardsCooldown) {
        Index oldestBackingIndex = dataStream.getIndices().get(0);
        IndexMetadata oldestIndexMeta = project.getIndexSafe(oldestBackingIndex);
        return dataStream.getAutoShardingEvent() == null ? TimeValue.timeValueMillis(Math.max(0L, oldestIndexMeta.getCreationDate() + decreaseShardsCooldown.millis() - this.nowSupplier.getAsLong())) : TimeValue.timeValueMillis(Math.max(0L, decreaseShardsCooldown.millis() - dataStream.getAutoShardingEvent().getTimeSinceLastAutoShardingEvent(this.nowSupplier).millis()));
    }

    private Decision.DecreaseCalculation calculateDecreaseShardsDecision(ProjectMetadata project, DataStream dataStream, Decision.Inputs inputs) {
        TimeValue remainingCooldownForDecrease = this.getRemainingDecreaseShardsCooldown(project, dataStream, inputs.decreaseShardsCooldown());
        Decision.DecreaseCalculation.MaxLoadWithinCooldown maxLoadWithinCooldownForDecrease = DataStreamAutoShardingService.getMaxIndexLoadWithinCoolingPeriod(project, dataStream, inputs, this.nowSupplier);
        int optimalShardCountForDecrease = DataStreamAutoShardingService.computeOptimalNumberOfShards(inputs.minWriteThreads(), inputs.maxWriteThreads(), maxLoadWithinCooldownForDecrease.load());
        if (optimalShardCountForDecrease < inputs.currentNumberOfWriteIndexShards()) {
            return new Decision.DecreaseCalculation(maxLoadWithinCooldownForDecrease, optimalShardCountForDecrease, new AutoShardingResult(remainingCooldownForDecrease.equals(TimeValue.ZERO) ? AutoShardingType.DECREASE_SHARDS : AutoShardingType.COOLDOWN_PREVENTED_DECREASE, inputs.currentNumberOfWriteIndexShards, optimalShardCountForDecrease, remainingCooldownForDecrease));
        }
        return new Decision.DecreaseCalculation(maxLoadWithinCooldownForDecrease, optimalShardCountForDecrease, null);
    }

    static int computeOptimalNumberOfShards(int minNumberWriteThreads, int maxNumberWriteThreads, double indexingLoad) {
        return Math.toIntExact(Math.max(Math.max(Math.min(DataStreamAutoShardingService.roundUp(indexingLoad / ((double)minNumberWriteThreads / 2.0)), 3L), DataStreamAutoShardingService.roundUp(indexingLoad / ((double)maxNumberWriteThreads / 2.0))), 1L));
    }

    private static long roundUp(double value) {
        return (long)Math.ceil(value);
    }

    static Decision.DecreaseCalculation.MaxLoadWithinCooldown getMaxIndexLoadWithinCoolingPeriod(ProjectMetadata project, DataStream dataStream, Decision.Inputs inputs, LongSupplier nowSupplier) {
        Map writeLoadsWithinCoolingPeriod = DataStream.getIndicesWithinMaxAgeRange(dataStream, project::getIndexSafe, inputs.decreaseShardsCooldown(), nowSupplier).stream().filter(index -> !index.equals(dataStream.getWriteIndex())).map(project::index).filter(Objects::nonNull).filter(metadata -> metadata.getStats() != null).filter(metadata -> metadata.getStats().indexWriteLoad() != null).collect(Collectors.toMap(metadata -> metadata.getIndex().getName(), metadata -> metadata.getStats().indexWriteLoad(), (unused1, unused2) -> {
            throw new IllegalStateException("Multiple indices with same name");
        }, LinkedHashMap::new));
        double maxLoadWithinCooldown = DataStreamAutoShardingService.pickMetric(inputs.decreaseShardsMetric(), inputs.writeIndexAllTimeLoad(), inputs.writeIndexRecentLoad(), inputs.writeIndexPeakLoad());
        String previousIndexInCooldownWithMaxLoad = null;
        for (Map.Entry entry : writeLoadsWithinCoolingPeriod.entrySet()) {
            String indexName = (String)entry.getKey();
            IndexWriteLoad writeLoad = (IndexWriteLoad)entry.getValue();
            double totalIndexLoad = 0.0;
            for (int shardId = 0; shardId < writeLoad.numberOfShards(); ++shardId) {
                Double writeLoadForShard = DataStreamAutoShardingService.pickMetric(inputs.decreaseShardsMetric(), DataStreamAutoShardingService.optionalDoubleToNullable(writeLoad.getWriteLoadForShard(shardId)), DataStreamAutoShardingService.optionalDoubleToNullable(writeLoad.getRecentWriteLoadForShard(shardId)), DataStreamAutoShardingService.optionalDoubleToNullable(writeLoad.getPeakWriteLoadForShard(shardId)));
                if (writeLoadForShard == null) continue;
                totalIndexLoad += writeLoadForShard.doubleValue();
            }
            if (!(totalIndexLoad > maxLoadWithinCooldown)) continue;
            maxLoadWithinCooldown = totalIndexLoad;
            previousIndexInCooldownWithMaxLoad = indexName;
        }
        return new Decision.DecreaseCalculation.MaxLoadWithinCooldown(maxLoadWithinCooldown, previousIndexInCooldownWithMaxLoad);
    }

    void updateIncreaseShardsCooldown(TimeValue scaleUpCooldown) {
        this.increaseShardsCooldown = scaleUpCooldown;
    }

    void updateReduceShardsCooldown(TimeValue scaleDownCooldown) {
        this.decreaseShardsCooldown = scaleDownCooldown;
    }

    void updateMinWriteThreads(int minNumberWriteThreads) {
        this.minWriteThreads = minNumberWriteThreads;
    }

    void updateMaxWriteThreads(int maxNumberWriteThreads) {
        this.maxWriteThreads = maxNumberWriteThreads;
    }

    private void updateDataStreamExcludePatterns(List<String> newExcludePatterns) {
        this.dataStreamExcludePatterns = newExcludePatterns;
    }

    private void updateIncreaseShardsMetric(WriteLoadMetric newMetric) {
        this.increaseShardsMetric = newMetric;
    }

    private void updateDecreaseShardsMetric(WriteLoadMetric newMetric) {
        this.decreaseShardsMetric = newMetric;
    }

    private static Double pickMetric(WriteLoadMetric metric, Double writeIndexLoad, Double writeIndexRecentLoad, Double writeIndexPeakLoad) {
        return switch (metric.ordinal()) {
            default -> throw new MatchException(null, null);
            case 0 -> writeIndexLoad;
            case 1 -> {
                if (writeIndexRecentLoad != null) {
                    yield writeIndexRecentLoad;
                }
                yield writeIndexLoad;
            }
            case 2 -> writeIndexPeakLoad != null ? writeIndexPeakLoad : writeIndexLoad;
        };
    }

    private static Double optionalDoubleToNullable(OptionalDouble optional) {
        return optional.isPresent() ? Double.valueOf(optional.getAsDouble()) : null;
    }

    static class PeriodicDecisionLogger {
        static final int BUFFER_SIZE = 10;
        static final long FLUSH_INTERVAL_MILLIS = TimeValue.timeValueMinutes(5L).millis();
        private static final Comparator<Decision> HIGHEST_LOAD_COMPARATOR = Comparator.comparing(d -> d.increaseCalculation().writeIndexLoadForIncrease());
        private final LongSupplier nowSupplier;
        private final Consumer<FlushedDecisions> logConsumer;
        private final AtomicLong lastFlushMillis;
        private final DecisionBuffer highestLoadIncreaseDecisions;
        private final DecisionBuffer highestLoadNonIncreaseDecisions;

        PeriodicDecisionLogger(LongSupplier nowSupplier) {
            this(nowSupplier, PeriodicDecisionLogger::logFlushedDecision);
        }

        PeriodicDecisionLogger(LongSupplier nowSupplier, Consumer<FlushedDecisions> logConsumer) {
            this.nowSupplier = nowSupplier;
            this.highestLoadIncreaseDecisions = new DecisionBuffer(10, HIGHEST_LOAD_COMPARATOR);
            this.highestLoadNonIncreaseDecisions = new DecisionBuffer(10, HIGHEST_LOAD_COMPARATOR);
            this.lastFlushMillis = new AtomicLong(nowSupplier.getAsLong());
            this.logConsumer = logConsumer;
        }

        void maybeLogDecision(Decision Decision2) {
            assert (Decision2.result != null) : "Attempting to log a decision with no result";
            logger.debug("Data stream auto-sharding result: {}", (Object)Decision2);
            if (Decision2.result.type() == AutoShardingType.INCREASE_SHARDS) {
                this.highestLoadIncreaseDecisions.insert(Decision2);
            } else {
                this.highestLoadNonIncreaseDecisions.insert(Decision2);
            }
            if (this.shouldFlush()) {
                FlushedDecisions flushedDecisions = new FlushedDecisions(this.highestLoadIncreaseDecisions.flush(), this.highestLoadNonIncreaseDecisions.flush());
                this.logConsumer.accept(flushedDecisions);
            }
        }

        private boolean shouldFlush() {
            long previous;
            long now = this.nowSupplier.getAsLong();
            return now - (previous = this.lastFlushMillis.getAndUpdate(last -> now - last >= FLUSH_INTERVAL_MILLIS ? now : last)) >= FLUSH_INTERVAL_MILLIS;
        }

        private static void logFlushedDecision(FlushedDecisions decisions) {
            if (!decisions.highestLoadIncreaseDecisions.isEmpty()) {
                logger.info("Data stream auto-sharding decisions in the last {} with highest load with an increase shards recommendation: \n{}", (Object)TimeValue.timeValueMillis(FLUSH_INTERVAL_MILLIS), (Object)decisions.highestLoadIncreaseDecisions.stream().map(d -> " - " + String.valueOf(d)).collect(Collectors.joining("\n")));
            }
            if (!decisions.highestLoadNonIncreaseDecisions.isEmpty()) {
                logger.info("Data stream auto-sharding decisions in the last {} with highest load without an increase shards recommendation: \n{}", (Object)TimeValue.timeValueMillis(FLUSH_INTERVAL_MILLIS), (Object)decisions.highestLoadNonIncreaseDecisions.stream().map(d -> " - " + String.valueOf(d)).collect(Collectors.joining("\n")));
            }
        }

        record FlushedDecisions(List<Decision> highestLoadIncreaseDecisions, List<Decision> highestLoadNonIncreaseDecisions) {
        }
    }

    public static enum WriteLoadMetric {
        ALL_TIME,
        RECENT,
        PEAK;

    }

    record Decision(Inputs inputs, IncreaseCalculation increaseCalculation, @Nullable DecreaseCalculation decreaseCalculation, AutoShardingResult result) {
        @Override
        public String toString() {
            return Strings.format("For data stream %s: %s based on [inc/dec cooldowns %s/%s, %d-%d threads, write index %s has all-time/recent/peak loads %g/%g/%g, current shards %d, increase calculation gives %d shards using %s load %g for write index%s]", new Object[]{this.inputs.dataStream, this.result, this.inputs.increaseShardsCooldown, this.inputs.decreaseShardsCooldown, this.inputs.minWriteThreads, this.inputs.maxWriteThreads, this.inputs.writeIndex, this.inputs.writeIndexAllTimeLoad, this.inputs.writeIndexRecentLoad, this.inputs.writeIndexPeakLoad, this.inputs.currentNumberOfWriteIndexShards, this.increaseCalculation.optimalShardCountForIncrease, this.inputs.increaseShardsMetric, this.increaseCalculation.writeIndexLoadForIncrease, this.decreaseCalculation == null ? "" : Strings.format(", decrease calculation gives %d shards using %s load %g for %s", new Object[]{this.decreaseCalculation.optimalShardCountForDecrease, this.inputs.decreaseShardsMetric, this.decreaseCalculation.maxLoadWithinCooldown.load, this.decreaseCalculation.maxLoadWithinCooldown.previousIndexWithMaxLoad != null ? this.decreaseCalculation.maxLoadWithinCooldown.previousIndexWithMaxLoad : "write index"})});
        }

        record Inputs(TimeValue increaseShardsCooldown, TimeValue decreaseShardsCooldown, int minWriteThreads, int maxWriteThreads, WriteLoadMetric increaseShardsMetric, WriteLoadMetric decreaseShardsMetric, String dataStream, String writeIndex, double writeIndexAllTimeLoad, double writeIndexRecentLoad, double writeIndexPeakLoad, int currentNumberOfWriteIndexShards) {
        }

        record IncreaseCalculation(double writeIndexLoadForIncrease, int optimalShardCountForIncrease, @Nullable AutoShardingResult increaseResult) {
        }

        record DecreaseCalculation(MaxLoadWithinCooldown maxLoadWithinCooldown, int optimalShardCountForDecrease, @Nullable AutoShardingResult decreaseResult) {

            record MaxLoadWithinCooldown(double load, @Nullable String previousIndexWithMaxLoad) {
            }
        }
    }

    private static class DecisionBuffer {
        private final Comparator<Decision> comparator;
        private final PriorityQueue<Decision> queue;

        DecisionBuffer(int maxSize, final Comparator<Decision> comparator) {
            this.comparator = comparator;
            this.queue = new PriorityQueue<Decision>(this, maxSize){

                @Override
                protected boolean lessThan(Decision decision1, Decision decision2) {
                    return comparator.compare(decision1, decision2) < 0;
                }
            };
        }

        synchronized void insert(Decision Decision2) {
            this.queue.insertWithOverflow(Decision2);
        }

        synchronized List<Decision> flush() {
            List<Decision> previousDecisions = StreamSupport.stream(this.queue.spliterator(), false).sorted(this.comparator.reversed()).toList();
            this.queue.clear();
            return previousDecisions;
        }
    }
}

