"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asErr = asErr;
exports.asOk = asOk;
exports.either = either;
exports.eitherAsync = eitherAsync;
exports.isErr = isErr;
exports.isOk = isOk;
exports.map = map;
exports.mapR = exports.mapOk = exports.mapErr = void 0;
exports.promiseResult = promiseResult;
exports.tryAsResult = tryAsResult;
exports.unwrap = unwrap;
exports.unwrapPromise = unwrapPromise;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function asOk(value) {
  return {
    tag: 'ok',
    value
  };
}
function asErr(error) {
  return {
    tag: 'err',
    error
  };
}
function isOk(result) {
  return result.tag === 'ok';
}
function isErr(result) {
  return !isOk(result);
}
function tryAsResult(fn) {
  try {
    return asOk(fn());
  } catch (e) {
    return asErr(e);
  }
}
async function promiseResult(future) {
  try {
    return asOk(await future);
  } catch (e) {
    const error = e.error ? e.error : e;
    return asErr(error);
  }
}
async function unwrapPromise(future) {
  return future.catch(
  // catch rejection as we expect the result of the rejected promise
  // to be wrapped in a Result - sadly there's no way to "Type" this
  // requirment in Typescript as Promises do not enfore a type on their
  // rejection
  // The `then` will then unwrap the Result from around `ex` for us
  ex => ex).then(result => map(result, value => Promise.resolve(value), err => Promise.reject(err)));
}
function unwrap(result) {
  return isOk(result) ? result.value : result.error;
}
function either(result, onOk, onErr) {
  map(result, onOk, onErr);
  return result;
}
async function eitherAsync(result, onOk, onErr) {
  return await map(result, onOk, onErr);
}
function map(result, onOk, onErr) {
  return isOk(result) ? onOk(result.value) : onErr(result.error);
}
const mapR = exports.mapR = (0, _lodash.curry)(function (onOk, onErr, result) {
  return map(result, onOk, onErr);
});
const mapOk = exports.mapOk = (0, _lodash.curry)(function (onOk, result) {
  return isOk(result) ? onOk(result.value) : result;
});
const mapErr = exports.mapErr = (0, _lodash.curry)(function (onErr, result) {
  return isOk(result) ? result : onErr(result.error);
});