"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asTrees = asTrees;
exports.buildStreamRows = buildStreamRows;
exports.isParentName = isParentName;
var _streamsSchema = require("@kbn/streams-schema");
var _helpers = require("../data_management/stream_detail_lifecycle/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isParentName(parent, descendant) {
  return parent !== descendant && descendant.startsWith(parent + '.');
}
function buildStreamRows(streams, sortField, sortDirection) {
  const enrich = node => {
    let retentionMs = 0;
    const lc = node.effective_lifecycle;
    if ((0, _streamsSchema.isDslLifecycle)(lc)) {
      var _lc$dsl$data_retentio;
      retentionMs = (0, _helpers.parseDurationInSeconds)((_lc$dsl$data_retentio = lc.dsl.data_retention) !== null && _lc$dsl$data_retentio !== void 0 ? _lc$dsl$data_retentio : '') * 1000;
    } else if ((0, _streamsSchema.isIlmLifecycle)(lc)) {
      retentionMs = Number.POSITIVE_INFINITY;
    }
    const nameSortKey = `${(0, _streamsSchema.getSegments)(node.name).length}_${node.name.toLowerCase()}`;
    return {
      ...node,
      nameSortKey,
      documentsCount: 0,
      retentionMs,
      children: node.children.map(enrich)
    };
  };
  const enrichedTrees = asTrees(streams).map(enrich);
  const compare = (a, b) => {
    const av = a[sortField];
    const bv = b[sortField];
    if (typeof av === 'string' && typeof bv === 'string') {
      return sortDirection === 'asc' ? av.localeCompare(bv) : bv.localeCompare(av);
    }
    if (typeof av === 'number' && typeof bv === 'number') {
      return sortDirection === 'asc' ? av - bv : bv - av;
    }
    return 0;
  };
  const shouldSortChildren = sortField === 'nameSortKey' || sortField === 'retentionMs';
  const result = [];
  const pushNode = (node, level, rootMeta) => {
    result.push({
      ...node,
      level,
      ...rootMeta
    });
    const children = shouldSortChildren ? [...node.children].sort(compare) : node.children;
    children.forEach(child => pushNode(child, level + 1, rootMeta));
  };
  [...enrichedTrees].sort(compare).forEach(root => {
    const rootMeta = {
      rootNameSortKey: root.nameSortKey,
      rootDocumentsCount: root.documentsCount,
      rootRetentionMs: root.retentionMs
    };
    pushNode(root, 0, rootMeta);
  });
  return result;
}
function asTrees(streams) {
  const trees = [];
  const sortedStreams = streams.slice().sort((a, b) => (0, _streamsSchema.getSegments)(a.stream.name).length - (0, _streamsSchema.getSegments)(b.stream.name).length);
  sortedStreams.forEach(streamDetail => {
    let currentTree = trees;
    let existingNode;
    while (existingNode = currentTree.find(node => isParentName(node.name, streamDetail.stream.name))) {
      currentTree = existingNode.children;
    }
    if (!existingNode) {
      const newNode = {
        ...streamDetail,
        name: streamDetail.stream.name,
        children: [],
        type: _streamsSchema.Streams.UnwiredStream.Definition.is(streamDetail.stream) ? 'classic' : (0, _streamsSchema.isRootStreamDefinition)(streamDetail.stream) ? 'root' : 'wired'
      };
      currentTree.push(newNode);
    }
  });
  return trees;
}