"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDocumentsCollectorActor = createDocumentsCollectorActor;
exports.createDocumentsCountCollectorActor = createDocumentsCountCollectorActor;
exports.routingSamplesMachine = exports.createRoutingSamplesMachineImplementations = void 0;
var _xstate = require("xstate5");
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _xstateUtils = require("@kbn/xstate-utils");
var _i18n = require("@kbn/i18n");
var _formatters = require("../../../../../util/formatters");
var _condition = require("../../../../../util/condition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SAMPLES_SIZE = 100;
const PROBABILITY_THRESHOLD = 100000;
const SEARCH_TIMEOUT_MS = 10000; // 10 seconds

const routingSamplesMachine = exports.routingSamplesMachine = (0, _xstate.setup)({
  types: {
    input: {},
    context: {},
    events: {}
  },
  actors: {
    collectDocuments: (0, _xstateUtils.getPlaceholderFor)(createDocumentsCollectorActor),
    collectDocumentsCount: (0, _xstateUtils.getPlaceholderFor)(createDocumentsCountCollectorActor),
    subscribeTimeUpdates: (0, _xstateUtils.getPlaceholderFor)(createTimeUpdatesActor)
  },
  actions: {
    updateCondition: (0, _xstate.assign)((_, params) => ({
      condition: params.condition
    })),
    storeDocuments: (0, _xstate.assign)((_, params) => ({
      documents: params.documents
    })),
    storeDocumentsError: (0, _xstate.assign)((_, params) => ({
      documentsError: params.error
    })),
    storeDocumentCounts: (0, _xstate.assign)((_, params) => ({
      approximateMatchingPercentage: params.count,
      approximateMatchingPercentageError: undefined
    })),
    storeDocumentCountsError: (0, _xstate.assign)((_, params) => ({
      approximateMatchingPercentageError: params.error
    }))
  },
  delays: {
    conditionUpdateDebounceTime: 500
  },
  guards: {
    isValidSnapshot: (_, params) => params.context !== undefined
  }
}).createMachine({
  /** @xstate-layout N4IgpgJg5mDOIC5QCcD2BXALgSwHZQGUBDAWwAcAbOAYjSz0NMrgDpkwAzd2ACwG0ADAF1EoMqljYcqXKJAAPRADYAjCwDsAJiUCBSgJwqAHAGYVAViXqlAGhABPRABZLLc9aUmr6rSpMBffzs6HHxicipYWgxQxgjWdDIIIkwwAGEZCClsGUERJBBxSWlZAsUEL3MWIxUVJ2N9ARV9cxVbB2cTExYdXU16vyNzIydA4JiGcOZYFggwACMMXABjBgzcLJLqeVhMFLAWIg5U5AAKZczsmQBVJP2AEQWl5bAAFWwSMABKaPowpkisye6BWa0uJTyciKV1KoHKJnUVTq6hMTicAjM6IEljsjgQnm6mi0NRM+nq5j0+jGIBCkwBrA4YEwyx4DFmqGW6E+uEwMwoqCIWXw1AgMgOeAAbqgANYHC4UKjLTD3DlcsA82CQgrQkpycrmExGFhOKxGAyIymafS4xD6O0sO0Y7HqCwYlTU2n-eIzRnM1n4dmc7m8lj8wUMahgZBoZAsSgpDioZAkFjyxXK1XBzXCKESGF6xAG-QsPzuQxE-RKE02hBGdRG8zmJx2pxGTReTzqD0TL3TFi+lls0VB9WYDIgkNhoVQEViliSmVy1AKsBKlUjjXjnlasR53VlQtoh3qASNJ31EbWjoIPwqdRuUwmPpafp+bt-OJ9gf+qCBtU8rdJwFadI2jJM4woBMkxTNNVwzDdeUAndCj3HJYQUQ8qk8Iw22MLo-HUK88RUTQBE0Y1zErAR1CcDxdHMQIghAXBUDmeACk9T9IlzYo0ILBAsMNUxhhwusTRUGtNCMAQS10AR0TvTRtBcd9YimQE5kWEFVnwdZNj47VUJkfj0WLZsVGomjaKaPQa0sGS2jLEZsUbRtVLpb1+yZQd8B4-MDwE7psOE0SxNUGsT26RtzOaC9K00dze0Bb8h0zUd2N3XjjICizyJRFp9FMQihn0JSIsMFhsWbA1DSJQiqSYzj1IZbyfz-LNQ2Ahg-P3OFEFdDRSUoorCsosrryMfR70aVtRpafCGMansuJav1UoQmZRVwMAeoMvrayaB1SMRA1NAsk0TBrSajXrCxrFPWp9BMBKlo-ZqfVa9b-zHJZeV27L9ostRgpMEScJo8Lr1qDFjS8JSDWk9xnsSlaPrWgNh2+wC+S63zDKy9DyjqYsQbBsKa0emSntUIlaZcFoUferz0d-THg2x9ltv+wnEGk7oyIsO8alIswJKhs7yOpto2wNJGXsCIA */
  id: 'routingSamples',
  context: ({
    input
  }) => ({
    condition: input.condition,
    definition: input.definition,
    approximateMatchingPercentage: undefined,
    documents: [],
    documentsError: undefined,
    approximateMatchingPercentageError: undefined
  }),
  initial: 'fetching',
  invoke: {
    id: 'subscribeTimeUpdatesActor',
    src: 'subscribeTimeUpdates'
  },
  on: {
    'routingSamples.refresh': {
      target: '.fetching',
      reenter: true
    },
    'routingSamples.updateCondition': {
      target: '.debouncingCondition',
      reenter: true,
      actions: [{
        type: 'updateCondition',
        params: ({
          event
        }) => event
      }]
    }
  },
  states: {
    debouncingCondition: {
      after: {
        conditionUpdateDebounceTime: 'fetching'
      }
    },
    fetching: {
      type: 'parallel',
      states: {
        documents: {
          initial: 'loading',
          states: {
            loading: {
              entry: [{
                type: 'storeDocumentsError',
                params: {
                  error: undefined
                }
              }],
              invoke: {
                id: 'collectDocuments',
                src: 'collectDocuments',
                input: ({
                  context
                }) => ({
                  condition: context.condition,
                  definition: context.definition
                }),
                onSnapshot: {
                  guard: {
                    type: 'isValidSnapshot',
                    params: ({
                      event
                    }) => ({
                      context: event.snapshot.context
                    })
                  },
                  actions: [{
                    type: 'storeDocuments',
                    params: ({
                      event
                    }) => {
                      var _event$snapshot$conte;
                      return {
                        documents: (_event$snapshot$conte = event.snapshot.context) !== null && _event$snapshot$conte !== void 0 ? _event$snapshot$conte : []
                      };
                    }
                  }]
                },
                onDone: {
                  target: 'done'
                },
                onError: {
                  target: 'done',
                  actions: [{
                    type: 'storeDocuments',
                    params: {
                      documents: []
                    }
                  }, {
                    type: 'storeDocumentsError',
                    params: ({
                      event
                    }) => ({
                      error: event.error
                    })
                  }]
                }
              }
            },
            done: {
              type: 'final'
            }
          }
        },
        documentCounts: {
          initial: 'loading',
          states: {
            loading: {
              invoke: {
                id: 'collectDocumentsCount',
                src: 'collectDocumentsCount',
                input: ({
                  context
                }) => ({
                  condition: context.condition,
                  definition: context.definition
                }),
                onSnapshot: {
                  guard: {
                    type: 'isValidSnapshot',
                    params: ({
                      event
                    }) => ({
                      context: event.snapshot.context
                    })
                  },
                  actions: [{
                    type: 'storeDocumentCounts',
                    params: ({
                      event
                    }) => {
                      var _event$snapshot$conte2;
                      return {
                        count: (_event$snapshot$conte2 = event.snapshot.context) !== null && _event$snapshot$conte2 !== void 0 ? _event$snapshot$conte2 : ''
                      };
                    }
                  }]
                },
                onDone: {
                  target: 'done'
                },
                onError: {
                  target: 'done',
                  actions: [{
                    type: 'storeDocumentCountsError',
                    params: ({
                      event
                    }) => ({
                      error: event.error
                    })
                  }]
                }
              }
            },
            done: {
              type: 'final'
            }
          }
        }
      }
    }
  }
});
const createRoutingSamplesMachineImplementations = ({
  data,
  timeState$
}) => ({
  actors: {
    collectDocuments: createDocumentsCollectorActor({
      data
    }),
    collectDocumentsCount: createDocumentsCountCollectorActor({
      data
    }),
    subscribeTimeUpdates: createTimeUpdatesActor({
      timeState$
    })
  }
});
exports.createRoutingSamplesMachineImplementations = createRoutingSamplesMachineImplementations;
function createDocumentsCollectorActor({
  data
}) {
  return (0, _xstate.fromObservable)(({
    input
  }) => {
    return collectDocuments({
      data,
      input
    });
  });
}
function createDocumentsCountCollectorActor({
  data
}) {
  return (0, _xstate.fromObservable)(({
    input
  }) => {
    return collectDocumentCounts({
      data,
      input
    });
  });
}
function createTimeUpdatesActor({
  timeState$
}) {
  return (0, _xstate.fromEventObservable)(() => timeState$.pipe((0, _rxjs.map)(() => ({
    type: 'routingSamples.refresh'
  }))));
}
function collectDocuments({
  data,
  input
}) {
  const abortController = new AbortController();
  const {
    start,
    end
  } = getAbsoluteTimestamps(data);
  const params = buildDocumentsSearchParams({
    ...input,
    start,
    end
  });
  return new _rxjs.Observable(observer => {
    const subscription = data.search.search({
      params
    }, {
      abortSignal: abortController.signal,
      retrieveResults: true
    }).pipe((0, _rxjs.filter)(result => !(0, _common.isRunningResponse)(result) || !(0, _lodash.isEmpty)(result.rawResponse.hits.hits)), (0, _rxjs.timeout)(SEARCH_TIMEOUT_MS), (0, _rxjs.map)(result => result.rawResponse.hits.hits.map(hit => hit._source)), (0, _rxjs.catchError)(handleTimeoutError)).subscribe(observer);
    return () => {
      abortController.abort();
      subscription.unsubscribe();
    };
  });
}
const percentageFormatter = (0, _formatters.getPercentageFormatter)({
  precision: 2
});
function collectDocumentCounts({
  data,
  input
}) {
  const abortController = new AbortController();
  const {
    start,
    end
  } = getAbsoluteTimestamps(data);
  const searchParams = {
    ...input,
    start,
    end
  };
  const params = buildDocumentCountSearchParams(searchParams);
  return new _rxjs.Observable(observer => {
    const subscription = data.search.search({
      params
    }, {
      abortSignal: abortController.signal
    }).pipe((0, _rxjs.filter)(result => !(0, _common.isRunningResponse)(result)), (0, _rxjs.timeout)(SEARCH_TIMEOUT_MS), (0, _rxjs.switchMap)(countResult => {
      const docCount = !countResult.rawResponse.hits.total || (0, _lodash.isNumber)(countResult.rawResponse.hits.total) ? countResult.rawResponse.hits.total : countResult.rawResponse.hits.total.value;
      return data.search.search({
        params: buildDocumentCountProbabilitySearchParams({
          ...searchParams,
          docCount
        })
      }, {
        abortSignal: abortController.signal
      }).pipe((0, _rxjs.filter)(result => !(0, _common.isRunningResponse)(result)), (0, _rxjs.timeout)(SEARCH_TIMEOUT_MS), (0, _rxjs.map)(result => {
        // Aggregations don't return partial results so we just wait until the end
        if (result.rawResponse.aggregations) {
          // We need to divide this by the sampling / probability factor:
          // https://www.elastic.co/guide/en/elasticsearch/reference/current/search-aggregations-random-sampler-aggregation.html#random-sampler-special-cases
          const sampleAgg = result.rawResponse.aggregations.sample;
          const randomSampleDocCount = sampleAgg.doc_count / sampleAgg.probability;
          const matchingDocCount = sampleAgg.matching_docs.doc_count;
          return percentageFormatter.format(matchingDocCount / randomSampleDocCount);
        }
        return undefined;
      }), (0, _rxjs.catchError)(handleTimeoutError));
    }), (0, _rxjs.catchError)(handleTimeoutError)).subscribe(observer);
    return () => {
      abortController.abort();
      subscription.unsubscribe();
    };
  });
}
const createTimestampRangeQuery = (start, end) => ({
  range: {
    '@timestamp': {
      gte: start,
      lte: end,
      format: 'epoch_millis'
    }
  }
});
const getAbsoluteTimestamps = data => {
  const time = data.query.timefilter.timefilter.getAbsoluteTime();
  return {
    start: new Date(time.from).getTime(),
    end: new Date(time.to).getTime()
  };
};

/**
 * Create runtime mappings for fields that aren't mapped.
 * Conditions could be using fields which are not indexed or they could use it with other types than they are eventually mapped as.
 * Because of this we can't rely on mapped fields to draw a sample, instead we need to use runtime fields to simulate what happens during
 * ingest in the painless condition checks.
 */
function getRuntimeMappings(definition, condition) {
  if (!condition) return {};
  const mappedFields = Object.keys({
    ...definition.inherited_fields,
    ...definition.stream.ingest.wired.fields
  });
  return Object.fromEntries((0, _streamsSchema.getConditionFields)(condition).filter(field => !mappedFields.includes(field.name)).map(field => [field.name, {
    type: field.type === 'string' ? 'keyword' : 'double'
  }]));
}
function processCondition(condition) {
  if (!condition) return undefined;
  const convertedCondition = (0, _condition.emptyEqualsToAlways)(condition);
  return convertedCondition && (0, _streamsSchema.isAlwaysCondition)(convertedCondition) ? undefined : convertedCondition;
}
function handleTimeoutError(error) {
  if (error.name === 'TimeoutError') {
    return (0, _rxjs.throwError)(() => new Error(_i18n.i18n.translate('xpack.streams.routingSamples.documentsSearchTimeoutErrorMessage', {
      defaultMessage: 'Documents search timed out after 10 seconds. Refresh the preview or try simplifying the routing condition.'
    })));
  }
  return (0, _rxjs.throwError)(() => error);
}
function buildDocumentsSearchParams({
  condition,
  start,
  end,
  definition
}) {
  const finalCondition = processCondition(condition);
  const runtimeMappings = getRuntimeMappings(definition, finalCondition);
  return {
    index: definition.stream.name,
    query: {
      bool: {
        must: [finalCondition ? (0, _streamsSchema.conditionToQueryDsl)(finalCondition) : {
          match_all: {}
        }, createTimestampRangeQuery(start, end)]
      }
    },
    runtime_mappings: runtimeMappings,
    size: SAMPLES_SIZE,
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }],
    terminate_after: SAMPLES_SIZE,
    track_total_hits: false,
    allow_partial_search_results: true
  };
}
function buildDocumentCountSearchParams({
  start,
  end,
  definition
}) {
  return {
    index: definition.stream.name,
    query: createTimestampRangeQuery(start, end),
    size: 0,
    track_total_hits: true
  };
}
function buildDocumentCountProbabilitySearchParams({
  condition,
  definition,
  docCount,
  end,
  start
}) {
  const finalCondition = processCondition(condition);
  const runtimeMappings = getRuntimeMappings(definition, finalCondition);
  const query = finalCondition ? (0, _streamsSchema.conditionToQueryDsl)(finalCondition) : {
    match_all: {}
  };
  const probability = calculateProbability(docCount);
  return {
    index: definition.stream.name,
    query: createTimestampRangeQuery(start, end),
    aggs: {
      sample: {
        random_sampler: {
          probability
        },
        aggs: {
          matching_docs: {
            filter: query
          }
        }
      }
    },
    runtime_mappings: runtimeMappings,
    size: 0,
    _source: false,
    track_total_hits: false
  };
}

/**
 * Calculates sampling probability based on document count
 */
function calculateProbability(docCount) {
  if (!docCount || docCount <= PROBABILITY_THRESHOLD) {
    return 1;
  }
  const probability = PROBABILITY_THRESHOLD / docCount;
  // Values between 0.5 and 1 are not supported by the random sampler
  return probability <= 0.5 ? probability : 1;
}