"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HistoricalSummaryClient = void 0;
exports.getFixedIntervalAndBucketsPerDay = getFixedIntervalAndBucketsPerDay;
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
var _models = require("../domain/models");
var _services = require("../domain/services");
var _get_slices_from_date_range = require("./utils/get_slices_from_date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class HistoricalSummaryClient {
  constructor(esClient) {
    this.esClient = esClient;
  }
  async fetch(params) {
    const dateRangeBySlo = params.list.reduce((acc, {
      sloId,
      timeWindow,
      range
    }) => {
      acc[sloId] = getDateRange(timeWindow, range);
      return acc;
    }, {});
    const searches = params.list.flatMap(({
      sloId,
      revision,
      budgetingMethod,
      instanceId,
      groupBy,
      timeWindow,
      remoteName
    }) => [{
      index: remoteName ? `${remoteName}:${_constants.SLI_DESTINATION_INDEX_PATTERN}` : _constants.SLI_DESTINATION_INDEX_PATTERN
    }, generateSearchQuery({
      groupBy,
      sloId,
      revision,
      instanceId,
      timeWindow,
      budgetingMethod,
      dateRange: dateRangeBySlo[sloId]
    })]);
    const historicalSummary = [];
    if (searches.length === 0) {
      return historicalSummary;
    }
    const result = await this.esClient.msearch({
      searches
    });
    for (let i = 0; i < result.responses.length; i++) {
      var _result$responses$i$a, _result$responses$i$a2;
      const {
        sloId,
        instanceId,
        timeWindow,
        budgetingMethod,
        objective
      } = params.list[i];
      if ('error' in result.responses[i]) {
        // handle erroneous responses with an empty historical summary data
        historicalSummary.push({
          sloId,
          instanceId,
          data: []
        });
        continue;
      }

      // @ts-ignore typing msearch is hard, we cast the response to what it is supposed to be.
      const buckets = ((_result$responses$i$a = result.responses[i].aggregations) === null || _result$responses$i$a === void 0 ? void 0 : (_result$responses$i$a2 = _result$responses$i$a.daily) === null || _result$responses$i$a2 === void 0 ? void 0 : _result$responses$i$a2.buckets) || [];
      if (_sloSchema.rollingTimeWindowSchema.is(timeWindow)) {
        if (_sloSchema.timeslicesBudgetingMethodSchema.is(budgetingMethod)) {
          historicalSummary.push({
            sloId,
            instanceId,
            data: handleResultForRollingAndTimeslices(objective, timeWindow, buckets, dateRangeBySlo[sloId])
          });
          continue;
        }
        if (_sloSchema.occurrencesBudgetingMethodSchema.is(budgetingMethod)) {
          historicalSummary.push({
            sloId,
            instanceId,
            data: handleResultForRollingAndOccurrences(objective, buckets, dateRangeBySlo[sloId])
          });
          continue;
        }
        (0, _std.assertNever)(budgetingMethod);
      }
      if (_sloSchema.calendarAlignedTimeWindowSchema.is(timeWindow)) {
        if (_sloSchema.timeslicesBudgetingMethodSchema.is(budgetingMethod)) {
          const dateRange = dateRangeBySlo[sloId];
          historicalSummary.push({
            sloId,
            instanceId,
            data: handleResultForCalendarAlignedAndTimeslices(objective, buckets, dateRange)
          });
          continue;
        }
        if (_sloSchema.occurrencesBudgetingMethodSchema.is(budgetingMethod)) {
          historicalSummary.push({
            sloId,
            instanceId,
            data: handleResultForCalendarAlignedAndOccurrences(objective, buckets)
          });
          continue;
        }
        (0, _std.assertNever)(budgetingMethod);
      }
      (0, _std.assertNever)(timeWindow);
    }
    return historicalSummary;
  }
}
exports.HistoricalSummaryClient = HistoricalSummaryClient;
function handleResultForCalendarAlignedAndOccurrences(objective, buckets) {
  const initialErrorBudget = 1 - objective.target;
  return buckets.map(bucket => {
    var _bucket$cumulative_go, _bucket$cumulative_go2, _bucket$cumulative_to, _bucket$cumulative_to2;
    const good = (_bucket$cumulative_go = (_bucket$cumulative_go2 = bucket.cumulative_good) === null || _bucket$cumulative_go2 === void 0 ? void 0 : _bucket$cumulative_go2.value) !== null && _bucket$cumulative_go !== void 0 ? _bucket$cumulative_go : 0;
    const total = (_bucket$cumulative_to = (_bucket$cumulative_to2 = bucket.cumulative_total) === null || _bucket$cumulative_to2 === void 0 ? void 0 : _bucket$cumulative_to2.value) !== null && _bucket$cumulative_to !== void 0 ? _bucket$cumulative_to : 0;
    const sliValue = (0, _services.computeSLI)(good, total);
    const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
    const errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget, true);
    return {
      date: bucket.key_as_string,
      errorBudget,
      sliValue,
      status: (0, _services.computeSummaryStatus)(objective, sliValue, errorBudget)
    };
  });
}
function handleResultForCalendarAlignedAndTimeslices(objective, buckets, dateRange) {
  const initialErrorBudget = 1 - objective.target;
  const totalSlices = (0, _get_slices_from_date_range.getSlicesFromDateRange)(dateRange.range, objective.timesliceWindow);
  return buckets.map(bucket => {
    var _bucket$cumulative_go3, _bucket$cumulative_go4, _bucket$cumulative_to3, _bucket$cumulative_to4;
    const good = (_bucket$cumulative_go3 = (_bucket$cumulative_go4 = bucket.cumulative_good) === null || _bucket$cumulative_go4 === void 0 ? void 0 : _bucket$cumulative_go4.value) !== null && _bucket$cumulative_go3 !== void 0 ? _bucket$cumulative_go3 : 0;
    const total = (_bucket$cumulative_to3 = (_bucket$cumulative_to4 = bucket.cumulative_total) === null || _bucket$cumulative_to4 === void 0 ? void 0 : _bucket$cumulative_to4.value) !== null && _bucket$cumulative_to3 !== void 0 ? _bucket$cumulative_to3 : 0;
    const sliValue = (0, _services.computeSLI)(good, total, totalSlices);
    const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
    const errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget);
    return {
      date: bucket.key_as_string,
      errorBudget,
      sliValue,
      status: (0, _services.computeSummaryStatus)(objective, sliValue, errorBudget)
    };
  });
}
function handleResultForRollingAndOccurrences(objective, buckets, dateRange) {
  const initialErrorBudget = 1 - objective.target;
  return buckets.filter(bucket => (0, _moment.default)(bucket.key_as_string).isSameOrAfter(dateRange.range.from) && (0, _moment.default)(bucket.key_as_string).isSameOrBefore(dateRange.range.to)).map(bucket => {
    var _bucket$cumulative_go5, _bucket$cumulative_go6, _bucket$cumulative_to5, _bucket$cumulative_to6;
    const good = (_bucket$cumulative_go5 = (_bucket$cumulative_go6 = bucket.cumulative_good) === null || _bucket$cumulative_go6 === void 0 ? void 0 : _bucket$cumulative_go6.value) !== null && _bucket$cumulative_go5 !== void 0 ? _bucket$cumulative_go5 : 0;
    const total = (_bucket$cumulative_to5 = (_bucket$cumulative_to6 = bucket.cumulative_total) === null || _bucket$cumulative_to6 === void 0 ? void 0 : _bucket$cumulative_to6.value) !== null && _bucket$cumulative_to5 !== void 0 ? _bucket$cumulative_to5 : 0;
    const sliValue = (0, _services.computeSLI)(good, total);
    const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
    const errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget);
    return {
      date: bucket.key_as_string,
      errorBudget,
      sliValue,
      status: (0, _services.computeSummaryStatus)(objective, sliValue, errorBudget)
    };
  });
}
function handleResultForRollingAndTimeslices(objective, timeWindow, buckets, dateRange) {
  const initialErrorBudget = 1 - objective.target;
  const totalSlices = Math.ceil(timeWindow.duration.asSeconds() / objective.timesliceWindow.asSeconds());
  return buckets.filter(bucket => (0, _moment.default)(bucket.key_as_string).isSameOrAfter(dateRange.range.from) && (0, _moment.default)(bucket.key_as_string).isSameOrBefore(dateRange.range.to)).map(bucket => {
    var _bucket$cumulative_go7, _bucket$cumulative_go8, _bucket$cumulative_to7, _bucket$cumulative_to8;
    const good = (_bucket$cumulative_go7 = (_bucket$cumulative_go8 = bucket.cumulative_good) === null || _bucket$cumulative_go8 === void 0 ? void 0 : _bucket$cumulative_go8.value) !== null && _bucket$cumulative_go7 !== void 0 ? _bucket$cumulative_go7 : 0;
    const total = (_bucket$cumulative_to7 = (_bucket$cumulative_to8 = bucket.cumulative_total) === null || _bucket$cumulative_to8 === void 0 ? void 0 : _bucket$cumulative_to8.value) !== null && _bucket$cumulative_to7 !== void 0 ? _bucket$cumulative_to7 : 0;
    const sliValue = (0, _services.computeSLI)(good, total, totalSlices);
    const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
    const errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget);
    return {
      date: bucket.key_as_string,
      errorBudget,
      sliValue,
      status: (0, _services.computeSummaryStatus)(objective, sliValue, errorBudget)
    };
  });
}
function generateSearchQuery({
  sloId,
  groupBy,
  revision,
  instanceId,
  dateRange,
  timeWindow,
  budgetingMethod
}) {
  const unit = (0, _sloSchema.toMomentUnitOfTime)(timeWindow.duration.unit);
  const timeWindowDurationInDays = _moment.default.duration(timeWindow.duration.value, unit).asDays();
  const queryRangeDurationInDays = Math.ceil((0, _moment.default)(dateRange.range.to).diff(dateRange.range.from, 'days'));
  const {
    fixedInterval,
    bucketsPerDay
  } = getFixedIntervalAndBucketsPerDay(queryRangeDurationInDays);
  const extraFilterByInstanceId = !!groupBy && ![groupBy].flat().includes(_sloSchema.ALL_VALUE) && instanceId !== _sloSchema.ALL_VALUE ? [{
    term: {
      'slo.instanceId': instanceId
    }
  }] : [];
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            'slo.id': sloId
          }
        }, {
          term: {
            'slo.revision': revision
          }
        }, {
          range: {
            '@timestamp': {
              gte: dateRange.queryRange.from.toISOString(),
              lte: dateRange.queryRange.to.toISOString()
            }
          }
        }, ...extraFilterByInstanceId]
      }
    },
    aggs: {
      daily: {
        date_histogram: {
          field: '@timestamp',
          fixed_interval: fixedInterval,
          extended_bounds: {
            min: dateRange.queryRange.from.toISOString(),
            max: dateRange.queryRange.to.toISOString()
          }
        },
        aggs: {
          ...(_sloSchema.occurrencesBudgetingMethodSchema.is(budgetingMethod) && {
            good: {
              sum: {
                field: 'slo.numerator'
              }
            },
            total: {
              sum: {
                field: 'slo.denominator'
              }
            }
          }),
          ...(_sloSchema.timeslicesBudgetingMethodSchema.is(budgetingMethod) && {
            good: {
              sum: {
                field: 'slo.isGoodSlice'
              }
            },
            total: {
              value_count: {
                field: 'slo.isGoodSlice'
              }
            }
          }),
          cumulative_good: {
            moving_fn: {
              buckets_path: 'good',
              window: timeWindowDurationInDays * bucketsPerDay,
              shift: 1,
              script: 'MovingFunctions.sum(values)',
              gap_policy: 'insert_zeros'
            }
          },
          cumulative_total: {
            moving_fn: {
              buckets_path: 'total',
              window: timeWindowDurationInDays * bucketsPerDay,
              shift: 1,
              script: 'MovingFunctions.sum(values)',
              gap_policy: 'insert_zeros'
            }
          }
        }
      }
    }
  };
}

/**
 * queryRange is used for the filter range on the query,
 * while range is used for storing the actual range requested
 * For rolling SLO, the query range starts 1 timeWindow duration before the actual range from.
 * For calendar aligned SLO, the query range is the same as the range.
 *
 * @param timeWindow
 * @param range
 * @returns the request {range} and the query range {queryRange}
 *
 */
function getDateRange(timeWindow, range) {
  if (_sloSchema.rollingTimeWindowSchema.is(timeWindow)) {
    const unit = (0, _sloSchema.toMomentUnitOfTime)(timeWindow.duration.unit);
    if (range) {
      return {
        range,
        queryRange: {
          from: (0, _moment.default)(range.from).subtract(timeWindow.duration.value, unit).startOf('minute').toDate(),
          to: (0, _moment.default)(range.to).startOf('minute').toDate()
        }
      };
    }
    const now = (0, _moment.default)();
    return {
      range: {
        from: now.clone().subtract(timeWindow.duration.value, unit).startOf('minute').toDate(),
        to: now.clone().startOf('minute').toDate()
      },
      queryRange: {
        from: now.clone().subtract(timeWindow.duration.value * 2, unit).startOf('minute').toDate(),
        to: now.clone().startOf('minute').toDate()
      }
    };
  }
  if (_sloSchema.calendarAlignedTimeWindowSchema.is(timeWindow)) {
    var _range$from;
    // If range is provided, we use the lower bound to calculate the calendar aligned range.
    const now = (0, _moment.default)((_range$from = range === null || range === void 0 ? void 0 : range.from) !== null && _range$from !== void 0 ? _range$from : new Date());
    const unit = (0, _models.toCalendarAlignedTimeWindowMomentUnit)(timeWindow);
    const from = _moment.default.utc(now).startOf(unit);
    const to = _moment.default.utc(now).endOf(unit);
    const calendarRange = {
      from: from.toDate(),
      to: to.toDate()
    };
    return {
      range: calendarRange,
      queryRange: calendarRange
    };
  }
  (0, _std.assertNever)(timeWindow);
}
function getFixedIntervalAndBucketsPerDay(durationInDays) {
  if (durationInDays <= 3) {
    return {
      fixedInterval: '30m',
      bucketsPerDay: 48
    };
  }
  if (durationInDays <= 7) {
    return {
      fixedInterval: '1h',
      bucketsPerDay: 24
    };
  }
  if (durationInDays <= 30) {
    return {
      fixedInterval: '4h',
      bucketsPerDay: 6
    };
  }
  if (durationInDays <= 90) {
    return {
      fixedInterval: '12h',
      bucketsPerDay: 2
    };
  }
  return {
    fixedInterval: '1d',
    bucketsPerDay: 1
  };
}