"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createIngestStatsTaskConfig = createIngestStatsTaskConfig;
var _types = require("../types");
var _helpers = require("../helpers");
var _events = require("../event_based/events");
var _configuration = require("../configuration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const COUNTER_LABELS = ['security_solution', 'pipelines-stats'];
function createIngestStatsTaskConfig() {
  const taskType = 'security:ingest-pipelines-stats-telemetry';
  return {
    type: taskType,
    title: 'Security Solution Telemetry Ingest Pipelines Stats task',
    interval: '24h',
    timeout: '5m',
    version: '1.0.0',
    getLastExecutionTime: _helpers.getPreviousDailyTaskTimestamp,
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('indices-metadata'), mdc);
      const trace = taskMetricsService.start(taskType);
      const taskConfig = _configuration.telemetryConfiguration.ingest_pipelines_stats_config;
      const start = performance.now();
      try {
        logger.info('Running ingest stats task');
        if (!taskConfig.enabled) {
          logger.info('Ingest stats task is disabled, skipping');
          await taskMetricsService.end(trace);
          return 0;
        }
        const ingestStats = await receiver.getIngestPipelinesStats('3m');
        logger.info('Got ingest stats, about to publish EBT events', {
          count: ingestStats.length
        });
        ingestStats.forEach(stats => {
          sender.reportEBT(_events.TELEMETRY_NODE_INGEST_PIPELINES_STATS_EVENT, stats);
        });
        const telemetryUsageCounter = sender.getTelemetryUsageCluster();
        telemetryUsageCounter === null || telemetryUsageCounter === void 0 ? void 0 : telemetryUsageCounter.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(COUNTER_LABELS.concat('events')),
          counterType: _types.TelemetryCounter.DOCS_SENT,
          incrementBy: ingestStats.length
        });
        await taskMetricsService.end(trace);
        log.info('Ingest stats task completed', {
          count: ingestStats.length,
          elapsed: performance.now() - start
        });
        return ingestStats.length;
      } catch (error) {
        log.warn(`Error running ingest stats task`, {
          error,
          elapsed: performance.now() - start
        });
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}