"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsStartRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _audit = require("./util/audit");
var _authz = require("./util/authz");
var _retry = require("./util/retry");
var _with_license = require("./util/with_license");
var _tracing = require("./util/tracing");
var _with_existing_migration_id = require("./util/with_existing_migration_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsStartRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_START_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.StartRuleMigrationRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.StartRuleMigrationRequestBody)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const migrationId = req.params.migration_id;
    const {
      langsmith_options: langsmithOptions,
      settings: {
        connector_id: connectorId,
        skip_prebuilt_rules_matching: skipPrebuiltRulesMatching = false
      },
      retry
    } = req.body;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution);
    try {
      const ctx = await context.resolve(['core', 'actions', 'alerting', 'securitySolution']);

      // Check if the connector exists and user has permissions to read it
      const connector = await ctx.actions.getActionsClient().get({
        id: connectorId
      });
      if (!connector) {
        return res.badRequest({
          body: `Connector with id ${connectorId} not found`
        });
      }
      const ruleMigrationsClient = ctx.securitySolution.getSiemRuleMigrationsClient();
      if (retry) {
        const {
          updated
        } = await ruleMigrationsClient.task.updateToRetry(migrationId, (0, _retry.getRetryFilter)(retry));
        if (!updated) {
          return res.ok({
            body: {
              started: false
            }
          });
        }
      }
      const callbacks = (0, _tracing.createTracersCallbacks)(langsmithOptions, logger);
      const {
        exists,
        started
      } = await ruleMigrationsClient.task.start({
        migrationId,
        connectorId,
        invocationConfig: {
          callbacks,
          configurable: {
            skipPrebuiltRulesMatching
          }
        }
      });
      if (!exists) {
        return res.notFound();
      }
      await siemMigrationAuditLogger.logStart({
        migrationId
      });
      return res.ok({
        body: {
          started
        }
      });
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logStart({
        migrationId,
        error
      });
      return res.customError({
        statusCode: 500,
        body: error.message
      });
    }
  })));
};
exports.registerSiemRuleMigrationsStartRoute = registerSiemRuleMigrationsStartRoute;