"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseAndTransformRuleActions = exports.isIndexPatternsBulkEditAction = exports.hasAlertSuppressionBulkEditAction = void 0;
var _lodash = require("lodash");
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _rule_actions = require("../../normalization/rule_actions");
var _transform_actions = require("../../../../../../common/detection_engine/transform_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * helper utility that defines whether bulk edit action is related to index patterns, i.e. one of:
 * 'add_index_patterns', 'delete_index_patterns', 'set_index_patterns'
 * @param editAction {@link BulkActionEditType}
 * @returns {boolean}
 */
const isIndexPatternsBulkEditAction = editAction => {
  const indexPatternsActions = [_rule_management.BulkActionEditTypeEnum.add_index_patterns, _rule_management.BulkActionEditTypeEnum.delete_index_patterns, _rule_management.BulkActionEditTypeEnum.set_index_patterns];
  return indexPatternsActions.includes(editAction);
};

/**
 * helper utility that defines whether bulk edit action is related to alert suppression, i.e. one of:
 * 'set_alert_suppression_for_threshold', 'delete_alert_suppression', 'set_alert_suppression'
 * @param editAction {@link BulkActionEditType}
 * @returns {boolean}
 */
exports.isIndexPatternsBulkEditAction = isIndexPatternsBulkEditAction;
const isAlertSuppressionBulkEditAction = editAction => {
  const bulkActions = [_rule_management.BulkActionEditTypeEnum.set_alert_suppression_for_threshold, _rule_management.BulkActionEditTypeEnum.delete_alert_suppression, _rule_management.BulkActionEditTypeEnum.set_alert_suppression];
  return bulkActions.includes(editAction);
};

/**
 * Checks if any of the actions is related to alert suppression, i.e. one of:
 * 'set_alert_suppression_for_threshold', 'delete_alert_suppression', 'set_alert_suppression'
 * @param actions {@link BulkActionEditPayload[][]}
 * @returns {boolean}
 */
const hasAlertSuppressionBulkEditAction = actions => {
  return actions.some(action => isAlertSuppressionBulkEditAction(action.type));
};

/**
 * Separates system actions from actions and performs necessary transformations for
 * alerting rules client bulk edit operations.
 * @param actionsClient
 * @param actions
 * @param throttle
 * @returns
 */
exports.hasAlertSuppressionBulkEditAction = hasAlertSuppressionBulkEditAction;
const parseAndTransformRuleActions = (actionsClient, actions, throttle) => {
  const [systemActions, extActions] = !(0, _lodash.isEmpty)(actions) ? (0, _lodash.partition)(actions, action => actionsClient.isSystemAction(action.id)) : [[], actions];
  return [...(systemActions !== null && systemActions !== void 0 ? systemActions : []), ...(0, _rule_actions.transformToActionFrequency)(extActions !== null && extActions !== void 0 ? extActions : [], throttle).map(_transform_actions.transformNormalizedRuleToAlertAction)];
};
exports.parseAndTransformRuleActions = parseAndTransformRuleActions;